/* WCEMidiOutputPort.java -- MIDI Output port for Windows Mobile.
   Copyright (C) 2009 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

package gnu.javax.sound.midi.wce;

import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MetaEventListener;
import javax.sound.midi.ControllerEventListener;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiDevice.Info;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.Sequence;
import javax.sound.midi.Sequencer;
import javax.sound.midi.Track;
import javax.sound.midi.Transmitter;

/**
 * MIDI output port for Windows Mobile device.
 */
public class WCEMidiOutputPort implements MidiDevice
{
  /**
   * Singleton.
   */
  private static final WCEMidiOutputPort INSTANCE = new WCEMidiOutputPort();

  private final Receiver RECEIVER = new PortReceiver();

  /**
   * flags for isOpen() and close()
   */
  private boolean explicitOpen;
  private boolean opened;

  /**
   * target Sequence object.
   */
  private Sequence sequence;

  /**
   * Receiver class.
   */
  private class PortReceiver implements Receiver
  {
    public void send(MidiMessage message, long timeStamp)
    {
      throw new UnsupportedOperationException("Not implemented");
    }

    public void close()
    {
      if (! explicitOpen)
      {
        WCEMidiOutputPort.this.close();
      }
    }
  }

  /**
   * returns an instance.
   */
  public static WCEMidiOutputPort getInstance()
  {
    return INSTANCE;
  }

  public MidiDevice.Info getDeviceInfo()
  {
    return WCEMidiDeviceProvider.OUTPUT_PORT_INFO;
  }

  /**
   * open this output port device.
   */
  public synchronized void open()
  {
    if (!isOpen())
    {
      this.explicitOpen = true;
      this.opened = true;

      openNative();
    }
  }

  private native void openNative();
  
  /**
   * close this output port device.
   */
  public synchronized void close()
  {
    if (isOpen())
    {
      this.opened = false;
      this.explicitOpen = false;
      closeNative();
    }
  }

  private native void closeNative();

  public boolean isOpen()
  {
    return this.opened;
  }

  public long getMicrosecondPosition()
  {
    return -1L;
  }

  public int getMaxReceivers()
  {
    return 1;
  }

  public int getMaxTransmitters()
  {
    return 0;
  }

  public Receiver getReceiver()
                     throws MidiUnavailableException
  {
    return RECEIVER;
  }

  public Transmitter getTransmitter()
                           throws MidiUnavailableException
  {
    throw new MidiUnavailableException();
  }

  // Methods for a Sequencer
  
  void setSequence(Sequence sequence)
  {
    // stop current MIDI file
    stopPlaySequence();
    closeNativeMidiFile();

    this.sequence = sequence;
    if (isPlaying())
    {
      startPlaySequence();
    }
  }

  /**
   * returns playing status.
   */
  boolean isPlaying()
  {
    return isNativePlaying();
  }

  private native boolean isNativePlaying();

  /**
   * start playing.
   */
  void start()
  {
    startPlaySequence();
  }

  /**
   * play a Sequence.
   */
  private void startPlaySequence()
  {
    if (this.sequence == null)
    {
      return;
    }
    // get MIDI file from specified Sequence
    File file = WCEMidiFileReader.getFileFor(this.sequence);
    if (file == null)
    {
      throw new IllegalStateException("this.sequence not loaded by WCEMidiFileReader");
    }
    startNativeMidiFile(file.getAbsolutePath());
  }

  private native void startNativeMidiFile(String fileName);

  /**
   * stop playing Sequence.
   */
  void stop()
  {
    stopPlaySequence();
  }

  private void stopPlaySequence()
  {
    if (this.sequence == null)
    {
      return;
    }
    stopNativeMidiFile();
  }

  /**
   * stop playing MIDI file, but not close it.
   */
  private native void stopNativeMidiFile();

  /**
   * close current MIDI file.
   */
  private native void closeNativeMidiFile();

}
