/* gnu_java_awt_peer_wce_WCEGraphics2D.cpp
   Copyright (C) 2006 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

#include <windows.h>
#include <jni.h>
#include <assert.h>
#include "wce-peer.h"
#include <gnu_java_awt_peer_wce_WCEGraphics2D.h>
#include "region.h"
#include "FreeTypeGlyphVector.h"

#define BufferedImage_TYPE_CUSTOM		0
#define BufferedImage_TYPE_INT_RGB		1
#define BufferedImage_TYPE_INT_ARGB		2
#define BufferedImage_TYPE_INT_ARGB_PRE		3
#define BufferedImage_TYPE_INT_BGR		4
#define BufferedImage_TYPE_3BYTE_BGR		5
#define BufferedImage_TYPE_4BYTE_ABGR		6
#define BufferedImage_TYPE_4BYTE_ABGR_PRE	7
#define BufferedImage_TYPE_USHORT_565_RGB	8
#define BufferedImage_TYPE_USHORT_555_RGB	9
#define BufferedImage_TYPE_BYTE_GRAY		10
#define BufferedImage_TYPE_USHORT_GRAY		11
#define BufferedImage_TYPE_BYTE_BINARY		12
#define BufferedImage_TYPE_BYTE_INDEXED		13

#define PathIterator_WIND_EVEN_ODD 	0
#define PathIterator_WIND_NON_ZERO 	1

/**
 * AffineTransform
 */
class AffineTransform
{
private:
  float m00;
  float m01;
  float m02;
  float m10;
  float m11;
  float m12;

public:
  inline AffineTransform ():m00 (1), m01 (0), m02 (0), m10 (0), m11 (1),
    m12 (0)
  {
  }
  AffineTransform (float m00, float m01, float m02, float m10, float m11,
		   float m12);

  inline void setToTranslation (float tx, float ty)
  {
    m00 = m11 = 1.0f;
    m01 = m10 = 0.0f;
    m02 = tx;
    m12 = ty;
  }

  inline void translate (float tx, float ty)
  {
    m02 += tx * m00 + ty * m01;
    m12 += tx * m10 + ty * m11;
  }

  inline void transform (int *px, int *py)
  {
    int x = *px;
    int y = *py;
    *px = (int) (m00 * x + m01 * y + m02);
    *py = (int) (m10 * x + m11 * y + m12);
  }
};


/**
 * GradientPaint struct.
 */
typedef struct gradient_paint_t
{
  int x1;
  int y1;
  COLORREF c1;
  int x2;
  int y2;
  COLORREF c2;
  BOOL cyclic;
} gradient_paint;

/**
 * TexturePaint struct
 */
typedef struct texture_paint_t
{
  /**
   * pattern bitmap.
   */
  HBITMAP hBitmap;
  int bitmapWidth;
  int bitmapHeight;

  /**
   * anchor
   */
  int anchorX;
  int anchorY;
  int anchorWidth;
  int anchorHeight;
} texture_paint;


typedef enum paint_info_type_t
{
  PAINT_TYPE_UNDEFINED = 0,	// UNDEFINED
  PAINT_TYPE_COLOR,		// Color
  PAINT_TYPE_GRADIENT_PAINT,	// GradientPaint
  PAINT_TYPE_TEXTURE_PAINT,	// TexturePaint

  PAINT_TYPE_PAINT,		// other Paint subclass.
} paint_info_type;

/**
 * Paint struct.
 */
typedef struct paint_info_t
{
  /**
   * type of Paint
   */
  paint_info_type type;

  union
  {
    /**
     * COLORREF (Color class info)
     */
    COLORREF color;

    /**
     * GradientPaint
     */
    gradient_paint gradient;

    /**
     * TexturePaint
     */
    texture_paint texture;

    /**
     * other Paint subclass.
     */
    jobject paint_object;
  };
} paint_info;

/**
 * nativedata.
 * store information of Graphics2D object.
 */
typedef struct native_graphics_t
{
  /**
   * handle of the window.
   */
  HWND hWnd;

  /**
   * device context
   */
  HDC hDC;

  /**
   * target bitmap
   * NULL if this native_graphics has no target bitmap.
   */
  HBITMAP hBitmap;

  /**
   * BufferedImage flag.
   */
  BOOL bufferedImageGraphics;

  /**
   * CreateCompatibleDC()ĂяoWindows
   * rbg}bṽnh
   * DCGetDC()Ŏ擾ĂꍇNULL
   */
  HBITMAP hDefaultBitmap;

  /**
   * yCg[h xorModȅꍇ
   */
  BOOL xorMode;

  /**
   * XOR[hɎgpF
   */
  COLORREF xorColor;

  /**
   * y̕
   */
  int penWidth;

  /**
   * Paint
   */
  paint_info paint;

  /**
   * AffineTransform
   */
  AffineTransform transform;

  // ̑A܂܂ȏێ
} native_graphics;

/**
 * `ΏۂƂȂrbg}bvIԂɂ
 * `OɂȂ炸̃}NĂяoKv
 */
#define PREPARE_BITMAP(ngr) \
{ \
  if (ngr->hBitmap) \
  { \
    HGDIOBJ hDefaultBitmap = SelectObject(ngr->hDC, ngr->hBitmap); \
    assert(hDefaultBitmap == ngr->hDefaultBitmap); \
  } \
}

/**
 * `ΏۂƂȂrbg}bvIԂɂ
 * `ɂȂ炸̃}NĂяoKv
 */
#define UNPREPARE_BITMAP(ngr) \
{ \
    if (ngr->hBitmap) { \
	HGDIOBJ hBitmap = SelectObject(ngr->hDC, ngr->hDefaultBitmap); \
	assert(hBitmap == ngr->hBitmap); \
    } \
}

/**
 * uV쐬đIԂɂ
 */
#define SELECT_BRUSH_FOR_PAINT(env, ngr, x, y, width, height) \
{ \
    HBRUSH hBrush = create_brush(env, ngr, x, y, width, height); \
    DeleteObject(SelectObject(ngr->hDC, hBrush)); \
}

/**
 * uVIԂɂ폜
 */
#define UNSELECT_BRUSH_FOR_PAINT(env, ngr) \
{ \
  HBRUSH hBrush = (HBRUSH) SelectObject(ngr->hDC, GetStockObject(NULL_BRUSH)); \
  DeleteObject(hBrush); \
}

/**
 * WCEGraphics2D.updateBufferedImage()Ăяo
 */
#define UPDATE_BUFFERED_IMAGE(env, obj, ngr) \
{ \
    update_buffered_image(env, obj, ngr); \
}

/**
 * updateBufferedImage()̃\bhID
 */
static jmethodID g_updateBufferedImage_mid;

/**
 * method id of WCEGraphics2D.getPixelsOfBufferedImage()
 */
static jmethodID g_getPixelsOfBufferedImage_mid;

/**
 * update a content of BufferedImage
 */
static inline void
update_buffered_image (JNIEnv * env, jobject obj, native_graphics * ngr,
		       int x = INT_MIN, int y = INT_MIN, int width =
		       INT_MAX, int height = INT_MAX)
{
  if (!ngr->bufferedImageGraphics)
    {
      return;
    }

  const HBITMAP hBitmapHandle = ngr->hBitmap;
  jobject pixels =
    env->CallObjectMethod (obj, g_getPixelsOfBufferedImage_mid);
  if (pixels == NULL)
    {
      // cannot access pixel data directly, call java method instead.
      env->CallVoidMethod (obj, g_updateBufferedImage_mid);
      return;
    }

  BITMAP bitmap;
  if (!GetObject (hBitmapHandle, sizeof (BITMAP), &bitmap))
    {
      throw_AWTError_with_error_code (env, "GetObject() failed.",
				      GetLastError ());
      return;
    }

  // validate array size.
  jsize pixel_count = bitmap.bmWidth * bitmap.bmHeight;
  jsize array_length = env->GetArrayLength ((jarray) pixels);
  if (pixel_count != array_length)
    {
      throw_AWTError (env, "pixel_count!=array_length");
      return;
    }
  // calc. size of update area
  if (width > bitmap.bmWidth)
    {
      width = bitmap.bmWidth;
    }
  if (height > bitmap.bmHeight)
    {
      height = bitmap.bmHeight;
    }
  if (x < 0)
    x = 0;
  if (y < 0)
    y = 0;
  
  if (x + width > bitmap.bmWidth)
    {
      width = bitmap.bmWidth - x;
    }
  if (y + height > bitmap.bmHeight)
    {
      height = bitmap.bmHeight - y;
    }

  if (x > bitmap.bmWidth || y > bitmap.bmHeight || width < 0 || height < 0)
    {
      // No copy
      return;
    }


  // copy pixel data directly
  switch (bitmap.bmBitsPixel)
    {
    case 32:
      {
	int offset = y * bitmap.bmWidth + x;
	int dibOffset = y * bitmap.bmWidthBytes + x * sizeof (DWORD);
	const int size = width * sizeof (jint);
	jint *data = env->GetIntArrayElements ((jintArray) pixels, NULL);
	const char *src = (char *) bitmap.bmBits;
	if (data)
	  {
	    for (int i = 0; i <  height; ++i)
	      {
		memcpy (data + offset, src + dibOffset, size);
		offset += bitmap.bmWidth;
		dibOffset += bitmap.bmWidthBytes;
	      }
	    env->ReleaseIntArrayElements ((jintArray) pixels, data, 0);
	  }
      }
      break;

    case 16:
      {
	int destOffset = y * bitmap.bmWidth + x;
	int srcOffset = y * bitmap.bmWidthBytes + x * sizeof (WORD);
	const int size = width * sizeof (jshort);

	jshort *data =
	  env->GetShortArrayElements ((jshortArray) pixels, NULL);
	const char *src = (char *) bitmap.bmBits;
	if (data)
	  {
	    for (int i = 0; i < height; ++i)
	      {
		memcpy (data + destOffset, src + srcOffset, size);
		destOffset += bitmap.bmWidth;
		srcOffset += bitmap.bmWidthBytes;
	      }
	  }
	env->ReleaseShortArrayElements ((jshortArray) pixels, data, 0);
      }
      break;

    default:
      throw_AWTError (env, "Unexpeted bitmap.bmBitsPixel");
      return;
    }
}

/**
 * lCeBuyCg폜
 */
static void
delete_paint (native_graphics * ngr)
{
  if (ngr->paint.type == PAINT_TYPE_TEXTURE_PAINT)
    {
      // eNX`p̃rbg}bv폜
      DeleteObject (ngr->paint.texture.hBitmap);
      ngr->paint.type = PAINT_TYPE_UNDEFINED;
    }
}

/**
 * Of[Vp^[rbg}bv𐶐
 *
 * @param	x	`̈̍W(x)
 * @param	y	`̈̍W(y)
 * @param	w	`̈̕
 * @param	h	`̈̍
 * @param	g	Of[V
 */
static HBITMAP
create_gradient_pattern (int x, int y, int w, int h, native_graphics * ngd)
{
  gradient_paint *g = &ngd->paint.gradient;
  int x1 = g->x1;
  int y1 = g->y1;
  COLORREF c1 = g->c1;
  int x2 = g->x2;
  int y2 = g->y2;
  COLORREF c2 = g->c2;
  BOOL cyclic = g->cyclic;
  float distance;
  HBITMAP result;
  WORD *data;

  // XOR[h̏s
  if (ngd->xorMode)
    {
      c1 ^= ngd->xorColor;
      c2 ^= ngd->xorColor;
    }

  // Of[Vp^[𐶐
  if (x1 == x2)
    {
      // Jn^IxWĩOf[Vj
      distance = fabsf ((float) y2 - y1);
      // p^[̕2sNZŏ\
      w = 2;
    }
  else if (y1 == y2)
    {
      // Jn^IyWĩOf[Vj
      distance = fabsf ((float) x2 - x1);
      // p^[̍2sNZŏ\
      h = 2;
    }
  else
    {
      // Cӂ̃̕Of[V
      distance =
	sqrtf ((float) (x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
    }

  // ܂͍2ȏɂȂƂȂ
  // ̊֐ԂuVɑ΂āASetBrushOrgEx()Ō_ύXꍇɁA
  // uV_rbg}bv̕i܂͍j𒴂ꍇAi܂͍jŐV_̒l]肪ݒ肳v
  // ƂH
  // http://msdn.microsoft.com/library/default.asp?url=/library/en-us/wceui40/html/cerefsetbrushorgex.asp
  if (w < 2)
    {
      w = 2;
    }
  if (h < 2)
    {
      h = 2;
    }

  // DIBZNV쐬
  // tH[}bg RGB565 Ƃ
  BITMAPINFO_RGB565 biBMP = { 0 };
  biBMP.bmiHeader.biSize = sizeof (BITMAPINFOHEADER);
  biBMP.bmiHeader.biBitCount = 16;	// 
  biBMP.bmiHeader.biCompression = BI_BITFIELDS;	// biBitCount = 16, biCompression = BI_BITFIELDS̏ꍇRGB565Ӗ
  biBMP.bmiHeader.biPlanes = 1;
  biBMP.bmiHeader.biWidth = w;
  biBMP.bmiHeader.biHeight = -h;	// gbv_EDIB
  biBMP.dwRedMask = 0x1f << (5 + 6);
  biBMP.dwGreenMask = 0x3f << 5;
  biBMP.dwBlueMask = 0x1f;
  result = CreateDIBSection (NULL,
			     (BITMAPINFO *) & biBMP,
			     DIB_RGB_COLORS, (void **) (&data), NULL, 0);
  if (result)
    {
      BITMAP bitmap;
      if (!GetObject (result, sizeof (BITMAP), &bitmap))
	{
	  assert (false);
	  return result;
	}

      WORD *line = (WORD *) malloc (sizeof (WORD) * w);
      if (!line)
	{
	  assert (false);
	  return result;
	}
      WORD *dest = data;
      for (int r = 0; r < h; r++)
	{
	  for (int c = 0; c < w; c++)
	    {
	      float u;
	      if (distance != 0)
		{
		  u =
		    (((x + c) - x1) * (x2 - x1) +
		     ((y + r) - y1) * (y2 - y1)) / distance;
		}
	      else
		{
		  u = 0.0;
		}
	      float ratio = u / distance;
	      if (cyclic)
		{
		  ratio =
		    fabsf (ratio - floorf ((ratio + 1.0f) / 2.0f) * 2.0f);
		}
	      else
		{
		  ratio = max (0.0f, min (1.0f, ratio));
		}
	      // 8rbg̒lred=5bit, green=6bit, blue=5bit ɕϊ
	      BYTE red =
		(BYTE) (GetRValue (c1) +
			ratio * (GetRValue (c2) - GetRValue (c1))) >> 3;
	      BYTE green =
		(BYTE) (GetGValue (c1) +
			ratio * (GetGValue (c2) - GetGValue (c1))) >> 2;
	      BYTE blue =
		(BYTE) (GetBValue (c1) +
			ratio * (GetBValue (c2) - GetBValue (c1))) >> 3;

	      // RGB565`Œli[
	      line[c] =
		(WORD) (((BYTE) (blue) | (((WORD) green) << 5)) |
			(((WORD) red) << (5 + 6)));
	    }
	  dest = (WORD *) ((char *) data + r * bitmap.bmWidthBytes);
	  memcpy (dest, line, sizeof (WORD) * w);
	}
      free (line);
    }
  return result;
}

/**
 * y쐬
 */
static HPEN
create_pen (JNIEnv * env, native_graphics * ngd)
{
  HPEN hPen = CreatePen (PS_SOLID,
			 ngd->penWidth,
			 (ngd->xorMode ? ngd->paint.color ^ ngd->
			  xorColor : ngd->paint.color));
  if (!hPen)
    {
      throw_AWTError (env, "Failed to create a pen");
    }
  return hPen;
}

/**
 * uV쐬
 */
static HBRUSH
create_brush (JNIEnv * env, native_graphics * ngd, int x, int y, int width,
	      int height)
{
  char errmsg[256];
  HBRUSH result = NULL;

  if (!width || !height)
    {
      // 傫0̏ꍇɂ͍쐬łȂ
      return (HBRUSH) GetStockObject (NULL_BRUSH);
    }

  switch (ngd->paint.type)
    {
    case PAINT_TYPE_COLOR:
      // PFuV
      result =
	CreateSolidBrush (ngd->
			  xorMode ? (ngd->paint.color ^ ngd->xorColor) : ngd->
			  paint.color);
      break;

    case PAINT_TYPE_GRADIENT_PAINT:
      // GradientPaint
      {
	DWORD lasterror;
	HBITMAP hBitmap = create_gradient_pattern (x, y, width, height, ngd);
	if (!hBitmap)
	  {
	    lasterror = GetLastError ();
	    _snprintf (errmsg,
		       sizeof (errmsg) / sizeof (errmsg[0]),
		       "create_gradient_pattern(%d, %d, %d, %d) failed: GetLastError()=%u",
		       x, y, width, height, lasterror);
	    throw_AWTError (env, errmsg);
	    break;
	  }
	result = CreatePatternBrush (hBitmap);
	lasterror = GetLastError ();
	DeleteObject (hBitmap);	// rbg}bv͂ɍ폜Ă܂Ă悢
	if (!result)
	  {
	    _snprintf (errmsg,
		       sizeof (errmsg) / sizeof (errmsg[0]),
		       "CreatePatternBrush() failed: GetLastError()=%u",
		       lasterror);
	    throw_AWTError (env, "CreatePatternBrush() failed");
	    break;
	  }
	// uV̌_ړĂ
	SetBrushOrgEx (ngd->hDC, x, y, NULL);
      }
      break;

    case PAINT_TYPE_TEXTURE_PAINT:
      // TexturePaint
      {
	BITMAP bitmap = { 0 };
	HBITMAP hResized = NULL;
	HBITMAP hbmpPattern = NULL;
	char msg[128];

	// AJ[̈̃TCYƃC[W̃TCYƂr
	if (!GetObject (ngd->paint.texture.hBitmap, sizeof (BITMAP), &bitmap))
	  {
	    _snprintf (msg,
		       sizeof (msg) / sizeof (msg[0]),
		       "GetObject() failed: GetLastError()=%d",
		       GetLastError ());
	    throw_AWTError (env, msg);
	    break;
	  }
	if ((ngd->paint.texture.anchorWidth != bitmap.bmWidth)
	    || (ngd->paint.texture.anchorHeight != bitmap.bmHeight))
	  {
	    // TCY
	    HBITMAP hOldSrcBitmap = NULL;
	    HBITMAP hOldDestBitmap = NULL;
	    HDC hdcSrc = CreateCompatibleDC (ngd->hDC);
	    HDC hdcDest = CreateCompatibleDC (ngd->hDC);
	    if (!hdcSrc || !hdcDest)
	      {
		_snprintf (msg,
			   sizeof (msg) / sizeof (msg[0]),
			   "CreateCompatibleDC() failed: GetLastError()=%d",
			   GetLastError ());
		throw_AWTError (env, msg);
		break;
	      }
	    hResized =
	      CreateCompatibleBitmap (ngd->hDC,
				      ngd->paint.texture.anchorWidth,
				      ngd->paint.texture.anchorHeight);
	    if (!hResized)
	      {
		DeleteDC (hdcSrc);
		DeleteDC (hdcDest);
		_snprintf (msg,
			   sizeof (msg) / sizeof (msg[0]),
			   "CreateCompatibleBitmap() failed: GetLastError()=%d",
			   GetLastError ());
		throw_AWTError (env, msg);
		break;
	      }
	    hOldDestBitmap = (HBITMAP) SelectObject (hdcDest, hResized);
	    hOldSrcBitmap =
	      (HBITMAP) SelectObject (hdcSrc, ngd->paint.texture.hBitmap);
	    StretchBlt (hdcDest, 0, 0, ngd->paint.texture.anchorWidth,
			ngd->paint.texture.anchorHeight, hdcSrc, 0, 0,
			bitmap.bmWidth, bitmap.bmHeight, SRCCOPY);
	    SelectObject (hdcDest, hOldDestBitmap);
	    SelectObject (hdcSrc, hOldSrcBitmap);
	    DeleteDC (hdcSrc);
	    DeleteDC (hdcDest);
	    hbmpPattern = hResized;
	  }
	else
	  {
	    hbmpPattern = ngd->paint.texture.hBitmap;
	  }

	result = CreatePatternBrush (hbmpPattern);
	if (!result)
	  {
	    _snprintf (msg,
		       sizeof (msg) / sizeof (msg[0]),
		       "CreatePatternBrush() failed: GetLastError()=%d",
		       GetLastError ());
	    throw_AWTError (env, msg);
	  }
	else
	  {
	    // uV̌_ύX
	    SetBrushOrgEx (ngd->hDC, -ngd->paint.texture.anchorX,
			   -ngd->paint.texture.anchorY, NULL);
	  }

	if (hResized)
	  {
	    DeleteObject (hResized);
	  }
      }
      break;

    default:
      // ToDo: 
      throw_AWTError (env, "Unsupported Paint type");
    }
  return result;
}

/**
 * eLXgJ[ݒ肷
 */
static void
set_text_color (JNIEnv * env, native_graphics * ngr)
{
  COLORREF color;
  switch (ngr->paint.type)
    {
    case PAINT_TYPE_COLOR:
      color =
	ngr->xorMode ? ngr->paint.color ^ ngr->xorColor : ngr->paint.color;
      break;

    case PAINT_TYPE_GRADIENT_PAINT:
      // ToDo: Of[VT|[g
      color =
	ngr->xorMode ? ngr->paint.gradient.c1 ^ ngr->xorColor : ngr->paint.
	gradient.c1;
      break;

    default:
      // ToDo: Strokel
      // ToDo: Fy𐶐
      color = RGB (0, 0, 0);
    }
  SetTextColor (ngr->hDC, color);
}

/**
 * JgPaintݒ肷
 */
static void
set_paint (JNIEnv * env, native_graphics * ngr)
{
  // eLXgJ[ݒ肷
  set_text_color (env, ngr);
  // uV͕`sтɍ쐬
}

/***************************************************************************************************
 * JNI functions
 ***************************************************************************************************/

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    initIDs
 * Signature: ()V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_initIDs (JNIEnv * env, jclass clazz)
{
  g_getPixelsOfBufferedImage_mid =
    env->GetMethodID (clazz, "getPixelsOfBufferedImage",
		      "()Ljava/lang/Object;");
  assert (g_getPixelsOfBufferedImage_mid);

  g_updateBufferedImage_mid =
    env->GetMethodID (clazz, "updateBufferedImage", "()V");
  assert (g_updateBufferedImage_mid);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    createNativeGraphicsForScreen
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_createNativeGraphicsForScreen (JNIEnv
									* env,
									jobject
									obj,
									jint
									windowHandle)
{
  int result;
  HWND hWnd = (HWND) windowHandle;
  native_graphics *ngr =
    (native_graphics *) calloc (sizeof (native_graphics), 1);
  if (ngr)
    {
      ngr->hWnd = hWnd;
      ngr->hDC = GetDC (hWnd);
      init_DC (ngr->hDC);
      result = (jint) ngr;
    }
  else
    {
      throw_AWTError (env, "No memory");
      result = 0;
    }
  return result;
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    createNativeGraphicsForBitmap
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_createNativeGraphicsForBitmap (JNIEnv
									* env,
									jobject
									obj,
									jint
									windowHandle,
									jint
									bitmapHandle)
{
  HWND hWnd = (HWND) windowHandle;
  HBITMAP hBitmap = (HBITMAP) bitmapHandle;
  HDC hCompatibleDC = NULL;
  jint result;

  // native_graphics쐬
  native_graphics *ngr =
    (native_graphics *) calloc (sizeof (native_graphics), 1);

  if (!ngr)
    {
      throw_AWTError (env, "No memory");
      result = 0;
    }
  else
    {
      // createCompatible DC
      {
	HDC hDC = GetDC (hWnd);
	hCompatibleDC = CreateCompatibleDC (hDC);
	ReleaseDC (hWnd, hDC);
      }
      init_DC (hCompatibleDC);
      // save a default bitmap handle
      ngr->hDefaultBitmap =
	(HBITMAP) GetCurrentObject (hCompatibleDC, OBJ_BITMAP);
      ngr->hWnd = hWnd;
      ngr->hDC = hCompatibleDC;
      ngr->hBitmap = hBitmap;
      result = (jint) ngr;
    }
  return result;
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    createNativeGraphicsForBufferedImage
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_createNativeGraphicsForBufferedImage
  (JNIEnv * env, jobject obj, jint bitmapHandle)
{
  HBITMAP hBitmap = (HBITMAP) bitmapHandle;
  HDC hCompatibleDC = NULL;
  jint result;
  // create native_graphics
  native_graphics *ngr =
    (native_graphics *) calloc (sizeof (native_graphics), 1);
  if (!ngr)
    {
      throw_AWTError (env, "No memory");
      result = 0;
    }
  else
    {
      // create Compatible DC.
      {
	HDC hDC = GetDC (NULL);
	hCompatibleDC = CreateCompatibleDC (hDC);
	ReleaseDC (NULL, hDC);
      }
      init_DC (hCompatibleDC);
      // save a default bitmap handle
      ngr->hDefaultBitmap =
	(HBITMAP) GetCurrentObject (hCompatibleDC, OBJ_BITMAP);
      ngr->hWnd = NULL;
      ngr->hDC = hCompatibleDC;
      ngr->hBitmap = hBitmap;

      // set a flag
      ngr->bufferedImageGraphics = TRUE;
      result = (jint) ngr;
    }
  return result;
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    translateNative
 * Signature: (III)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_translateNative (JNIEnv * env,
							  jobject obj,
							  jint nativeGraphics,
							  jint x, jint y)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  assert (ngr);
  ngr->transform.translate ((float) x, (float) y);
}


/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativeColor
 * Signature: (IIII)V
 */
JNIEXPORT VOID JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativeColor (JNIEnv * env,
							 jobject obj,
							 jint nativeGraphics,
							 jint red, jint green,
							 jint blue)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;

  // ȑOɐݒ肳Ă\[X폜
  delete_paint (ngr);

  // w肳ꂽlݒ肷
  ngr->paint.type = PAINT_TYPE_COLOR;
  ngr->paint.color = RGB (red, green, blue);

  set_paint (env, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativePaintMode
 * Signature: (I)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativePaintMode (JNIEnv * env,
							     jobject obj,
							     jint
							     nativeGraphics)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  ngr->xorMode = FALSE;
  ngr->xorColor = 0;
  SetROP2 (ngr->hDC, R2_COPYPEN);

  set_paint (env, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativeXORMode
 * Signature: (IIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativeXORMode (JNIEnv * env,
							   jobject obj,
							   jint
							   nativeGraphics,
							   jint red,
							   jint green,
							   jint blue)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  ngr->xorMode = TRUE;
  ngr->xorColor = RGB (red, green, blue);
  SetROP2 (ngr->hDC, R2_XORPEN);

  set_paint (env, ngr);
}


/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativeFont
 * Signature: (II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativeFont (JNIEnv * env,
							jobject obj,
							jint nativeGraphics,
							jint fontHandle)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  SelectObject (ngr->hDC, (HFONT) fontHandle);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    copyNativeArea
 * Signature: (I)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_copyNativeArea (JNIEnv * env,
							 jobject peer_obj,
							 jint nativeGraphics,
							 int x,
							 int y, int width,
							 int height, int dx,
							 int dy)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  HDC hDC = ngr->hDC;

  PREPARE_BITMAP (ngr);
  BitBlt (hDC, x + dx, y + dy, width, height, hDC, x, y, SRCCOPY);
  UNPREPARE_BITMAP (ngr);

  // BufferedImageXV
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativeClip
 * Signature: (IZIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativeClip (JNIEnv * env,
							jobject obj,
							jint nativeGraphics,
							jboolean clipEnabled,
							jint clipx,
							jint clipy,
							jint clipWidth,
							jint clipHeight)
{
  native_graphics *ngd = (native_graphics *) nativeGraphics;
  if (clipEnabled)
    {
      HRGN region =
	CreateRectRgn (clipx, clipy, clipx + clipWidth + 1,
		       clipy + clipHeight + 1);
      if (!region)
	{
	  throw_AWTError (env, "CreateRectRgn() failed.");
	  return;
	}
      SelectClipRgn (ngd->hDC, region);
      DeleteObject (region);	// ɍ폜Ă悢
    }
  else
    {
      SelectClipRgn (ngd->hDC, NULL);
    }
}


/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    drawNativeLine
 * Signature: (IIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_drawNativeLine (JNIEnv * env,
							 jobject peer_obj,
							 jint nativeGraphics,
							 int x1, int y1,
							 int x2, int y2)
{
  native_graphics *ngd = (native_graphics *) nativeGraphics;
  HDC hDC = ngd->hDC;
  HPEN hPen = create_pen (env, ngd);
  if (hPen)
    {
      HGDIOBJ hOldObj = SelectObject (hDC, hPen);
      PREPARE_BITMAP (ngd);
      MoveToEx (hDC, x1, y1, NULL);
      LineTo (hDC, x2, y2);
      LineTo (hDC, x1, y1);
      UNPREPARE_BITMAP (ngd);

      // BufferedImageXV
      UPDATE_BUFFERED_IMAGE (env, peer_obj, ngd);

      SelectObject (hDC, hOldObj);
      DeleteObject (hPen);
    }
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    drawNativeRect
 * Signature: (IIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_drawNativeRect (JNIEnv * env,
							 jobject peer_obj,
							 jint nativeGraphics,
							 jint x, jint y,
							 jint width,
							 jint height)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  // ngr->transform.transform(&x, &y);

  HDC hDC = ngr->hDC;

  // uVƃy쐬
  // iuV NULL_BRUSH gj
  HBRUSH hBrush = (HBRUSH) SelectObject (hDC, GetStockObject (NULL_BRUSH));
  HPEN hPen = create_pen (env, ngr);
  if (hBrush && hPen)
    {
      HGDIOBJ hOldPen = SelectObject (hDC, hPen);
      PREPARE_BITMAP (ngr);

      Rectangle (hDC, x, y, x + width + 1, y + height + 1);

      // y폜
      SelectObject (hDC, hOldPen);
      DeleteObject (hPen);

      // uVɖ߂
      SelectObject (hDC, hBrush);

      UNPREPARE_BITMAP (ngr);
      // BufferedImageXV
      UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
    }
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    fillNativeRect
 * Signature: (IIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_fillNativeRect (JNIEnv * env,
							 jobject peer_obj,
							 jint nativeGraphics,
							 jint x, jint y,
							 jint width,
							 jint height)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;

  SELECT_BRUSH_FOR_PAINT (env, ngr, x, y, width, height);
  PREPARE_BITMAP (ngr);

  PatBlt (ngr->hDC, x, y, width, height, PATCOPY);
  UNPREPARE_BITMAP (ngr);
  UNSELECT_BRUSH_FOR_PAINT (env, ngr);

  // BufferedImageXV
  // UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
  update_buffered_image (env, peer_obj, ngr, x, y, width, height);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    clearNativeRect
 * Signature: (IIIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_clearNativeRect (JNIEnv * env,
							  jobject peer_obj,
							  jint nativeGraphics,
							  jint x, jint y,
							  jint width,
							  jint height,
							  jint rgb)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  HDC hdc = ngr->hDC;

  // NAFŃuV쐬
  HBRUSH brush = CreateSolidBrush (TO_COLORREF (rgb));
  HBRUSH oldbrush = (HBRUSH) SelectObject (hdc, brush);

  PREPARE_BITMAP (ngr);
  PatBlt (hdc, x, y, width + 1, height + 1, PATCOPY);
  UNPREPARE_BITMAP (ngr);

  // uV폜
  DeleteObject (SelectObject (hdc, oldbrush));

  // BufferedImageXV
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    nativeRoundRect
 * Signature: (IIIIIIIZ)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_nativeRoundRect (JNIEnv * env,
							  jobject peer_obj,
							  jint nativeGraphics,
							  jint x, jint y,
							  jint width,
							  jint height,
							  jint arcWidth,
							  jint arcHeight,
							  jboolean fill)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  HDC hdc = ngr->hDC;
  HBRUSH brush;
  int x2, y2;
  HGDIOBJ hPen = NULL, hOldPen = NULL;
  if (!fill)
    {
      // hԂȂ
      hPen = create_pen (env, ngr);
      hOldPen = SelectObject (hdc, hPen);
      brush = (HBRUSH) SelectObject (hdc, GetStockObject (NULL_BRUSH));
      x2 = x + width + 1;
      y2 = y + height + 1;
    }
  else
    {
      // hԂ
      hPen = GetStockObject (NULL_PEN);
      hOldPen = SelectObject (hdc, hPen);
      SELECT_BRUSH_FOR_PAINT (env, ngr, x, y, width, height);
      x2 = x + width;
      y2 = y + height;
    }

  PREPARE_BITMAP (ngr);
  RoundRect (hdc, x, y, x2, y2, arcWidth, arcHeight);
  UNPREPARE_BITMAP (ngr);

  if (!fill)
    {
      // hԂȂ
      assert (hOldPen && hPen);
      SelectObject (hdc, hOldPen);
      DeleteObject (hPen);
      SelectObject (hdc, brush);
    }
  else
    {
      // hԂ

      // yGetStockObject()̂Ȃ̂ŁADeleteObject()͕sv
      SelectObject (hdc, hOldPen);
      UNSELECT_BRUSH_FOR_PAINT (env, ngr);
    }

  // BufferedImageXV
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    nativeOval
 * Signature: (IIIIIZ)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_nativeOval (JNIEnv * env,
						     jobject peer_obj,
						     jint nativeGraphics,
						     jint x, jint y,
						     jint width, jint height,
						     jboolean fill)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  HDC hdc = ngr->hDC;
  HBRUSH brush;
  HGDIOBJ hPen = NULL, hOldPen = NULL;
  if (!fill)
    {
      // hԂȂ
      hPen = create_pen (env, ngr);
      hOldPen = SelectObject (hdc, hPen);
      brush = (HBRUSH) SelectObject (hdc, GetStockObject (NULL_BRUSH));
    }
  else
    {
      // hԂ
      hPen = GetStockObject (NULL_PEN);
      hOldPen = SelectObject (hdc, hPen);

      // ֊s`悵ȂA傫L
      x--;
      y--;
      width++;
      height++;
      SELECT_BRUSH_FOR_PAINT (env, ngr, x, y, width, height);
    }

  PREPARE_BITMAP (ngr);
  Ellipse (hdc, x, y, x + width, y + height);
  UNPREPARE_BITMAP (ngr);

  if (!fill)
    {
      // hԂȂ
      assert (hPen && hOldPen);
      DeleteObject (SelectObject (hdc, hOldPen));
      SelectObject (hdc, brush);
    }
  else
    {
      // hԂ
      UNSELECT_BRUSH_FOR_PAINT (env, ngr);

      // yGetStockObject()擾̂Ȃ̂ŁADeleteObject()͕sv
      SelectObject (hdc, hOldPen);
    }

  // BufferedImageXV
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    drawNativePolyline
 * Signature: (I[I[II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_drawNativePolyline (JNIEnv * env,
							     jobject peer_obj,
							     jint
							     nativeGraphics,
							     jintArray
							     xPoints,
							     jintArray
							     yPoints,
							     jint nPoints)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;

  int *x = env->GetIntArrayElements (xPoints, NULL);
  int *y = env->GetIntArrayElements (yPoints, NULL);
  POINT *pt = (POINT *) malloc (sizeof (POINT) * nPoints);

  if (x && y && pt)
    {
      // y쐬
      HPEN hPen = create_pen (env, ngr);
      if (hPen)
	{
	  HGDIOBJ hOldPen = SelectObject (ngr->hDC, hPen);
	  jint i;
	  for (i = 0; i < nPoints; ++i)
	    {
	      pt[i].x = x[i];
	      pt[i].y = y[i];
	    }
	  PREPARE_BITMAP (ngr);
	  if (!Polyline (ngr->hDC, pt, nPoints))
	    {
	      // AWTError𓊂
	      throw_AWTError (env, "Polyline() failed");
	    }
	  UNPREPARE_BITMAP (ngr);
	  // BufferedImageXV
	  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);

	  // y폜
	  SelectObject (ngr->hDC, hOldPen);
	  DeleteObject (hPen);
	}
    }
  free (pt);
  env->ReleaseIntArrayElements (xPoints, x, 0);
  env->ReleaseIntArrayElements (yPoints, y, 0);

}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    drawNativePolygon
 * Signature: (I[I[II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_drawNativePolygon (JNIEnv * env,
							    jobject peer_obj,
							    jint
							    nativeGraphics,
							    jintArray xPoints,
							    jintArray yPoints,
							    jint nPoints)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  int *x = env->GetIntArrayElements (xPoints, NULL);
  int *y = env->GetIntArrayElements (yPoints, NULL);
  POINT *pt = (POINT *) malloc (sizeof (POINT) * nPoints);
  HDC hdc = ngr->hDC;

  if (x && y && pt)
    {
      HGDIOBJ hPen = create_pen (env, ngr);
      if (hPen)
	{
	  HGDIOBJ hOldPen = SelectObject (hdc, hPen);
	  HBRUSH hbrushOrg =
	    (HBRUSH) SelectObject (hdc, GetStockObject (NULL_BRUSH));
	  jint i;
	  for (i = 0; i < nPoints; ++i)
	    {
	      pt[i].x = x[i];
	      pt[i].y = y[i];
	    }

	  PREPARE_BITMAP (ngr);
	  if (!Polygon (hdc, pt, nPoints))
	    {
	      // AWTError𓊂
	      throw_AWTError (env, "Polygon() failed");
	    }
	  UNPREPARE_BITMAP (ngr);
	  // BufferedImageXV
	  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);

	  // y폜
	  SelectObject (hdc, hbrushOrg);
	  DeleteObject (SelectObject (hdc, hOldPen));
	}
    }

  free (pt);
  env->ReleaseIntArrayElements (xPoints, x, 0);
  env->ReleaseIntArrayElements (yPoints, y, 0);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    fillNativePolygon
 * Signature: (I[I[II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_fillNativePolygon (JNIEnv * env,
							    jobject peer_obj,
							    jint
							    nativeGraphics,
							    jintArray xPoints,
							    jintArray yPoints,
							    jint nPoints)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  int *x = env->GetIntArrayElements (xPoints, NULL);
  int *y = env->GetIntArrayElements (yPoints, NULL);
  POINT *pt = (POINT *) malloc (sizeof (POINT) * nPoints);
  jint i;
  if (x && y && pt)
    {
      int minx = x[0];
      int miny = y[0];
      int maxx = minx;
      int maxy = miny;

      for (i = 0; i < nPoints; ++i)
	{
	  pt[i].x = x[i];
	  pt[i].y = y[i];
	  if (minx > x[i])
	    {
	      minx = x[i];
	    }
	  else if (maxx < x[i])
	    {
	      maxx = x[i];
	    }
	  if (miny > y[i])
	    {
	      miny = y[i];
	    }
	  else if (maxy < y[i])
	    {
	      maxy = y[i];
	    }
	}

      SELECT_BRUSH_FOR_PAINT (env, ngr, minx, miny, (maxx - minx),
			      (maxy - miny));

      PREPARE_BITMAP (ngr);
      if (!Polygon (ngr->hDC, pt, nPoints))
	{
	  // AWTError
	  throw_AWTError_with_error_code (env, "Polygon() failed", GetLastError());
	}
      UNPREPARE_BITMAP (ngr);
      UNSELECT_BRUSH_FOR_PAINT (env, ngr);
      free (pt);
    }
  env->ReleaseIntArrayElements (xPoints, x, 0);
  env->ReleaseIntArrayElements (yPoints, y, 0);

  // BufferedImageXV
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    fillNativePolyPolygon
 * Signature: (I[I[I[II)V
 */
JNIEXPORT void JNICALL
  Java_gnu_java_awt_peer_wce_WCEGraphics2D_fillNativePolyPolygon (JNIEnv *
								  env,
								  jobject obj,
								  jint
								  nativeGraphics,
								  jintArray
								  xPoints,
								  jintArray
								  yPoints,
								  jintArray
								  polyCounts,
								  jint
								  windingRule)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  int i;
  POINT *points;
  AWT_HRGN hrgn;
  int *x = env->GetIntArrayElements (xPoints, NULL);
  int *y = env->GetIntArrayElements (yPoints, NULL);
  int minx = x[0];
  int miny = y[0];
  int maxx = x[0];
  int maxy = y[0];
  int *counts = env->GetIntArrayElements (polyCounts, NULL);
  jsize xsize = env->GetArrayLength (xPoints);
  jsize ysize = env->GetArrayLength (yPoints);
  jsize psize = env->GetArrayLength (polyCounts);
  assert (xsize == ysize);

  points = (POINT *) malloc (sizeof (POINT) * xsize);
  if (!points)
    {
      throw_AWTError (env, "No memory");
      goto END;
    }
  for (i = 0; i < xsize; ++i)
    {
      points[i].x = x[i];
      points[i].y = y[i];
      if (minx > x[i])
	{
	  minx = x[i];
	}
      else if (maxx < x[i])
	{
	  maxx = x[i];
	}
      if (miny > y[i])
	{
	  miny = y[i];
	}
      else if (maxy < y[i])
	{
	  maxy = y[i];
	}
    }

  hrgn = AWT_CreatePolyPolygonRgn (points,
				   counts,
				   psize,
				   (windingRule ==
				    PathIterator_WIND_EVEN_ODD) ? ALTERNATE :
				   WINDING);
  if (!hrgn)
    {
      throw_AWTError (env, "Failed to create AWT_HRGN");
      goto END;
    }

  SELECT_BRUSH_FOR_PAINT (env, ngr, minx, miny, (maxx - minx), (maxy - miny));
  PREPARE_BITMAP (ngr);

  AWT_PatBltRegion (hrgn, ngr->hDC);

  UNPREPARE_BITMAP (ngr);
  UNSELECT_BRUSH_FOR_PAINT (env, ngr);

  AWT_DeleteRgn (hrgn);

END:
  free (points);

  env->ReleaseIntArrayElements (xPoints, x, 0);
  env->ReleaseIntArrayElements (yPoints, y, 0);
  env->ReleaseIntArrayElements (polyCounts, counts, 0);

}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    drawNativeString
 * Signature: (ILjava/lang/String;IIZ)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_drawNativeString (JNIEnv * env,
							   jobject peer_obj,
							   jint
							   nativeGraphics,
							   jstring str,
							   jint x, jint y,
							   jboolean xorMode)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  HDC hdc = ngr->hDC;

  jint len = env->GetStringLength (str);
  const jchar *buff = env->GetStringChars (str, NULL);
  TEXTMETRIC tm;
  GetTextMetrics (hdc, &tm);
  PREPARE_BITMAP (ngr);
  if (buff && len)
    {
      // adjust baseline
      RECT rect = { x, y - tm.tmAscent, x, y };
      if (xorMode)
	{
	  // ToDo: DrawText ignores SetROP2().
	  if (!DrawText
	      (hdc, (LPCWSTR) buff, len, &rect,
	       DT_TOP | DT_LEFT | DT_NOCLIP | DT_NOPREFIX | DT_SINGLELINE))
	    {
	      throw_AWTError_with_error_code (env, "DrawText() failed.",
					      GetLastError ());
	    }
	}
      else
	{
	  if (!DrawText
	      (hdc, (LPCWSTR) buff, len, &rect,
	       DT_TOP | DT_LEFT | DT_NOCLIP | DT_NOPREFIX | DT_SINGLELINE))
	    {
	      throw_AWTError_with_error_code (env, "DrawText() failed.",
					      GetLastError ());
	    }
	}
      env->ReleaseStringChars (str, buff);
    }
  UNPREPARE_BITMAP (ngr);
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    nativeBitBlt
 * Signature: (IIIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_nativeBitBlt (JNIEnv * env,
						       jobject obj,
						       jint
						       destinationNativeGraphics,
						       jint x, jint y,
						       jint width,
						       jint height,
						       jint
						       sourceWindowHandle,
						       jint
						       sourceBitmapHandle)
{
  native_graphics *dest_g = (native_graphics *) destinationNativeGraphics;
  HDC hDC = GetDC ((HWND) sourceWindowHandle);
  HDC hdcSrc = CreateCompatibleDC (hDC);
  HBITMAP hOldBitmap =
    (HBITMAP) SelectObject (hdcSrc, (HBITMAP) sourceBitmapHandle);
  ReleaseDC ((HWND) sourceWindowHandle, hDC);

  PREPARE_BITMAP (dest_g);
  if (!BitBlt (dest_g->hDC, x, y, width, height, hdcSrc, 0, 0, SRCCOPY))
    {
      DWORD lastError = GetLastError ();
      lastError = lastError;
    }
  UNPREPARE_BITMAP (dest_g);

  SelectObject (hdcSrc, hOldBitmap);
  DeleteDC (hdcSrc);

  UPDATE_BUFFERED_IMAGE (env, obj, dest_g);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    nativeTransparentBlt
 * Signature: (IIIIIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_nativeTransparentBlt (JNIEnv * env,
							       jobject obj,
							       jint
							       destinationNativeGraphics,
							       jint x, jint y,
							       jint width,
							       jint height,
							       jint
							       sourceWindowHandle,
							       jint
							       sourceBitmapHandle,
							       jint
							       transparentColor)
{
  native_graphics *dest_g = (native_graphics *) destinationNativeGraphics;
  HDC hDC = GetDC ((HWND) sourceWindowHandle);
  HDC hdcSrc = CreateCompatibleDC (hDC);
  HBITMAP hOldBitmap =
    (HBITMAP) SelectObject (hdcSrc, (HBITMAP) sourceBitmapHandle);
  ReleaseDC ((HWND) sourceWindowHandle, hDC);

  PREPARE_BITMAP (dest_g);
  TransparentBlt (dest_g->hDC,
		  x,
		  y,
		  width,
		  height,
		  hdcSrc, 0, 0, width, height, (COLORREF) transparentColor);
  UNPREPARE_BITMAP (dest_g);

  SelectObject (hdcSrc, hOldBitmap);
  DeleteDC (hdcSrc);

  UPDATE_BUFFERED_IMAGE (env, obj, dest_g);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    nativeStretchBlt
 * Signature: (IIIIIIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_nativeStretchBlt (JNIEnv * env,
							   jobject obj,
							   jint
							   nativeGraphics,
							   jint x, jint y,
							   jint width,
							   jint height,
							   jint
							   sourceWindowHandle,
							   jint
							   sourceBitmapHandle,
							   jint srcx,
							   jint srcy,
							   jint srcwidth,
							   jint srcheight)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  HDC hDC = GetDC ((HWND) sourceWindowHandle);
  HDC hdcSrc = CreateCompatibleDC (hDC);
  HBITMAP hOldBitmap =
    (HBITMAP) SelectObject (hdcSrc, (HBITMAP) sourceBitmapHandle);
  ReleaseDC ((HWND) sourceWindowHandle, hDC);

  PREPARE_BITMAP (ngr);

  StretchBlt (ngr->hDC,
	      x,
	      y,
	      width,
	      height, hdcSrc, srcx, srcy, srcwidth, srcheight, SRCCOPY);
  UNPREPARE_BITMAP (ngr);
  SelectObject (hdcSrc, hOldBitmap);
  DeleteDC (hdcSrc);

  UPDATE_BUFFERED_IMAGE (env, obj, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics
 * Method:    disposeNative
 * Signature: (II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_disposeNative (JNIEnv * env,
							jobject peer_obj,
							jint nativeGraphics)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;

  // foCXReLXgGDIIuWFNg𔍂Ă
  HDC hDC = ngr->hDC;

  // PenBrush폜
  DeleteObject (SelectObject (hDC, GetStockObject (NULL_PEN)));	// y
  DeleteObject (SelectObject (hDC, GetStockObject (NULL_BRUSH)));	// uV

  // NbsÖ폜(svHj
  SelectClipRgn (hDC, NULL);

  // foCXReLXg
  if (ngr->hBitmap)
    {
      // CreateCompatibleDC()ō쐬ꂽDC
      SelectObject (hDC, ngr->hDefaultBitmap);
      DeleteDC (hDC);
    }
  else
    {
      // GetDC()Ŏ擾DC
      ReleaseDC (ngr->hWnd, hDC);
    }

  // lCeBuyCg폜
  delete_paint (ngr);

  // J
  free (ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    drawNativeFreeTypeGlyphVector
 * Signature: (IIFF)V
 */
JNIEXPORT void JNICALL
  Java_gnu_java_awt_peer_wce_WCEGraphics2D_drawNativeFreeTypeGlyphVector
  (JNIEnv * env, jobject peer_obj, jint nativeGraphics,
   jint nativeFreeTypeGlyphVectorPointer, jfloat x, jfloat y)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  FreeTypeGlyphVector *ftgv =
    (FreeTypeGlyphVector *) nativeFreeTypeGlyphVectorPointer;
  int i;
  int numGlyphs = ftgv->getNumGlyphs ();
  FT_Pos boundsX, boundsY, boundsWidth, boundsHeight;

  ftgv->getLogicalBounds (&boundsX, &boundsY, &boundsWidth, &boundsHeight);

  SELECT_BRUSH_FOR_PAINT (env, ngr, (int) x, (int) y, (boundsWidth >> 6),
			  0 - (boundsHeight >> 6));
  PREPARE_BITMAP (ngr);

  HDC hdcSrc = CreateCompatibleDC (ngr->hDC);
  for (i = 0; i < numGlyphs; ++i)
    {
      HBITMAP hMaskBitmap = ftgv->createBitmapMask (i);
      BITMAP bitmap;
      GetObject (hMaskBitmap, sizeof (BITMAP), &bitmap);
      const FT_Vector & glyphPos = ftgv->getGlyphPosition (i);
      const FT_Int bitmapLeft = ftgv->getBitmapLeft (i);
      const FT_Int bitmapTop = ftgv->getBitmapTop (i);

      MaskBlt (ngr->hDC,
	       (int) x + (glyphPos.x >> 6) + bitmapLeft,
	       (int) y + (0 - (glyphPos.y >> 6)) - bitmapTop,
	       bitmap.bmWidth,
	       bitmap.bmHeight,
	       hdcSrc, 0, 0, hMaskBitmap, 0, 0, MAKEROP4 (PATCOPY, 0xAA0029));

      DeleteObject (hMaskBitmap);
    }
  DeleteDC (hdcSrc);

  UNPREPARE_BITMAP (ngr);
  UNSELECT_BRUSH_FOR_PAINT (env, ngr);

  // BufferedImageXV
  UPDATE_BUFFERED_IMAGE (env, peer_obj, ngr);
}


/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    createNativeDIBSection
 * Signature: (IIII)I
 */
JNIEXPORT jint JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_createNativeDIBSection (JNIEnv * env,
								 jobject obj,
								 jint width,
								 jint height,
								 jint
								 bitCount,
								 jint type)
{
  HBITMAP result = NULL;
  BOOL invalid_type = FALSE;

  BITMAPINFO *lpBitmapInfo = NULL;
  BITMAPINFO bi = { 0 };
  BITMAPINFO_RGB565 bi565 = { 0 };

  if (bitCount == 32)
    {
      // 32rbg
      if (type == BufferedImage_TYPE_INT_RGB
	  || type == BufferedImage_TYPE_INT_ARGB)
	{
	  bi.bmiHeader.biSize = sizeof (BITMAPINFOHEADER);
	  bi.bmiHeader.biBitCount = bitCount;
	  bi.bmiHeader.biCompression = BI_RGB;
	  bi.bmiHeader.biPlanes = 1;
	  bi.bmiHeader.biWidth = width;
	  bi.bmiHeader.biHeight = -height;	// gbv_EDIB
	  lpBitmapInfo = &bi;
	}
      else
	{
	  invalid_type = TRUE;
	}
    }
  else if (bitCount == 16)
    {
      // 16rbg
      if (type == BufferedImage_TYPE_USHORT_555_RGB)
	{
	  bi.bmiHeader.biSize = sizeof (BITMAPINFOHEADER);
	  bi.bmiHeader.biBitCount = bitCount;
	  bi.bmiHeader.biCompression = BI_RGB;
	  bi.bmiHeader.biPlanes = 1;
	  bi.bmiHeader.biWidth = width;
	  bi.bmiHeader.biHeight = -height;	// gbv_EDIB
	  lpBitmapInfo = &bi;
	}
      else if (type == BufferedImage_TYPE_USHORT_565_RGB)
	{
	  // RGB565
	  bi565.bmiHeader.biSize = sizeof (BITMAPINFOHEADER);
	  bi565.bmiHeader.biBitCount = bitCount;
	  bi565.bmiHeader.biCompression = BI_BITFIELDS;	// biBitCount = 16, biCompression = BI_BITFIELDS̏ꍇRGB565Ӗ
	  bi565.bmiHeader.biPlanes = 1;
	  bi565.bmiHeader.biWidth = width;
	  bi565.bmiHeader.biHeight = -height;	// gbv_EDIB
	  bi565.dwRedMask = 0x1f << (5 + 6);
	  bi565.dwGreenMask = 0x3f << 5;
	  bi565.dwBlueMask = 0x1f;
	  lpBitmapInfo = (BITMAPINFO *) & bi565;
	}
      else
	{
	  invalid_type = TRUE;
	}
    }
  else
    {
      // 16rbg, 32rbgȊO
      invalid_type = TRUE;
    }

  if (invalid_type)
    {
      char msg[128];
      _snprintf (msg,
		 sizeof (msg) / sizeof (msg[0]),
		 "Unsupported format bitCount=%d type=%d", bitCount, type);
      throw_AWTError (env, msg);
    }
  else
    {
      result = CreateDIBSection (NULL,
				 lpBitmapInfo, DIB_RGB_COLORS, NULL, NULL, 0);
      if (!result)
	{

	  DWORD dwLastError = GetLastError ();
	  char msg[128];
	  _snprintf (msg,
		     sizeof (msg) / sizeof (msg[0]),
		     "CreateDIBSection() failed. GetLastError()=" +
		     dwLastError);
	  throw_AWTError (env, msg);
	}
    }

  assert (result);
  return (jint) result;
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    deleteNativeObject
 * Signature: (I)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_deleteNativeObject (JNIEnv * env,
							     jobject obj,
							     jint
							     gdiObjectHandle)
{
  if (!DeleteObject ((HGDIOBJ) gdiObjectHandle))
    {
      char msg[128];
      _snprintf (msg, sizeof (msg) / sizeof (msg[0]),
		 "DeleteObject() failed. GetLastError()=%d", GetLastError ());
      throw_AWTError (env, msg);
    }
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    writeNativeDIBSectionInt
 * Signature: (I[II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_writeNativeDIBSectionInt (JNIEnv *
								   env,
								   jobject
								   obj,
								   jint
								   bitmapHandle,
								   jintArray
								   pixels,
								   jint type)
{
  BITMAP bitmap = { 0 };
  if (!GetObject ((HBITMAP) bitmapHandle, sizeof (BITMAP), &bitmap))
    {
      throw_AWTError_with_error_code (env, "GetObject() failed.",
				      GetLastError ());
      return;
    }

  if (bitmap.bmBitsPixel != 32)
    {
      // 32rbgłȂꍇ̓G[ɂ
      throw_AWTError (env, "Pixel format must be 32bit");
    }

  // ]sNZvZ
  const int array_length = env->GetArrayLength (pixels);
  const int pixel_count = bitmap.bmWidth * bitmap.bmHeight;
  if (array_length != pixel_count)
    {
      throw_AWTError (env, "array_length!=pixel_count");
    }

  jint *source = env->GetIntArrayElements (pixels, NULL);

  // copy pixel data
  if (bitmap.bmWidth * sizeof (DWORD) == bitmap.bmWidthBytes)
    {
      memcpy (bitmap.bmBits, source, sizeof (jint) * array_length);
    }
  else
    {
      char *dest = (char *) bitmap.bmBits;
      for (int i = 0; i < bitmap.bmHeight; ++i)
	{
	  memcpy (dest, source + (i * bitmap.bmWidth),
		  bitmap.bmWidth * sizeof (jint));
	  dest += bitmap.bmWidthBytes;
	}
    }
  env->ReleaseIntArrayElements (pixels, source, 0);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    writeNativeDIBSectionUShort
 * Signature: (I[SI)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_writeNativeDIBSectionUShort (JNIEnv *
								      env,
								      jobject
								      obj,
								      jint
								      bitmapHandle,
								      jshortArray
								      pixels,
								      jint
								      type)
{
  BITMAP bitmap = { 0 };
  if (!GetObject ((HBITMAP) bitmapHandle, sizeof (BITMAP), &bitmap))
    {
      throw_AWTError_with_error_code (env, "GetObject() failed.",
				      GetLastError ());
      return;
    }

  if (bitmap.bmBitsPixel != 16)
    {
      // 16rbgłȂꍇ̓G[ɂ
      throw_AWTError (env, "bitmap.bmBitsPixel!=16");
      return;
    }

  // check array size.
  const int array_length = env->GetArrayLength (pixels);
  const int pixel_count = bitmap.bmWidth * bitmap.bmHeight;
  if (pixel_count != array_length)
    {
      throw_AWTError (env, "pixel_count!=array_length");
      return;
    }

  // copy pixel data
  jshort *source = env->GetShortArrayElements (pixels, NULL);
  if (bitmap.bmWidth * sizeof (WORD) == bitmap.bmWidthBytes)
    {
      memcpy (bitmap.bmBits, source, sizeof (jshort) * array_length);
    }
  else
    {
      char *dest = (char *) bitmap.bmBits;
      for (int i = 0; i < bitmap.bmHeight; ++i)
	{
	  memcpy (dest, source + i * bitmap.bmWidth,
		  bitmap.bmWidth * sizeof (jshort));
	  dest += bitmap.bmWidthBytes;
	}
    }
  env->ReleaseShortArrayElements (pixels, source, 0);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    readNativeDIBSectionInt
 * Signature: (I[II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_readNativeDIBSectionInt (JNIEnv *
								  env,
								  jobject obj,
								  jint
								  bitmapHandle,
								  jintArray
								  pixels,
								  jint type)
{
  BITMAP bitmap = { 0 };
  if (!GetObject ((HBITMAP) bitmapHandle, sizeof (BITMAP), &bitmap))
    {
      throw_AWTError_with_error_code (env, "GetObject() failed.",
				      GetLastError ());
      return;
    }

  if (bitmap.bmBitsPixel != 32)
    {
      throw_AWTError (env, "bitmap.bmBitsPixel != 32");
      return;
    }

  // check array size.
  const int array_length = env->GetArrayLength (pixels);
  const int pixel_count = bitmap.bmWidth * bitmap.bmHeight;
  if (pixel_count != array_length)
    {
      throw_AWTError (env, "pixel_count!=array_length");
      return;
    }

  // copy pixel data
  jint *dest = env->GetIntArrayElements (pixels, NULL);
  if (bitmap.bmWidth * sizeof (DWORD) == bitmap.bmWidthBytes)
    {
      memcpy (dest, bitmap.bmBits, sizeof (jint) * array_length);
    }
  else
    {
      char *dib = (char *) bitmap.bmBits;
      for (int i = 0; i < bitmap.bmHeight; ++i)
	{
	  memcpy (dest + i * bitmap.bmWidth, dib,
		  bitmap.bmWidth * sizeof (jint));
	  dib += bitmap.bmWidthBytes;
	}
    }
  env->ReleaseIntArrayElements (pixels, dest, 0);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    readNativeDIBSectionUShort
 * Signature: (I[SI)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_readNativeDIBSectionUShort (JNIEnv *
								     env,
								     jobject
								     obj,
								     jint
								     bitmapHandle,
								     jshortArray
								     pixels,
								     jint
								     type)
{
  BITMAP bitmap = { 0 };
  if (!GetObject ((HBITMAP) bitmapHandle, sizeof (BITMAP), &bitmap))
    {
      throw_AWTError_with_error_code (env, "GetObject() failed.",
				      GetLastError ());
      return;
    }

  if (bitmap.bmBitsPixel != 16)
    {
      throw_AWTError (env, "bitmap.bmBitsPixel != 16");
      return;
    }

  // check array size.
  const int array_length = env->GetArrayLength (pixels);
  const int pixel_count = bitmap.bmWidth * bitmap.bmHeight;
  if (pixel_count != array_length)
    {
      throw_AWTError (env, "pixel_count!=array_length");
      return;
    }

  // copy pixel data
  jshort *dest = env->GetShortArrayElements (pixels, NULL);
  if (bitmap.bmWidth * sizeof (WORD) == bitmap.bmWidthBytes)
    {
      memcpy (dest, bitmap.bmBits, sizeof (jshort) * array_length);
    }
  else
    {
      char *dib = (char *) bitmap.bmBits;
      for (int i = 0; i < bitmap.bmHeight; ++i)
	{
	  memcpy (dest + i * bitmap.bmWidth, dib,
		  bitmap.bmWidth * sizeof (jshort));
	  dib += bitmap.bmWidthBytes;
	}
    }
  env->ReleaseShortArrayElements (pixels, dest, 0);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativeGradientPaint
 * Signature: (IIIIIIIIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativeGradientPaint (JNIEnv * env,
								 jobject obj,
								 jint
								 nativeGraphics,
								 jint x1,
								 jint y1,
								 jint red1,
								 jint green1,
								 jint blue1,
								 jint x2,
								 jint y2,
								 jint red2,
								 jint green2,
								 jint blue2,
								 jboolean
								 cyclic)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  gradient_paint *gpaint;

  // ȑOɐݒ肳Ă\[X폜
  delete_paint (ngr);

  gpaint = &ngr->paint.gradient;
  ngr->paint.type = PAINT_TYPE_GRADIENT_PAINT;

  gpaint->x1 = x1;
  gpaint->y1 = y1;
  gpaint->c1 = RGB (red1, green1, blue1);

  gpaint->x2 = x2;
  gpaint->y2 = y2;
  gpaint->c2 = RGB (red2, green2, blue2);

  gpaint->cyclic = (cyclic == (jboolean) JNI_TRUE) ? TRUE : FALSE;

  set_paint (env, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativeTexturePaint
 * Signature: (IIIIII)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativeTexturePaint (JNIEnv * env,
								jobject obj,
								jint
								nativeGraphics,
								jint
								bitmapHandle,
								jint anchorX,
								jint anchorY,
								jint
								anchorWidth,
								jint
								anchorHeight)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  assert (ngr);

  ngr->paint.type = PAINT_TYPE_TEXTURE_PAINT;
  ngr->paint.texture.hBitmap = (HBITMAP) bitmapHandle;
  ngr->paint.texture.anchorX = anchorX;
  ngr->paint.texture.anchorY = anchorY;
  ngr->paint.texture.anchorWidth = anchorWidth;
  ngr->paint.texture.anchorHeight = anchorHeight;

  set_paint (env, ngr);
}

/*
 * Class:     gnu_java_awt_peer_wce_WCEGraphics2D
 * Method:    setNativePenWidth
 * Signature: (II)V
 */
JNIEXPORT void JNICALL
Java_gnu_java_awt_peer_wce_WCEGraphics2D_setNativePenWidth (JNIEnv * env,
							    jobject obj,
							    jint
							    nativeGraphics,
							    jint width)
{
  native_graphics *ngr = (native_graphics *) nativeGraphics;
  assert (ngr);
  ngr->penWidth = width;
}
