/* TG01Accelerometer.java
   Copyright (C) 2009 freebeans

This file is part of Mysaifu JVM.

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package com.mysaifu.mobile.device.tg01;

import java.awt.Frame;
import java.util.TooManyListenersException;
import java.security.AccessController;
import java.security.PrivilegedAction;

/**
 * Toshiba TG01 accelerometer API library. I wish to express my gratitude to the
 * developer of these excellent information. <a href="http://smartmobidevice.blogspot.com/2009/08/toshiba-tg01-and-accelerometer.html"
 * >Toshiba TG01 and Accelerometer</a>
 * <p>
 * Usage:<br>
 * 0. Check TG01 device <code><pre>
 * if (TG01Accelerometer.isAvailable())
 *  {
 *    ...
 *  }
 * else
 *  {
 *   // Cannot use TG01Accelerometer class. 
 *  } 
 * </pre></code> 1. Get the instance. <code><pre>
 * TG01Accelerometer acc = TG01Accelerometer.getInstance();
 * </code></pre> 2. Add listener<code><pre>
 * acc.addTG01AccelerationListener(new TG01AccelerationListener() { ... });
 * acc.addTG01ActionListener(new TG01ActionListener() { ... });
 * </pre></code> 3. Open acceleromter device. <code><pre>
 * acc.open();
 * </pre></code> 4. Close accelerometer device. <code><pre>
 * acc.close();
 * </pre></code>
 */
public class TG01Accelerometer
{
  /**
   * Default interval.
   */
  public final int DEFAULT_INTERVAL = 200;

  /**
   * Singleton.
   */
  private static TG01Accelerometer INSTANCE;

  /**
   * open flag.
   */
  private boolean opened;

  /**
   * TG01AccelerationListener
   */
  private TG01AccelerationListener accelerationListener;

  /**
   * TG01ActionListener
   */
  private TG01ActionListener actionListener;

  /**
   * event notification thread.
   */
  private Thread notificationThread;

  /**
   * Foreground frame.
   * 
   * @see #suspendSystemActionEventResponse(Frame)
   */
  private Frame foregroundFrame;

  private TG01Accelerometer() throws TG01Exception
  {
    // Load native library.
    if (System.getSecurityManager() != null)
      {
        AccessController.doPrivileged(new PrivilegedAction<Void>()
        {
          public Void run()
          {
            System.loadLibrary("devtg01");
            return null;
          }
        });
      }
    else
      {
        System.loadLibrary("devtg01");
      }
    initNative();
  }

  private native void initNative() throws TG01Exception;

  /**
   * Creates new instance.
   * 
   * @return TG01 accelerometer device instance.
   * @throws java.lang.UnsatisfiedLinkError Cannot detect TG01 device.
   */
  public synchronized static TG01Accelerometer getInstance()
      throws TG01Exception
  {
    if (INSTANCE == null)
      {
        INSTANCE = new TG01Accelerometer();
      }
    return INSTANCE;
  }

  /**
   * Checks TG01 accelerometer is available.
   * 
   * @return true if available.
   */
  public static boolean isAvailable()
  {
    try
      {
        getInstance();
      }
    catch (TG01Exception te)
      {
        return false;
      }
    catch (UnsatisfiedLinkError ule)
      {
        return false;
      }
    return true;
  }

  /**
   * open accelerometer (interval is 100ms)
   * 
   * @throws TG01Exception
   */
  public void open() throws TG01Exception
  {
    open(DEFAULT_INTERVAL);
  }

  /**
   * open accelerometer with specified interval.
   * 
   * @param interval accelerometer intarval.
   * @throws TG01Exception error occurred.
   */
  public synchronized void open(int interval) throws TG01Exception
  {
    if (interval <= 0)
      {
        throw new IllegalArgumentException("interval<=0");
      }
    if (this.opened)
      {
        throw new TG01Exception("Already open.");
      }
    this.opened = true;

    if (interval < 100)
      {
        // minimum value is 100
        interval = 100;
      }
    else if ((interval % 100) != 0)
      {
        // minimum resolution is 100.
        interval += (100 - (interval % 100));
      }
    // first, start notification thread.
    startNotificationThread(interval);

    // second, open native device.
    openNative(interval);
  }

  private native void openNative(int interval) throws TG01Exception;

  /**
   * close accelerometer.
   * 
   * @throws TG01Exception error occurred.
   */
  public synchronized void close() throws TG01Exception
  {
    if (this.opened)
      {
        this.opened = false;
        closeNative();
        this.notificationThread = null;
      }
  }

  private native void closeNative() throws TG01Exception;

  /**
   * Add specified listener. Only one listener can be added.
   * 
   * @param listener event listener.
   * @throws TooManyListenersException listener already exists.
   */
  public void addTG01AccelerationListener(TG01AccelerationListener listener)
      throws TooManyListenersException
  {
    if (this.accelerationListener != null)
      {
        throw new TooManyListenersException();
      }
    this.accelerationListener = listener;
  }

  /**
   * Remove specified listener.
   * 
   * @param listener event listener.
   */
  public void removeTG01AccelerationListener(TG01AccelerationListener listener)
  {
    if (this.accelerationListener == listener)
      {
        this.accelerationListener = null;
      }
  }

  /**
   * Return registered <code>TG01AccelerationListener</code>
   * 
   * @return Registered listener, or <code>null</code> if no listener exists.
   */
  protected TG01AccelerationListener getTG01AccelerationListener()
  {
    return this.accelerationListener;
  }

  /**
   * Add specified listener. Only one listener can be added.
   * 
   * @param listener event listener.
   * @throws TooManyListenersException listener already exists.
   */
  public void addTG01ActionListener(TG01ActionListener listener)
      throws TooManyListenersException
  {
    if (this.actionListener != null)
      {
        throw new TooManyListenersException();
      }
    this.actionListener = listener;
  }

  /**
   * Remove specified listener.
   * 
   * @param listener event listener.
   */
  public void removeTG01ActionListener(TG01ActionListener listener)
  {
    if (this.actionListener == listener)
      {
        this.actionListener = null;
      }
  }

  /**
   * Return registered <code>TG01ActionListener</code>
   * 
   * @return Registered listener, or <code>null</code> if no listener exists.
   */
  protected TG01ActionListener getTG01ActionListener()
  {
    return this.actionListener;
  }

  /**
   * Notify specified event to the registered listener.
   * 
   * @param eventType Event type.
   * @param eventParameter Event parameter.
   */
  static void notifyTG01AccelerationListener(int x, int y, int z)
  {
    if (INSTANCE == null)
      {
        throw new IllegalStateException("INSTANCE==null");
      }
    TG01AccelerationListener listener = INSTANCE.getTG01AccelerationListener();
    if (listener != null)
      {
        listener.notify(new TG01AccelerationEvent(INSTANCE, x, y, z));
      }
  }

  /**
   * Notify specified event to the registered listener.
   * 
   * @param eventType Event type.
   * @param eventParameter Event parameter.
   */
  static void notifyTG01ActionListener(int eventType, int eventParameter)
  {
    if (INSTANCE == null)
      {
        throw new IllegalStateException("INSTANCE==null");
      }
    TG01ActionListener listener = INSTANCE.getTG01ActionListener();
    if (listener != null)
      {
        TG01ActionEvent event = new TG01ActionEvent(INSTANCE, eventType,
                                                    eventParameter);
        switch (eventType)
          {
          case TG01ActionEvent.TYPE_ROTATE:
            listener.rotated(event);
            break;

          case TG01ActionEvent.TYPE_SHAKE:
            listener.shaked(event);
            break;

          case TG01ActionEvent.TYPE_DOUBLE_TAP:
            listener.doubleTapped(event);
            break;

          case TG01ActionEvent.TYPE_TILT:
            listener.tilted(event);
            break;
          }
      }
  }

  /**
   * Suspend system action response. While specified <code>Frame</code> is a
   * foreground window, system action response is suspended.
   * 
   * @param foregroundFrame <code>Frame</code> object.
   * @throws NullPointerException <code>foregroundFrame</code> is
   *           <code>null</code>.
   * @throws IllegalArgumentException
   *           <code>foregroundFrame.isDisplayable()</code> returns
   *           <code>false</code>.
   * @throws TG01Exception error occurred.
   * @see java.awt.Component#isDisplayable()
   */
  public synchronized void suspendSystemActionResponse(Frame foregroundFrame)
      throws TG01Exception, NullPointerException, IllegalArgumentException
  {
    if (foregroundFrame == null)
      {
        throw new NullPointerException("foregroundFrame");
      }
    if (! foregroundFrame.isDisplayable())
      {
        throw new IllegalArgumentException(
                                           "foregroundFrame.isDisplayable() returns false");
      }
    this.foregroundFrame = foregroundFrame;
    suspendNativeSystemActionResponse(foregroundFrame);
  }

  private native void suspendNativeSystemActionResponse(Frame foregroundFrame);

  /**
   * Resume system action response.
   * 
   * @throws TG01Exception error occurred.
   */
  public synchronized void resumeSystemActionResponse() throws TG01Exception
  {
    if (this.foregroundFrame != null)
      {
        resumeNativeSystemActionResponse(this.foregroundFrame);
        this.foregroundFrame = null;
      }
  }

  private native void resumeNativeSystemActionResponse(Frame foregroundFrame);

  @Override
  protected void finalize() throws Exception
  {
    close();
  }

  /**
   * Start notification thread.
   */
  private void startNotificationThread(final int interval)
  {
    this.notificationThread = new Thread()
    {
      public void run()
      {
        nativeLoop(interval);
      }
    };
    this.notificationThread.start();
  }

  private native void nativeLoop(int interval);
}
