/* com_mysaifu_mobile_device_tg01_TG01Accelerometer.cpp
   Copyright (C) 2009 freebeans

This file is part of Mysaifu JVM.

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

/**
 * Toshiba TG01 accelerometer API library.
 *
 * I wish to express my gratitude to the developer of these excellent information.
 * http://smartmobidevice.blogspot.com/2009/08/toshiba-tg01-and-accelerometer.html
 */

#include "stdafx.h"
#include "jawt_md.h"
#include "com_mysaifu_mobile_device_tg01_TG01Accelerometer.h"

#define TG01EXCEPTION_CLASS_NAME  "com.mysaifu.mobile.device.tg01.TG01Exception"
#define MESSAGE_WINDOW_CLASS_NAME  _T("com_mysaifu_mobile_device_tg01_TG01ActionRecogniser")

#define APPMSG_OPEN (WM_APP+1)
#define APPMSG_CLOSE (APPMSG_OPEN+1)
#define APPMSG_SUSPEND_SYSTEM_ACTION_RESPONSE (APPMSG_OPEN+2)
#define APPMSG_RESUME_SYSTEM_ACTION_RESPONSE (APPMSG_OPEN+3)

/**
 * TG01 action event
 */
#define TS_ACTION_EVENT (0x80c8)

/**
 * Timer ID
 */
#define TIMER_ID  1

/**
 * TG01 APIs.
 */
/**
 * TG01 sensor API struct.
 */
typedef struct tagTS_ACCELERATION
{
  SHORT x;
  SHORT y;
  SHORT z;
  FILETIME time;
} TS_ACCELERATION;

typedef DWORD (*TSREGISTERACCELERATION)(HMODULE,UINT, PUINT);
typedef DWORD (*TSDEREGISTERACCELERATION)(UINT);
typedef DWORD (*TSGETACCELERATION)(UINT, TS_ACCELERATION* accel);

typedef DWORD (*TSREGISTEACTIONEVENT)(HWND, HMODULE,PUINT);
typedef DWORD (*TSDEREGISTERACTIONEVENT)(UINT);
typedef DWORD (*TSSUSPENDSYSTEMACTIONRESPONSE)(HWND);
typedef DWORD (*TSRESUMESYSTEMACTIONRESPONSE)(HWND);


static TSREGISTEACTIONEVENT               TSRegisterActionEvent;
static TSDEREGISTERACTIONEVENT            TSDeregisterActionEvent;
static TSSUSPENDSYSTEMACTIONRESPONSE TSSuspendSystemActionResponse;
static TSRESUMESYSTEMACTIONRESPONSE  TSResumeSystemActionResponse;

static TSREGISTERACCELERATION TSRegisterAcceleration;
static TSDEREGISTERACCELERATION TSDeregisterAcceleration;
static TSGETACCELERATION TSGetAcceleration;

// static variables.
static bool g_bInitialized;
static HWND g_hwndMessage;
static bool g_bRegisterClass;

/**
 * ID of TSRegisterAcceleration().
 */
static UINT g_accelerationID;

/**
 * ID of TSRegisterActionEvent().
 */
static UINT g_actionEventID;

static jclass g_TG01AccelerometerClass;
static jmethodID g_TG01Accelerometer_notifyTG01AccelerationListener_mid;
static jmethodID g_TG01Accelerometer_notifyTG01ActionListener_mid;

static jclass g_TG01ActionListenerClass;
static jmethodID g_TG01ActionListener_shaked_mid;
static jmethodID g_TG01ActionListener_rotated_mid;
static jmethodID g_TG01ActionListener_doubleTapped_mid;
static jmethodID g_TG01ActionListener_tilted_mid;

// initialized by DLLMain().
HANDLE g_hNotificationThreadInitialized;

/**
 * Window procedule.
 */
LRESULT CALLBACK WndProc(HWND hWnd, UINT msg, WPARAM wp, LPARAM lp);

/**
 * Static functions declarations.
 */
static bool init(JNIEnv* env);
static bool initTG01Accelerometer(JNIEnv* env);
static bool initIDs(JNIEnv* env);
static bool checkReturnCode(JNIEnv* env, const char* functionName, DWORD code);

extern JavaVM* g_javaVM;

#define GET_HWND(hwndResult, foregroundObject, awt, ds) \
{ \
  hwndResult = NULL; \
  (awt).version = JAWT_VERSION_1_3; \
  if (JAWT_GetAWT(env, &(awt)) == static_cast<jboolean>(JNI_TRUE)) \
  { \
    (ds) = awt.GetDrawingSurface(env, foregroundObject); \
    if ((ds) != NULL) \
    { \
      jint lock = (ds)->Lock((ds)); \
      if((lock & JAWT_LOCK_ERROR) == 0) \
      { \
        JAWT_DrawingSurfaceInfo* dsi = (ds)->GetDrawingSurfaceInfo(ds); \
        if ((dsi) != NULL) \
        { \
          JAWT_Win32DrawingSurfaceInfo* dsi_win = (JAWT_Win32DrawingSurfaceInfo*) (dsi)->platformInfo; \
          hwndResult = dsi_win->hwnd; \
        } \
      } \
    } \
  } \
}

#define RELEASE_HWND(hwnd, awt, ds) \
{ \
  (ds)->Unlock((ds)); \
  (awt).FreeDrawingSurface((ds)); \
  hwnd = NULL; \
}

static bool init(JNIEnv* env)
{
  if (g_bInitialized)
  {
    return true;
  }
  if (! initTG01Accelerometer(env))
    return false;
  if (! initIDs(env))
    return false;

  g_hNotificationThreadInitialized = CreateEvent(NULL, FALSE, FALSE, NULL);
  g_bInitialized = true;

  return true;
}

/**
 * Initializer TG01 accelerometer.
 */
static bool initTG01Accelerometer(JNIEnv* env)
{
  // Get function pointers from "axcon.dll".
  HMODULE hModule = LoadLibrary(_T("axcon"));
  if (hModule)
    {
      TSRegisterAcceleration   = (TSREGISTERACCELERATION)   GetProcAddress(hModule, _T("TSRegisterAcceleration"));
      TSDeregisterAcceleration = (TSDEREGISTERACCELERATION) GetProcAddress(hModule, _T("TSDeregisterAcceleration"));
      TSGetAcceleration        = (TSGETACCELERATION)        GetProcAddress(hModule, _T("TSGetAcceleration"));
      
      TSRegisterActionEvent   = (TSREGISTEACTIONEVENT)   GetProcAddress(hModule, _T("TSRegisterActionEvent"));
      TSDeregisterActionEvent = (TSDEREGISTERACTIONEVENT) GetProcAddress(hModule, _T("TSDeregisterActionEvent"));

      TSSuspendSystemActionResponse = (TSSUSPENDSYSTEMACTIONRESPONSE) GetProcAddress(hModule, _T("TSSuspendSystemActionResponse"));
      TSResumeSystemActionResponse  = (TSRESUMESYSTEMACTIONRESPONSE) GetProcAddress(hModule, _T("TSResumeSystemActionResponse"));

      if (TSRegisterAcceleration && TSDeregisterAcceleration && TSGetAcceleration
        && TSRegisterActionEvent && TSDeregisterActionEvent && TSSuspendSystemActionResponse && TSResumeSystemActionResponse)
        {
          return true;
        }
    }

  env->ThrowNew(env->FindClass(TG01EXCEPTION_CLASS_NAME), "Failed to initialize TG01 accelerometer");
  return false;
}

/**
 * Initialize jFieldIDs, and jmethodIDs
 *
 * @param env
 */
static bool initIDs(JNIEnv* env)
{
#define NONNULL(obj) { if (! obj) { env->ThrowNew(env->FindClass(TG01EXCEPTION_CLASS_NAME), #obj); return false; } }
  
  jclass tmp = env->FindClass("com/mysaifu/mobile/device/tg01/TG01Accelerometer");
  NONNULL(tmp);
  g_TG01AccelerometerClass = static_cast<jclass>(env->NewGlobalRef(tmp));
  NONNULL(g_TG01AccelerometerClass);
  env->DeleteLocalRef(tmp);

  g_TG01Accelerometer_notifyTG01AccelerationListener_mid = env->GetStaticMethodID(g_TG01AccelerometerClass, "notifyTG01AccelerationListener", "(III)V");
  NONNULL(g_TG01Accelerometer_notifyTG01AccelerationListener_mid);

  g_TG01Accelerometer_notifyTG01ActionListener_mid = env->GetStaticMethodID(g_TG01AccelerometerClass, "notifyTG01ActionListener", "(II)V");
  NONNULL(g_TG01Accelerometer_notifyTG01ActionListener_mid);

  tmp = env->FindClass("com/mysaifu/mobile/device/tg01/TG01ActionListener");
  NONNULL(tmp);
  g_TG01ActionListenerClass = static_cast<jclass>(env->NewGlobalRef(tmp));
  NONNULL(g_TG01ActionListenerClass);
  env->DeleteLocalRef(tmp);

  g_TG01ActionListener_shaked_mid = env->GetMethodID(g_TG01ActionListenerClass, "shaked", "(Lcom/mysaifu/mobile/device/tg01/TG01ActionEvent;)V");
  NONNULL(g_TG01ActionListener_shaked_mid);

  g_TG01ActionListener_rotated_mid = env->GetMethodID(g_TG01ActionListenerClass, "rotated", "(Lcom/mysaifu/mobile/device/tg01/TG01ActionEvent;)V");
  NONNULL(g_TG01ActionListener_rotated_mid);

  g_TG01ActionListener_doubleTapped_mid = env->GetMethodID(g_TG01ActionListenerClass, "doubleTapped", "(Lcom/mysaifu/mobile/device/tg01/TG01ActionEvent;)V");
  NONNULL(g_TG01ActionListener_doubleTapped_mid);

  g_TG01ActionListener_tilted_mid = env->GetMethodID(g_TG01ActionListenerClass, "tilted", "(Lcom/mysaifu/mobile/device/tg01/TG01ActionEvent;)V");
  NONNULL(g_TG01ActionListener_tilted_mid);

  return true;
}

/**
 * Checks return value of TSXXXX() APIs.
 *
 * @param env JNIEnv.
 * @param functionName
 * @param code  The return value of TSRegisterAcceleration(), TSGetAcceleration(), and TSDeregisterAcceleration()
 * @return  true if OK.
 */
static bool checkReturnCode(JNIEnv* env, const char* functionName, DWORD code)
{
  if (code == 0)
    {
      return true;
    }

  // throw IOException.
  const char* format = "%s failed. return code=%d";
  char msg[128];
  sprintf(msg, format, functionName, code);
  env->ThrowNew(env->FindClass(TG01EXCEPTION_CLASS_NAME), msg);
  return false;
}

/* Inaccessible static: INSTANCE */
/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    initNative
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_initNative
  (JNIEnv *env, jobject)
{
  init(env);
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    openNative
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_openNative
  (JNIEnv *env, jobject , jint interval)
{
  // Wait for notification thread initialized.
  WaitForSingleObject(g_hNotificationThreadInitialized, INFINITE);

  DWORD retcode = TSRegisterAcceleration(GetModuleHandle(NULL), (UINT) interval, &g_accelerationID);
  if (checkReturnCode(env, "TSRegisterAcceleration()", retcode))
  {
    DWORD retcode2 = TSRegisterActionEvent(g_hwndMessage, GetModuleHandle(NULL), &g_actionEventID);
    if (checkReturnCode(env, "TSRegisterActionEvent", retcode2))
      {
        // Set timer which notifies TG01AccelerationEvent.
        SetTimer(g_hwndMessage, TIMER_ID, static_cast<UINT>(interval), NULL);
      }
  }
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    closeNative
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_closeNative
  (JNIEnv *env, jobject)
{
  KillTimer(g_hwndMessage, TIMER_ID);

  DWORD retcode = TSDeregisterAcceleration(g_accelerationID);
  DWORD retcode2 = TSDeregisterActionEvent(g_actionEventID);

  g_accelerationID = 0;
  g_actionEventID = 0;

  checkReturnCode(env,"TSDeregisterAcceleration()", retcode);
  checkReturnCode(env, "TSDeregisterActionEvent()", retcode);
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    suspendNativeSystemActionResponse
 * Signature: (Ljava/awt/Frame;)V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_suspendNativeSystemActionResponse
  (JNIEnv *env, jobject, jobject foregroundFrame)
{
  JAWT awt;
  JAWT_DrawingSurface* ds;
  HWND hWnd;

  GET_HWND(hWnd, foregroundFrame, awt, ds);
  if (! hWnd)
    {
      while (true)
        {
          HWND hwndParent = GetParent(hWnd);
          if (!hwndParent)
            break;
          hWnd = hwndParent;
        }
    }
  DWORD retcode = TSSuspendSystemActionResponse(hWnd);

  RELEASE_HWND(hWnd, awt, ds);

  checkReturnCode(env, "TSSuspendSystemActionResponse()", retcode);
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    resumeNativeSystemActionResponse
 * Signature: (Ljava/awt/Frame;)V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_resumeNativeSystemActionResponse
  (JNIEnv *env, jobject, jobject foregroundFrame)
{
  JAWT awt;
  JAWT_DrawingSurface* ds;
  HWND hWnd;

  GET_HWND(hWnd, foregroundFrame, awt, ds);
  if (! hWnd)
    {
      while (true)
        {
          HWND hwndParent = GetParent(hWnd);
          if (!hwndParent)
            break;
          hWnd = hwndParent;
        }
    }
  DWORD retcode = TSResumeSystemActionResponse(hWnd);
  RELEASE_HWND(hWnd, awt, ds);

  checkReturnCode(env, "TSResumeSystemActionResponse()", retcode);
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    nativeLoop
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_nativeLoop
  (JNIEnv *env, jobject, jint interval)
{
  // Register message window class.
  if (! g_bRegisterClass)
    {
      WNDCLASS wndclass = {0};
      wndclass.hInstance = static_cast<HINSTANCE>(GetModuleHandle(NULL));
      wndclass.lpszClassName = MESSAGE_WINDOW_CLASS_NAME;
      wndclass.lpfnWndProc = WndProc;

      if (! RegisterClass(&wndclass))
        {
          env->ThrowNew(env->FindClass("java/lang/Error"), "RegisterClass() failed");
          return;
        }
      g_bRegisterClass = true;
    }

  // Create the message window.
  g_hwndMessage = CreateWindow(MESSAGE_WINDOW_CLASS_NAME,
                               _T(""),
                               0,
                               0, 0, 0, 0,
                               HWND_DESKTOP,
                               NULL,
                               GetModuleHandle(NULL),
                               NULL);
  if (! g_hwndMessage)
    {
      env->ThrowNew(env->FindClass(TG01EXCEPTION_CLASS_NAME), "CreateWindow() failed");
      return;
    }

  // notify to nativeOpen().
  SetEvent(g_hNotificationThreadInitialized);

  MSG msg;
  while (GetMessage(&msg, NULL, 0, 0))
    {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
}

/**
 * Message window procedure.
 */
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  LRESULT result;
  switch (message)
  {
    case WM_TIMER:
      {
        JNIEnv* env;
        g_javaVM->GetEnv(reinterpret_cast<void**>(&env), JNI_VERSION_1_4);
        if (env)
          {
            // Get acceleration and notify it to TG01AccelerationListener.
            TS_ACCELERATION acc;
            DWORD retcode = TSGetAcceleration(g_accelerationID, &acc);
            if (checkReturnCode(env, "TSGetAcceleration()", retcode))
              {
                env->CallStaticVoidMethod(g_TG01AccelerometerClass,
                                          g_TG01Accelerometer_notifyTG01AccelerationListener_mid,
                                          (jint) acc.x,
                                          (jint) acc.y,
                                          (jint) acc.z);
              }
          }
      }
      break;

    case TS_ACTION_EVENT:
      // TG01 action event recognized.
      {
        int eventType = LOWORD(lParam) & 0xffff;
        int eventParameter = HIWORD(lParam) & 0xffff;

        JNIEnv* env;
        g_javaVM->GetEnv(reinterpret_cast<void**>(&env), JNI_VERSION_1_4);
        if (env)
          {
            env->CallStaticVoidMethod(g_TG01AccelerometerClass,
                                           g_TG01Accelerometer_notifyTG01ActionListener_mid,
                                           eventType,
                                           eventParameter);
          }
      }
      break;

    case WM_CLOSE:
      DestroyWindow(hWnd);
      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      break;

    default:
      result = DefWindowProc(hWnd, message, wParam, lParam);
      break;
  }

  return result;
}
