/* com_mysaifu_mobile_device_tg01_TG01Accelerometer.cpp
   Copyright (C) 2009 freebeans

This file is part of Mysaifu JVM.

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

/**
 * Toshiba TG01 accelerometer API library.
 *
 * I wish to express my gratitude to the developer of these excellent information.
 * http://smartmobidevice.blogspot.com/2009/08/toshiba-tg01-and-accelerometer.html
 */

#include "stdafx.h"
#include "com_mysaifu_mobile_device_tg01_TG01Accelerometer.h"

/**
 * TG01 device detected.
 */
static BOOL g_bTG01Detected;

/**
 * TG01 sensor API struct.
 */
typedef struct tagTS_ACCELERATION
{
  SHORT x;
  SHORT y;
  SHORT z;
  FILETIME time;
} TS_ACCELERATION;


/**
 * TG01 Sensor APIs.
 */
typedef DWORD (*TSREGISTERACCELERATION)(HMODULE,UINT, PUINT);
typedef DWORD (*TSDEREGISTERACCELERATION)(UINT);
typedef DWORD (*TSGETACCELERATION)(UINT, TS_ACCELERATION* accel);

static TSREGISTERACCELERATION TSRegisterAcceleration;
static TSDEREGISTERACCELERATION TSDeregisterAcceleration;
static TSGETACCELERATION TSGetAcceleration;

/**
 * Static functions declarations.
 */
static void InitializeTG01Accelerometer();
static bool checkReturnCode(JNIEnv* env, const char* functionName, DWORD code);


/**
 * Initializer TG01 accelerometer.
 */
static void InitializeTG01Accelerometer()
{
  if (g_bTG01Detected)
  {
    return;
  }

  // Get function pointers from "axcon.dll".
  HMODULE hModule = LoadLibrary(_T("axcon"));
  if (hModule)
  {
    TSRegisterAcceleration   = (TSREGISTERACCELERATION)   GetProcAddress(hModule, _T("TSRegisterAcceleration"));
    TSDeregisterAcceleration = (TSDEREGISTERACCELERATION) GetProcAddress(hModule, _T("TSDeregisterAcceleration"));
    TSGetAcceleration        = (TSGETACCELERATION)        GetProcAddress(hModule, _T("TSGetAcceleration"));
    if (TSRegisterAcceleration && TSDeregisterAcceleration && TSGetAcceleration)
    {
      g_bTG01Detected = TRUE;
    }
  }
}

/**
 * Ensure that TG01 device detected.
 *
 * @param env pointer of JNIEnv
 * @return  true if OK. false means NG.
 */
static bool ensureTG01DeviceDetected(JNIEnv* env)
{
  if (! g_bTG01Detected)
  {
    InitializeTG01Accelerometer();
    if (! g_bTG01Detected)
    {
      env->ThrowNew(env->FindClass("java/io/IOException"), "TG01 acceleration sensor not found");
      return false;
    }
  }
  return true;
}

/**
 * Checks return value of TSXXXX() APIs.
 *
 * @param env JNIEnv.
 * @param functionName
 * @param code  The return value of TSRegisterAcceleration(), TSGetAcceleration(), and TSDeregisterAcceleration()
 * @return  true if OK.
 */
static bool checkReturnCode(JNIEnv* env, const char* functionName, DWORD code)
{
  if (code == 0)
  {
    return true;
  }

  // throw IOException.
  const char* format = "%s failed. return code=%d";
  char msg[128];
  sprintf(msg, format, functionName, code);
  env->ThrowNew(env->FindClass("java/io/IOException"), msg);
  return false;
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    nativeOpen
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_nativeOpen
  (JNIEnv *env, jobject , jint interval)
{
  if (! ensureTG01DeviceDetected(env))
  {
    return 0;
  }

  UINT id;
  DWORD retcode = TSRegisterAcceleration(GetModuleHandle(NULL), (UINT) interval, &id);
  if (! checkReturnCode(env, "TSRegisterAcceleration()", retcode))
  {
    return 0;
  }
  return (jint) id;
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    nativeClose
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_nativeClose
  (JNIEnv *env, jobject, jint id)
{
  if (! ensureTG01DeviceDetected(env))
  {
    return;
  }

  DWORD retcode = TSDeregisterAcceleration(id);
  checkReturnCode(env,"TSDeregisterAcceleration()", retcode);
}

/*
 * Class:     com_mysaifu_mobile_device_tg01_TG01Accelerometer
 * Method:    nativeGet
 * Signature: (I[S)V
 */
JNIEXPORT void JNICALL Java_com_mysaifu_mobile_device_tg01_TG01Accelerometer_nativeGet
  (JNIEnv *env, jobject, jint id, jshortArray data)
{
  if (! ensureTG01DeviceDetected(env))
  {
    return;
  }

  jshort* result = (jshort*) env->GetPrimitiveArrayCritical(data, 0);
  if (env->ExceptionCheck())
  {
    return;
  }
  TS_ACCELERATION acc;
  DWORD retcode = TSGetAcceleration(id, &acc);
  if (checkReturnCode(env, "TSGetAcceleration()", retcode))
  {
    result[0] = acc.x;
    result[1] = acc.y;
    result[2] = acc.z;
  }
  env->ReleasePrimitiveArrayCritical(data, result, 0);
}