/* TG01Accelerometer.java
   Copyright (C) 2009 freebeans

This file is part of Mysaifu JVM.

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package com.mysaifu.mobile.device.tg01;

import java.io.IOException;
import java.security.AccessController;
import java.security.PrivilegedAction;

/**
 * Toshiba TG01 accelerometer API library. I wish to express my gratitude to the
 * developer of these excellent information. <a href="http://smartmobidevice.blogspot.com/2009/08/toshiba-tg01-and-accelerometer.html"
 * >Toshiba TG01 and Accelerometer</a>
 * <p>
 * Usage:<br>
 * 1. Get the instance.
 * <code><pre>
 * TG01Accelerometer acc = TG01Accelerometer.getInstance();
 * </pre></code>
 * 
 * 2. Open acceleromter device.
 * <code><pre>
 * acc.open();
 * </code></pre>
 * 
 * 3. Get data from accelerometer
 * <code><pre>
 * short[] data = new short[3];
 * acc.get(data);
 * </pre></code>
 * 
 * 4. Close accelerometer device.
 * <code><pre>
 * acc.close();
 * </pre></code>
 */
public class TG01Accelerometer
{
  /**
   * Default interval.
   */
  public int DEFAULT_INTERVAL = 100;

  /**
   * Singleton.
   */
  private static TG01Accelerometer INSTANCE;

  /**
   * open flag.
   */
  private boolean opened;

  /**
   * sensor id.
   */
  private int id;

  private TG01Accelerometer()
  {
    // Load native library.
    if (System.getSecurityManager() != null)
      {
        AccessController.doPrivileged(new PrivilegedAction<Void>()
        {
          public Void run()
          {
            System.loadLibrary("devtg01");
            return null;
          }
        });
      }
    else
      {
        System.loadLibrary("devtg01");
      }
  }

  /**
   * Creates new instance.
   */
  public synchronized static TG01Accelerometer getInstance()
  {
    if (INSTANCE == null)
      {
        INSTANCE = new TG01Accelerometer();
      }
    return INSTANCE;
  }

  /**
   * open accelerometer (interval is 100ms)
   * 
   * @throws IOException
   */
  public void open() throws IOException
  {
    open(DEFAULT_INTERVAL);
  }

  /**
   * open accelerometer with specified interval.
   * 
   * @param interval accelerometer intarval.
   * @throws IOException error occurred.
   */
  public synchronized void open(int interval) throws IOException
  {
    this.id = nativeOpen(interval);
    this.opened = true;
  }

  private native int nativeOpen(int interval) throws IOException;

  /**
   * close accelerometer.
   * 
   * @throws IOException error occurred.
   */
  public synchronized void close() throws IOException
  {
    if (this.opened)
      {
        this.opened = false;
        nativeClose(this.id);
      }
  }

  private native void nativeClose(int id) throws IOException;

  /**
   * Get data from accelerometer.
   * 
   * @param data short array that receives acceleration data. data[0] is X axis
   *          value, data[1] is Y axis value, and data[2] is Z axis value. if
   *          data is null, new short array is allocated.
   * @return data or new allocated short array.
   * @throws IOException error occurred.
   */
  public synchronized short[] get(short[] data) throws IOException
  {
    if (! this.opened)
      {
        throw new IOException("Not opened.");
      }
    if (data == null)
      {
        data = new short[3];
      }
    nativeGet(this.id, data);
    return data;
  }

  private native void nativeGet(int id, short[] data) throws IOException;

  @Override
  protected void finalize() throws Exception
  {
    close();
  }
}
