/* class_loader.cpp
   Copyright (C) 2005 Free Software Foundation, Inc.

This file is part of Mysaifu JVM

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
*/


#include "StdAfx.h"
#include "ClassFile.h"
#include "class_loader.h"
#include "java_utf8.h"
#include "unzip/unzip.h"
#include "java_object.h"
#include "instruction.h"
#include "java_thread.h"
#include "jni_funcs.h"
#include "common_funcs.h"
#include "verifier.h"
#include "java_utf8.h"

/**
 * staticf[^
 */
struct static_data_status
{
	/**
	 * staticf[^
	 */
  jobject static_data;

	/**
	 * Xe[^X
	 */
  int status;

	/**
	 * ̃XbhID
	 */
  DWORD initializing_thread_id;
};

/**
 * NX[_̗vf
 */
struct class_loader_element
{
	/**
	 * Jn[_ꍇ trueB
	 */
  bool initiating_loader;

	/**
	 * [hClassFile\̂̎Q
	 */
  ClassFile *cfile;

	/**
	 * java.lang.Class̃CX^XiKvɂȂ_Őj
	 */
  jclass class_object;
};

/**
 * ClassCircularityError `FbN邽߂̍\
 */
struct loading_class
{
  loading_class *next;
  const java_utf8 *name;
};

/**
 * NX[_
 */
struct class_loader
{
	/**
	 * NX[_̃bN
	 */
  CRITICAL_SECTION *lock;

	/**
	 * NX[_̃CX^XB
	 * u[gXgbv[_̏ꍇNULLB
	 */
  jobject instance;

	/**
	 * Lȗvf
	 */
  unsigned int elements_count;

	/**
	 * vf̃TCYimۍς݂̃TCYj
	 */
  unsigned int elements_size;

	/**
	 * vf
	 */
  class_loader_element **elements;

	/**
	 * ̃NX[_[hNXɑ΂
	 * xt@CKv
	 */
  bool verify;

	/**
	 * ClassCircularityError`FbNp
	 */
  loading_class *circularity_info;

	/**
	 * ClassFile\̂蓖Ă邽߂̃̈
	 */
  PermanentHeap *heap;
};

// [h
struct loading_constraints
{
  loading_constraints *next;	// PXgƂĊǗ
  const java_utf8 *class_name;	// NX
  class_loader *loader1;	// P
  class_loader *loader2;	// Q
};

// [he[uiKȑfj
#define LOADING_CONSTRAINTS_TABLE_SIZE	1021

// [he[u
static loading_constraints
  *g_loading_constraints_table[LOADING_CONSTRAINTS_TABLE_SIZE];

// u[gXgbvENX[_gpNXpX
struct classpath_element
{
  char *path;			// pXBfBNg\ꍇAK "\" ŏIĂKv
  bool jar;			// jart@C̏ꍇtrue
  ZIPFILE *zipfile;		// jart@C𓀗p
};

/**
 * u[gXgbvNXpX
 */
static classpath_element **g_bootstrap_classpath;

/**
 * u[gXgbvNXpXij
 */
static _TCHAR *g_bootstrap_classpath_string;

/**
 * u[gXgbvNXpX̐
 */
static int g_bootstrap_classpath_count;

/**
 * VXeNXpX
 */
static _TCHAR *g_system_classpath;

/**
 * u[gXgbvENX[_
 */
class_loader *g_bootstrap_class_loader = NULL;

/**
 * ̑̃NX[_
 */
static unsigned int g_class_loaders_count;
static class_loader **g_class_loaders;

// static ֐̒`
/**
 * w肳ꂽNX[_ɂAw肳ꂽOClassFile\̂ǂݍށB
 */
static ClassFile *load_ClassFile (frame * frm, class_loader * loader,
				  const java_utf8 * class_name);

/**
 * u[gXgbvNX[_ɂAw肳ꂽÕNX[h
 *  class_name ͕KuC^[vĂKvB
 */
inline static ClassFile *
load_ClassFile (frame * current_frame, const java_utf8 * class_name)
{
  return load_ClassFile (current_frame, get_bootstrap_class_loader (),
			 class_name);
}
static ClassFile *load_ClassFile_from_file (frame * current_frame,
					    const java_utf8 * class_name,
					    class_loader * loader);
static ClassFile *load_ClassFile_from_jar (frame * current_frame,
					   const java_utf8 * class_name,
					   class_loader * loader,
					   classpath_element * elem,
					   const char *classfilename);


/**
 * w肳ꂽNXANX[_ɓo^
 */
static bool link_ClassFile (frame * current_frame, class_loader * loader,
			    ClassFile * cfile);

/**
 * w肳ꂽ[h̑Ó`FbNB
 */
static bool check_loading_constraints (frame * frm,
				       const java_utf8 * class_name,
				       class_loader * loader,
				       ClassFile * cfile);

/**
 * [hǉB
 * [h̒ǉɎsꍇi[hς݃NX̊ԂłłɃ[h񂪔jĂꍇj
 * falseԂB
 */
static bool add_loading_constraints (frame * frm,
				     const java_utf8 * class_name,
				     class_loader * loader1,
				     class_loader * loader2);

// ̑static֐
static jclass create_class_object (frame * current_frame, jobject static_data,
				   jobject pd);
static void add_static_data (jobject data);
static classpath_element *create_classpath_element (char *path, bool jar);
bool ensure_static_data_initialized (frame * current_frame,
				     ClassFile * cfile);


/**
 * w肳ꂽNXɑΉclass_loader_element\̂
 */
static class_loader_element *find_class_loader_element (class_loader * l,
							const java_utf8 *
							name);

/**
 * w肳ꂽClassFile\̂class_loaderɒǉ
 */
static void add_ClassFile (class_loader * l, ClassFile * cfile);

/**
 * class_loadermۂ
 */
static class_loader *allocate_class_loader (jobject cl);

ClassFile *g_primitive_classes[9];
jclass g_primitive_class_objects[9];

/**
 * Verify mode.
 */
static VerifyMode g_verify_mode;

/**
 * Annotation mode.
 */
static AnnotationMode g_annotation_mode;

/**
 * NX[_gpq[ṽftHgTCY
 */
#define DEFAULT_CLASS_LOADER_HEAP_SIZE (32*1024)

/**
 * NX[_ݒ
 */
void
init_class_loader_settings (VerifyMode verify_mode,
			    AnnotationMode annotation_mode)
{
  g_verify_mode = verify_mode;
  g_annotation_mode = annotation_mode;

  // u[gXgbvNX[_
  g_bootstrap_class_loader = allocate_class_loader (NULL);
}

/**
 * w肳ꂽÕv~eBuNX[h
 */
ClassFile *
load_primitive_ClassFile (frame * frm, const java_utf8 * class_name)
{
  class_name = intern_utf8 (class_name);

  // NeBJZNVɓ
  EnterCriticalSection (g_bootstrap_class_loader->lock);

  ClassFile *cfile = NULL;
  int i;
  for (i = 0; i < 9; ++i)
    {
      cfile = g_primitive_classes[i];
      if (!cfile)
	{
	  break;
	}
      else if (cfile->this_class_name == class_name)
	{
	  break;
	}
    }
  if (cfile == NULL && i < 9)
    {
      cfile = create_primitive_ClassFile (class_name);
      // NXt@C
      prepare_ClassFile (frm, cfile, NULL,	// X[p[NXȂ
			 NULL);	// C^tF[XȂ
      g_primitive_classes[i] = cfile;
      jobject static_data = alloc_static_data (cfile);
      if (static_data == NULL)
	{
	  throw_OutOfMemoryError (frm);
	}
      else
	{
	  add_static_data (static_data);
	}
    }
  // NeBJZNV甲
  LeaveCriticalSection (g_bootstrap_class_loader->lock);

  return cfile;
}

/**
 * v~eBuNX ClassIuWFNgԂB
 */
jclass
find_primitive_class_object (frame * current_frame, jobject static_data)
{
  ClassFile *cfile = get_ClassFile (static_data);
  jclass class_object = NULL;
  // NeBJZNVɓ
  EnterCriticalSection (g_bootstrap_class_loader->lock);
  for (int i = 0; i < 9; ++i)
    {
      if (cfile == g_primitive_classes[i])
	{
	  class_object = g_primitive_class_objects[i];
	  if (class_object == NULL)
	    {
	      class_object =
		create_class_object (current_frame, static_data, NULL);
	      g_primitive_class_objects[i] = class_object;
	    }
	}
    }
  // NeBJZNV甲
  LeaveCriticalSection (g_bootstrap_class_loader->lock);
  return class_object;
}

/**
 * w肳ꂽNX[_ɂăNX̒`ǂݍ݁AClassFile\̂ւ̃|C^Ԃ
 * [hɎsꍇ́ANULLԂ
 *  class_name ͕KuC^[vĂKvB
 *
 */
static ClassFile *
load_ClassFile (frame * current_frame, class_loader * l,
		const java_utf8 * class_name)
{
  // zNXw肳Ăꍇɂ́Az̍\vfɂĒ`[_܂
  // zvf̃ANZXtO
  u2 array_element_access_flags = ACC_PUBLIC;
  if (*class_name == '[')
    {
      ClassFile *acfile = NULL;
      if (class_name[1] == '[' || class_name[1] == 'L')
	{
	  // \vf[h
	  if (class_name[1] == '[')
	    {
	      const char *tmp = &class_name[1];
	      acfile = load_ClassFile (current_frame, l, intern_utf8 (tmp));
	      if (is_public (acfile->access_flags))
		{
		  array_element_access_flags = ACC_PUBLIC;
		}
	      else if (is_protected (acfile->access_flags))
		{
		  array_element_access_flags = ACC_PROTECTED;
		}
	      else if (is_private (acfile->access_flags))
		{
		  array_element_access_flags = ACC_PRIVATE;
		}
	    }
	  else if (class_name[1] == 'L')
	    {
	      int len = strlen (&class_name[2]);
	      char *tmp = (char *) malloc (len);
	      if (tmp == NULL)
		{
		  fatal_error (FATAL_ERROR_NO_MEMORY);
		}
	      strncpy (tmp, &class_name[2], len - 1);
	      tmp[len - 1] = '\0';
	      acfile = load_ClassFile (current_frame, l, intern_utf8 (tmp));
	      free (tmp);
	    }
	  if (acfile == NULL)
	    {
	      return NULL;
	    }
	  // \vf̒`NX[_AzNX̒`[_Ƃ
	  l = acfile->defining_loader;
	}
      else
	{
	  l = get_bootstrap_class_loader ();
	}
    }

  // ̊֐Ăяo_ŁuC^[vĂ邱ƂOɂĂ
  // ȉ̃̕Rg͂ƁȂO񂪐`FbNł
  // assert(class_name == intern_utf8(class_name));

  // [hς݂ł΁A̍\̂Ԃ
  // NeBJZNVɓAs
  ClassFile *cfile = NULL;
  EnterCriticalSection (l->lock);
  class_loader_element *elem = find_class_loader_element (l, class_name);
  if (elem)
    {
      cfile = elem->cfile;
      if (!elem->initiating_loader)
	{
	  elem->initiating_loader = true;
	  // [h`FbN
	  if (!check_loading_constraints
	      (current_frame, class_name, l, cfile))
	    {
	      // Jn[_̋L^P
	      elem->initiating_loader = false;
	      LeaveCriticalSection (l->lock);
	      return NULL;
	    }
	}
    }

  if (cfile != NULL)
    {
      // łɃ[hς݂̏ꍇ́Ãt@CԂ
      // NeBJZNV𔲂
      LeaveCriticalSection (l->lock);
      return cfile;
    }

  // ȍ~̓[hς݂łȂꍇɎs

  if (*class_name == '[')
    {
      // zNXVM
      cfile =
	create_array_ClassFile (l, class_name, array_element_access_flags);
      if (!link_ClassFile (current_frame, l, cfile))
	{
	  free_ClassFile (cfile);
	}

      // NeBJZNV𔲂
      LeaveCriticalSection (l->lock);

      return cfile;
    }

  // u[gXgbvNX[_ȊOw肳Ăꍇ́AloadClass()\bh
  // Ăяo
  if (l != g_bootstrap_class_loader)
    {
      // NeBJZNV𔲂
      LeaveCriticalSection (l->lock);

      // convert '/' to '.'.
      char *tmp = (char *) malloc (strlen (class_name) + 1);
      strcpy (tmp, class_name);
      char *work = tmp;
      do
	{
	  if (*work == '/')
	    {
	      *work = '.';
	    }
	}
      while (*work++);
      cfile = call_loadClass (current_frame, l->instance, tmp);
      free (tmp);
      if (cfile)
	{
	  elem = find_class_loader_element (l, class_name);
	  if (!elem)
	    {
	      // Jn[_ƂċL^邽߁Aœo^
	      add_ClassFile (l, cfile);
	      elem = find_class_loader_element (l, class_name);
	    }
	  assert (elem);
	  if (!elem->initiating_loader)
	    {
	      elem->initiating_loader = true;
	      // [h`FbN
	      if (!check_loading_constraints
		  (current_frame, class_name, l, cfile))
		{
		  // Jn[_̋L^P
		  elem->initiating_loader = false;
		  LeaveCriticalSection (l->lock);
		  return NULL;
		}
	    }
	}
    }

  // ́Au[gXgbvNX[_ɂ郍[h

  bool error = false;
  if (cfile == NULL)
    {
      // ʏ̃NX́At@C烍[h
      cfile = load_ClassFile_from_file (current_frame, class_name, l);
      if (cfile == NULL || exception_occurred (current_frame))
	{
	  error = true;
	}
      else
	{
	  elem = find_class_loader_element (l, class_name);
	  assert (elem);
	  if (!elem->initiating_loader)
	    {
	      elem->initiating_loader = true;
	      // [h`FbN
	      if (!check_loading_constraints
		  (current_frame, class_name, l, cfile))
		{
		  // Jn[_̋L^P
		  elem->initiating_loader = false;
		  LeaveCriticalSection (l->lock);
		  return NULL;
		}
	    }
	}
    }

  // NeBJZNVI
  LeaveCriticalSection (l->lock);

  if (cfile == NULL)
    {
      DBG (_T ("Bootstrap class loader could not find class="));
      DBG_UTF8 (class_name);
      DBG (_T ("\n"));
    }
  return (!error) ? cfile : NULL;
}

/*	
 * t@CNX[h
 */
static ClassFile *
load_ClassFile_from_file (frame * current_frame, const java_utf8 * class_name,
			  class_loader * loader)
{
  const char *ext = ".class";
  unsigned int cfnamelen = strlen (class_name) + strlen (ext) + 1;	// ".class" + '\0'
  char *classfilename = (char *) malloc (cfnamelen);
  if (!classfilename)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  _snprintf (classfilename, cfnamelen, "%s%s", class_name, ext);

  ClassFile *cfile = NULL;
  for (int i = 0; i < g_bootstrap_classpath_count; ++i)
    {
      classpath_element *path = g_bootstrap_classpath[i];
      if (path->jar)
	{
	  // Jart@Cǂݍ
	  cfile = load_ClassFile_from_jar (current_frame,
					   class_name,
					   loader, path, classfilename);
	}
      else
	{
	  const int pathlen =
	    strlen (path->path) + strlen (classfilename) + 1;
	  char *abspath = (char *) malloc (pathlen);
	  if (abspath == NULL)
	    {
	      fatal_error (FATAL_ERROR_NO_MEMORY);
	    }
	  _snprintf (abspath, pathlen, "%s%s", path->path, classfilename);

	  // t@CI[vAeǂݍ
	  FILE *file = fopen (abspath, "rb");
	  free (abspath);

	  if (file != NULL)
	    {
	      fseek (file, 0L, SEEK_END);
	      size_t file_size = (size_t) ftell (file);
	      fseek (file, 0L, SEEK_SET);
	      char *classbytes = (char *) malloc (file_size);
	      if (classbytes == NULL)
		{
		  fatal_error (FATAL_ERROR_NO_MEMORY);
		}
	      if (classbytes != NULL)
		{
		  // NXt@Cǂݍ݁Ap[X
		  fread (classbytes, file_size, file_size, file);
		  fclose (file);

		  cfile = define_ClassFile (current_frame,
					    class_name,
					    loader, classbytes, 0, file_size);
		}
	    }
	}
      if (cfile)
	{
	  break;
	}
    }
  free (classfilename);
  return cfile;
}

/**
 * w肳ꂽQ̃NXt@CÃpbP[WɑĂ邩ԂB
 */
bool
is_same_package (class_loader * loader, ClassFile * c1, ClassFile * c2)
{
  static u1 slash = (u1) '/';

  // ݂̎ł́AP̃NX[_͖
  const java_utf8 *name1 = c1->this_class_name;
  const java_utf8 *name2 = c2->this_class_name;

  char *pos1 = strrchr (name1, slash);
  char *pos2 = strrchr (name2, slash);
  if (pos1)
    {
      // XbV܂łr
      return ((pos1 - name1) == (pos2 - name2))
	&& (strncmp (name1, name2, pos1 - name1) == 0);
    }
  else
    {
      // ǂpbP[W̏ꍇ̂true
      return !pos2;
    }
}

/**
 * w肳ꂽjart@CAΉClassFileo
 */
static ClassFile *
load_ClassFile_from_jar (frame * current_frame, const java_utf8 * class_name,
			 class_loader * loader, classpath_element * elem,
			 const char *classfilename)
{
  // jart@CI[v
  if (elem->zipfile == NULL)
    {
      const char *jarfile = elem->path;
      int len = strlen (jarfile);
      _TCHAR *filename = (_TCHAR *) malloc (sizeof (_TCHAR) * (len + 1));
      if (filename == NULL)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      convert_to_TCHAR (jarfile, filename, len + 1);
      elem->zipfile = unzip_open (filename);
      free (filename);
    }
  if (elem->zipfile == NULL)
    {
      return NULL;
    }

  ClassFile *cfile = NULL;
  if (unzip_seek (elem->zipfile, classfilename))
    {
      DWORD uncompressed_size = unzip_get_uncompressed_size (elem->zipfile);
      char *classbytes = (char *) malloc (uncompressed_size);
      if (classbytes == NULL)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      if (unzip_read (elem->zipfile, classbytes, uncompressed_size))
	{
	  cfile = define_ClassFile (current_frame,
				    class_name,
				    loader, classbytes, 0, uncompressed_size);
	}
      free (classbytes);
    }

  return cfile;

}

/**
 * w肳ꂽClassFilestaticf[^
 * ݂ĂȂꍇNULLԂ
 *
 * @param	cfile	ΏۂƂȂClassFile\̂ւ̃|C^
 * @return	Ήstaticf[^B݂ȂꍇNULL
 */
jobject
get_static_data (ClassFile * cfile)
{
  if (cfile->static_data_stat != NULL)
    {
      return cfile->static_data_stat->static_data;
    }
  else
    {
      assert (false);
      return NULL;
    }
}

/**
 * w肳ꂽstaticf[^ɑΉ java.lang.Class̃CX^X𐶐
 */
static jclass
create_class_object (frame * current_frame, jobject static_data, jobject pd)
{
  // OꎞIɃNAĂ
  jthrowable t = current_frame->body->current_exception;
  jthrowable global_throwable = NULL;
  if (t)
    {
      // Kx[WRNgȂ悤ɃO[oQƂƂĕێĂ
      global_throwable = (jthrowable) new_global_reference (t);
    }
  current_frame->body->current_exception = NULL;

  // java/lang/Class [h
  jclass class_object = NULL;
  ClassFile *cfile = load_ClassFile (current_frame, CLASS_CLASS_NAME);
  ClassFile *target_cfile = get_ClassFile (static_data);

  jclass tmp = (jclass) alloc_object (cfile, current_frame);
  if (tmp != NULL)
    {
      DUP_STACK (current_frame);
      PUSH_OBJECT (current_frame, static_data);	// vmdata

      if (target_cfile->this_class_name == JAVA_LANG_VMCLASS_CLASS_NAME
	  || target_cfile->this_class_name == CLASS_CLASS_NAME)
	{
	  // java.lang.ClassIuWFNĝ̂\ Class ͓ʈB
	  // RXgN^Ăяoł͂ȂAtB[h vmdata ɒڒlݒ肷
	  // (ȂƖɍċAĂ܂j
	  jfieldID fid = get_field_id (cfile,
				       intern_utf8 ("vmdata"),
				       intern_utf8 ("Ljava/lang/Object;"));
	  assert (fid);
	  put_field (current_frame, cfile, fid);
	  class_object = tmp;

	}
      else
	{
	  // Class(Object vmdata, ProtectionDomain pd) Ăяo
	  jmethodID mid = get_method_id (cfile,
					 INIT_METHOD_NAME,
					 JAVA_LANG_CLASS_CONSTRUCTOR_DESCRIPTOR);
	  if (mid != NULL)
	    {
	      method_info *minfo = get_method_info (cfile, mid);
	      if (minfo != NULL)
		{
		  // ProtectionDomainɒǉ
		  PUSH_OBJECT (current_frame, pd);
		  // Class̃RXgN^ŎsĂfbhbN͔ȂH
		  invoke_method (current_frame, cfile, minfo);
		  if (!exception_occurred (current_frame))
		    {
		      class_object = tmp;
		    }
		}
	    }
	  else
	    {
	      fatal_error (_T ("Could not found Class constructor"));
	    }
	}
    }
  if (class_object != NULL)
    {
      // Kx[WRNgȂ悤ɁAO[oQƂƂēo^
      class_object = (jclass) new_global_reference (class_object);
      // X^bNpop
      jobject tmp;
      POP_OBJECT (current_frame, tmp);
    }

  class_loader_element *elem =
    find_class_loader_element (get_defining_loader (target_cfile),
			       target_cfile->this_class_name);
  if (elem)
    {
      assert (elem->cfile->defining_loader == target_cfile->defining_loader);
      elem->class_object = class_object;
    }
  else
    {
      // v~eBuNX̏ꍇ elem ݂͑Ȃ
    }

  // O߂
  if (t)
    {
      current_frame->body->current_exception = t;
      delete_global_reference (global_throwable);
    }

  return class_object;
}

/**
 * w肳ꂽstaticf[^ɑΉjava.lang.Class̃CX^XB
 * NX[_ɑ݂ĂȂꍇAVKɃCX^X𐶐
 *
 * @param	cfile	ΏۂƂȂClassFile\̂ւ̃|C^
 * @return	ΉClassIuWFNgB݂ȂꍇNULL
 */
jclass
find_class_object (frame * current_frame, jobject static_data, jobject pd)
{
  ClassFile *cfile = get_ClassFile (static_data);
  if (is_primitive_class_name (cfile->this_class_name))
    {
      // v~eBuNX̊֐Ń[h邱Ƃ͂łȂ
      return NULL;
    }

  class_loader *l = cfile->defining_loader;

  // NeBJZNVɓ
  EnterCriticalSection (l->lock);

  jclass class_object = NULL;
  class_loader_element *elem =
    find_class_loader_element (l, cfile->this_class_name);
  assert (elem != NULL);

  class_object = elem->class_object;
  if (!class_object)
    {
      // VKɃNXIuWFNg쐬
      class_object = create_class_object (current_frame, static_data, pd);
    }

  // NeBJZNV甲
  LeaveCriticalSection (l->lock);

  return class_object;
}

/**
 * w肳ꂽ OɑΉ郍[hς Class IuWFNgԂB
 * [hĂȂꍇNULLԂB
 */
jclass
find_loaded_class_object (class_loader * l, frame * current_frame,
			  const java_utf8 * name)
{
  // NeBJZNVɓ
  // ĂяoƂ ClassLoader.findLoadedClass()synchronizedȂ̂ŁȀ͕sv
  // (Bug #19458)
  // EnterCriticalSection (l->lock);

  jclass class_object = NULL;
  class_loader_element *elem = find_class_loader_element (l, name);
  if (elem)
    {
      class_object = elem->class_object;
    }

  // NeBJZNV甲
  // LeaveCriticalSection (l->lock);

  return class_object;
}

/**
 * w肳ꂽNXstaticf[^o^
 */
static void
add_static_data (jobject ref)
{
  ClassFile *cfile = get_ClassFile (ref);
  class_loader *l = get_defining_loader (cfile);
  static_data_status *status =
    (static_data_status *) malloc (sizeof (static_data_status));
  if (status == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  status->static_data = ref;
  status->status = STATIC_DATA_ALLOCATED;
  // ClassFile\̂ɐݒ肷
  cfile->static_data_stat = status;
}

/**
 * staticf[^̏ԂԂB
 */
int
get_static_data_status (jobject static_data)
{
  ClassFile *cfile = get_ClassFile (static_data);
  static_data_status *s = cfile->static_data_stat;
  if (s != NULL)
    {
      return s->status;
    }
  return STATIC_DATA_ERROR;
}

/**
 * staticf[^̏Ԃݒ肷B
 */
void
set_static_data_status (jobject static_data, int stat)
{
  ClassFile *cfile = get_ClassFile (static_data);
  cfile->static_data_stat->status = stat;
  if (stat == STATIC_DATA_INITIALIZING)
    {
      cfile->static_data_stat->initializing_thread_id = GetCurrentThreadId ();
    }
  else
    {
      cfile->static_data_stat->initializing_thread_id = NULL;
    }
}

/**
 * ̃Xbhstaticf[^ł邩ԂB
 */
bool
is_another_thread_initializing (jobject static_data)
{
  bool result = false;
  ClassFile *cfile = get_ClassFile (static_data);
  static_data_status *status = cfile->static_data_stat;
  if (status->status == STATIC_DATA_INITIALIZING
      && status->initializing_thread_id != GetCurrentThreadId ())
    {
      result = true;
    }
  return result;
}

/**
 * w肳ꂽ class_loader \̂ɑΉ ClassLoader ̃CX^XԂ
 */
jobject
get_ClassLoader (class_loader * loader)
{
  return loader->instance;
}

static class_loader *
allocate_class_loader (jobject cl)
{
  class_loader *loader = (class_loader *) calloc (1, sizeof (class_loader));
  if (loader == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  loader->instance = cl;
  loader->lock = (CRITICAL_SECTION *) malloc (sizeof (CRITICAL_SECTION));
  if (loader->lock == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  InitializeCriticalSection (loader->lock);

  if (g_verify_mode != VERIFY_MODE_NONE && loader->instance)
    {
      // xt@C[hݒ肷
      loader->verify = true;
    }

  // q[v̈mۂ
  loader->heap = new PermanentHeap (DEFAULT_CLASS_LOADER_HEAP_SIZE);
  if (!loader->heap)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  loader->heap->setMinimumAllocationSize (sizeof (double));	// K؂H
  return loader;
}

/**
 * w肳ꂽClassLoaderIuWFNgɑΉclass_loader\̂Ԃ
 */
class_loader *
get_class_loader (jobject cl)
{
  if (cl == NULL)
    {
      // u[gXgbvNX[_
      return g_bootstrap_class_loader;
    }
  else
    {
      class_loader *loader = NULL;
      // NeBJZNVɓ
      EnterCriticalSection (&g_global_critical_section);
      for (unsigned int i = 0; i < g_class_loaders_count; ++i)
	{
	  class_loader *tmp = g_class_loaders[i];
	  if (tmp->instance == cl)
	    {
	      loader = tmp;
	      break;
	    }
	}
      if (loader == NULL)
	{
	  // O[ozɒǉ
	  loader = allocate_class_loader (cl);
	  if (loader == NULL)
	    {
	      fatal_error (FATAL_ERROR_NO_MEMORY);
	    }
	  g_class_loaders_count++;
	  g_class_loaders =
	    (class_loader **) realloc (g_class_loaders,
				       sizeof (class_loader *) *
				       g_class_loaders_count);
	  g_class_loaders[g_class_loaders_count - 1] = loader;
	}

      LeaveCriticalSection (&g_global_critical_section);
      return loader;
    }
}

/**
 * w肳ꂽoCgz߂AClassFile\̂𐶐B
 *  loader Ŏw肳ꂽNX[_A`NX[_ƂċL^B
 */
ClassFile *
define_ClassFile (frame * current_frame,
		  const java_utf8 * name,
		  class_loader * loader,
		  const char *buff, int offset, int length)
{
  EnterCriticalSection (loader->lock);

  // ClassCircularityError̃`FbNs
  bool circularity_error = false;
  loading_class *c = loader->circularity_info;
  while (c)
    {
      if (c->name == name)
	{
	  // ClassCircularityError
	  circularity_error = true;
	  throw_exception (current_frame, "java/lang/ClassCircularityError",
			   name);
	  break;
	}
      c = c->next;
    }

  ClassFile *cfile = NULL;
  if (!circularity_error)
    {
      // NXp[X
      cfile = parse_ClassFile (current_frame, loader, buff + offset, length);

      // NXANX[_ɓo^
      if (cfile)
	{
	  if (cfile->this_class_name != name)
	    {
	      // OႤ
	      char *msg = (char *) malloc (128);
	      if (!msg)
		{
		  fatal_error (FATAL_ERROR_NO_MEMORY);
		}
	      _snprintf (msg,
			 128 - 1,
			 "%s (wrong name %s)", name, cfile->this_class_name);
	      msg[127] = '\0';
	      throw_exception (current_frame,
			       "java/lang/NoClassDefFoundError", msg);
	      free (msg);
	      free_ClassFile (cfile);
	      cfile = NULL;
	    }
	  else
	    {
	      // ClassCircularityError`FbNł悤ɁANX[_ɈꎞI
	      // o^Ă
	      loading_class lc;
	      lc.name = name;
	      lc.next = loader->circularity_info;
	      loader->circularity_info = &lc;

	      // Linkings
	      if (!link_ClassFile (current_frame, loader, cfile))
		{
		  free_ClassFile (cfile);
		}

	      // Ɠo^̂ŁAcircularity_info Xg͂
	      loading_class *c = loader->circularity_info;
	      assert (c == &lc);
	      loader->circularity_info = lc.next;
	    }
	}
    }
  LeaveCriticalSection (loader->lock);
  return cfile;
}

/**
 * NXt@CANX[_ɓo^
 * Linkingɑ鏈
 */
static bool
link_ClassFile (frame * current_frame, class_loader * loader,
		ClassFile * cfile)
{
  if (!cfile)
    {
      return true;
    }
  // X[p[NXċAIɃ[hĂ
  ClassFile *superclass = NULL;
  if (cfile->super_class_name != NULL)
    {
      superclass =
	load_ClassFile (current_frame, cfile->defining_loader,
			cfile->super_class_name);
      if (superclass == NULL)
	{
	  return false;
	}
      // ToDo:X[p[NX̃\bhI[o[ChĂꍇA
      // і߂l̃[hL^
    }

  // X[p[C^tF[XċAIɃ[hĂ
  ClassFile **interfaces =
    (ClassFile **) malloc (sizeof (ClassFile *) * cfile->interfaces_count);
  if (interfaces == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  for (unsigned int i = 0; i < cfile->interfaces_count; ++i)
    {
      const java_utf8 *iname =
	get_CONSTANT_Class_info (cfile, cfile->interfaces[i]);
      ClassFile *ifile =
	load_ClassFile (current_frame, cfile->defining_loader, iname);
      if (ifile == NULL)
	{
	  free (interfaces);
	  return false;
	}
      interfaces[i] = ifile;
    }

  // xt@C̃pXQs
  if (loader->verify)
    {
      if (!verify_pass2 (current_frame, cfile, superclass, interfaces))
	{
	  // xt@Cs
	  return false;
	}
    }

  // Preparations
  bool prepared =
    prepare_ClassFile (current_frame, cfile, superclass, interfaces);
  free (interfaces);

  if (!prepared)
    {
      return false;
    }
  // static_data 蓖Ă
  jobject result = alloc_static_data (cfile);
  if (result == NULL)
    {
      throw_OutOfMemoryError (current_frame);
      return false;
    }
  add_static_data (result);

  // NX[_̔zClassFileǉ
  add_ClassFile (loader, cfile);

  return true;
}

/**
 * u[gNXpX
 * xݒ肷ƁAύXłȂBij
 */
void
init_boot_classpath (const _TCHAR * classpath)
{
  if (g_bootstrap_classpath == NULL)
    {
      // O[oϐɃRs[Ă
      g_bootstrap_classpath_string =
	(_TCHAR *) malloc (sizeof (_TCHAR) * (_tcslen (classpath) + 1));
      if (g_bootstrap_classpath_string == NULL)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      _tcscpy (g_bootstrap_classpath_string, classpath);

      // ';' ŋ؂ꂽpXǂݎ
      char utf8[256];
      _TCHAR tmp[256];
      int tmppos = 0;
      tmp[0] = _T ('\0');
      while (*classpath)
	{
	  while (*classpath && *classpath != _T (';')
		 && tmppos < (sizeof (tmp) / sizeof (*tmp)) - 1)
	    {
	      tmp[tmppos++] = *classpath++;
	    }
	  tmp[tmppos] = _T ('\0');
	  tmppos = 0;
	  if (*classpath == _T (';'))
	    {
	      classpath++;
	    }

	  // ݂pXǂ𒲂ׂ
	  WIN32_FIND_DATA find_data;
	  HANDLE hFind = FindFirstFile (tmp, &find_data);
	  if (hFind != INVALID_HANDLE_VALUE)
	    {
	      int len = _tcslen (tmp);
	      convert_to_utf8 (tmp, -1, utf8,
			       sizeof (utf8) / sizeof (utf8[0]));
	      FindClose (hFind);
	      classpath_element *elem = NULL;
	      if (find_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
		{
		  elem = create_classpath_element (utf8, false);
		}
	      else
		{
		  elem = create_classpath_element (utf8, true);
		}
	      g_bootstrap_classpath_count++;
	      g_bootstrap_classpath
		= (classpath_element **) realloc (g_bootstrap_classpath,
						  sizeof (classpath_element) *
						  g_bootstrap_classpath_count);
	      g_bootstrap_classpath[g_bootstrap_classpath_count - 1] = elem;
	    }
	}
    }
}

/**
 * u[gNXpXԂ
 */
const _TCHAR *
get_boot_classpath ()
{
  return g_bootstrap_classpath_string;
}

/**
 * VXeNXpX
 * xݒ肷ƁAύXłȂBij
 */
void
init_system_classpath (const _TCHAR * classpath)
{
  if (g_system_classpath == NULL)
    {
      int len = _tcslen (classpath);
      g_system_classpath = (_TCHAR *) malloc (sizeof (_TCHAR) * (len + 1));
      if (g_system_classpath == NULL)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      _tcscpy (g_system_classpath, classpath);
    }
}

/**
 * VXeNXpXԂ
 */
const _TCHAR *
get_system_classpath ()
{
  return g_system_classpath;
}

static classpath_element *
create_classpath_element (char *path, bool jar)
{
  classpath_element *elem =
    (classpath_element *) malloc (sizeof (classpath_element));
  if (elem == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  elem->jar = jar;
  elem->zipfile = NULL;
  if (jar)
    {
      elem->path = (char *) malloc (strlen (path) + 1);
      if (elem->path == NULL)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      strcpy (elem->path, path);
    }
  else
    {
      // fBNg̏ꍇɂ́AK '\' ŏI悤ɂ
      int len = strlen (path);
      if (path[len - 1] != '\\')
	{
	  elem->path = (char *) malloc (len + 2);
	  if (elem->path == NULL)
	    {
	      fatal_error (FATAL_ERROR_NO_MEMORY);
	    }
	  strcpy (elem->path, path);
	  elem->path[len] = '\\';
	  elem->path[len + 1] = '\0';
	}
      else
	{
	  elem->path = (char *) malloc (len + 1);
	  if (elem->path)
	    {
	      fatal_error (FATAL_ERROR_NO_MEMORY);
	    }
	  strcpy (elem->path, path);
	}
    }
  return elem;
}

/**
 * w肳ꂽNX[_pāAw肳ꂽÕNX[h
 * w肳ꂽNXɂĂ̏istatic final tB[h̏<clinit>\bh̎sj
 * sB
 */
ClassFile *
find_ClassFile (frame * frm, class_loader * loader,
		const java_utf8 * class_name)
{
  // OꎞޔĂ
  jthrowable t = frm->body->current_exception;
  jthrowable global_throwable = NULL;
  if (t)
    {
      // Kx[WRNgȂ悤ɃO[oQƂƂĕێĂ
      global_throwable = (jthrowable) new_global_reference (t);
    }
  frm->body->current_exception = NULL;

  // ClassFile\̂[h
  // ̊֐Ăяo_ intern_utf8()Ă邱ƂOɂ
  // ClassFile* cfile = load_ClassFile(frm, loader, intern_utf8(class_name));
  ClassFile *cfile = load_ClassFile (frm, loader, class_name);

  if (cfile == NULL)
    {
      if (!exception_occurred (frm))
	{
	  // Oݒ肳ĂȂꍇ́ANoClassDefFoundErrorthrow
	  throw_exception (frm, "java/lang/NoClassDefFoundError", class_name);
	}
      return NULL;
    }
  if (!ensure_static_data_initialized (frm, cfile))
    {
      // ɎsBMuAbv
      if (!exception_occurred (frm))
	{
	  // Oݒ肳ĂȂꍇ́ANoClassDefFoundErrorthrow
	  throw_exception (frm, "java/lang/NoClassDefFoundError", class_name);
	}
      return NULL;
    }
  // O𕜌
  if (t)
    {
      frm->body->current_exception = t;
      delete_global_reference (global_throwable);
    }
  return cfile;
}

/**
 * w肳ꂽClassFile\̂ɑΉstatictB[hĂ邱Ƃ
 * ۏႷ
 */
bool
ensure_static_data_initialized (frame * current_frame, ClassFile * cfile)
{
  if (cfile == NULL)
    {
      return false;
    }

  jobject static_instance = get_static_data (cfile);
  if (static_instance == NULL)
    {
      assert (false);
      return false;
    }

  // łɏĂꍇɂ true Ԃ
  int status = get_static_data_status (static_instance);
  if (status == STATIC_DATA_INITIALIZED)
    {
      return true;
    }

  // 1. bN擾
  monitor_enter (static_instance);

  // 2. Xbh̊wait()
  while (is_another_thread_initializing (static_instance))
    {
      monitor_wait (static_instance, 0);
    }

  // 3. JgEXbhłꍇAċAIȏł
  //    bNāA^[
  if (!is_another_thread_initializing (static_instance)
      && status == STATIC_DATA_INITIALIZING)
    {
      monitor_exit (static_instance);
      return true;
    }

  // 4. łɏꂽCX^X݂Ăꍇɂ͂̂܂܃^[
  status = get_static_data_status (static_instance);
  if (status == STATIC_DATA_INITIALIZED)
    {
      monitor_exit (static_instance);
      return true;
    }

  // 5. G[Ԃ̏ꍇɂ̓^[
  else if (status == STATIC_DATA_ERROR)
    {
      monitor_exit (static_instance);
      return false;
    }

  // 6. JgXbh̃NXł邱ƂL^AbN
  set_static_data_status (static_instance, STATIC_DATA_INITIALIZING);
  monitor_exit (static_instance);

  // łɗOĂꍇ́A̗OێĂB
  jthrowable old_exception = exception_occurred (current_frame);

  // xt@C̃pXR(oCgR[hExt@Cjs
  if (needs_verify (get_defining_loader (cfile)))
    {
      if (!verify_pass3 (current_frame, cfile))
	{
	  // xt@C̃pXRs
	  // ̃NX̏ɎsƂo^
	  set_static_data_status (static_instance, STATIC_DATA_ERROR);
	  monitor_enter (static_instance);
	  monitor_notify (static_instance, true);	// notifyAll
	  monitor_exit (static_instance);
	  return false;
	}
    }

  // 7. X[p[NX^X[p[C^tF[X̏ċAIɎ{
  ClassFile *superClassFile = NULL;
  if (cfile->super_class_name != NULL)
    {
      superClassFile = find_ClassFile (current_frame,
				       get_defining_loader (cfile),
				       cfile->super_class_name);
      if (superClassFile == NULL)
	{
	  // s
	  // ̃NX̏ɎsƂo^
	  set_static_data_status (static_instance, STATIC_DATA_ERROR);
	  monitor_enter (static_instance);
	  monitor_notify (static_instance, true);	// notifyAll
	  monitor_exit (static_instance);
          if (!exception_occurred (current_frame))
	    {
              // Oݒ肳ĂȂꍇ́ANoClassDefFoundErrorthrow
	      throw_exception (current_frame, "java/lang/NoClassDefFoundError",
	        		     cfile->super_class_name);
            }
	  return false;
	}
    }

  // X[p[C^tF[X̏
  for (u2 i = 0; i < cfile->interfaces_count; ++i)
    {
      const java_utf8 *superinterface_name =
	get_CONSTANT_Class_info (cfile, cfile->interfaces[i]);
      ClassFile *interfaceClassFile = find_ClassFile (current_frame,
						      get_defining_loader
						      (cfile),
						      superinterface_name);
      if (interfaceClassFile == NULL)
	{
	  // s\
	  // ̃NX̏ɎsƂo^
	  set_static_data_status (static_instance, STATIC_DATA_ERROR);
	  monitor_enter (static_instance);
	  monitor_notify (static_instance, true);	// notifyAll
	  monitor_exit (static_instance);
	  // Oݒ肵falseԂ
	  throw_exception (current_frame, "java/lang/NoClassDefFoundError",
			   superinterface_name);
	  return false;
	}
    }

  // 8. NX̕ϐqƐÓIqAC^tF[X̃tB[hq
  // ԂɎs
  // finalw肳ꂽstatictB[h
  for (int i = 0; i < cfile->fields_count; ++i)
    {
      field_info *finfo = &cfile->fields[i];
      if (is_static (finfo->access_flags) && is_final (finfo->access_flags))
	{
	  // static finaltB[h
	  if (finfo->constantvalue_index)
	    {
	      jfieldID fid =
		get_static_field_id (cfile, finfo->name, finfo->descriptor);
	      cp_info *cinfo =
		cfile->constant_pool[finfo->constantvalue_index];
	      switch (cfile->constant_pool_tags[finfo->constantvalue_index])
		{
		case CONSTANT_Integer:
		  {
		    PUSH_INT (current_frame, (__int32) cinfo);
		    put_static_field (current_frame, static_instance, fid);
		    break;
		  }
		case CONSTANT_Long:
		  {
		    // ARMvZbTł32rbgPʂɃANZX邽߁AACgᔽɂ͂ȂȂ
		    __int64 value =
		      *(__int64 *) & cfile->constant_pool[finfo->
							  constantvalue_index];
		    PUSH_LONG (current_frame, value);
		    put_static_field (current_frame, static_instance, fid);
		    break;
		  }
		case CONSTANT_Float:
		  {
		    PUSH_FLOAT (current_frame, *(float *) &cinfo);
		    put_static_field (current_frame, static_instance, fid);
		    break;
		  }

		case CONSTANT_Double:
		  {
		    // ARMvZbTł32rbgPʂɃANZX邽߁AACgᔽɂ͂ȂȂ
		    double value =
		      *(double *) &cfile->constant_pool[finfo->
							constantvalue_index];
		    PUSH_DOUBLE (current_frame, value);
		    put_static_field (current_frame, static_instance, fid);
		    break;
		  }

		case CONSTANT_String:
		  {
		    jobject string_reference;
		    if (IS_SPECIAL_POINTER (cinfo))
		      {
			// jstringĂ
			string_reference =
			  (jstring) MAKE_NORMAL_POINTER (cinfo);
		      }
		    else
		      {
			// java_utf8* Ă
			java_utf8 *utf = (java_utf8 *) cinfo;
			string_reference = intern_string (utf, current_frame);
			if (exception_occurred (current_frame))
			  {
			    return false;
			  }
			// lLbV
			cfile->constant_pool[finfo->constantvalue_index]
			  =
			  (cp_info *) MAKE_SPECIAL_POINTER (string_reference);
		      }
		    PUSH_OBJECT (current_frame, string_reference);
		    put_static_field (current_frame, static_instance, fid);
		    break;
		  }
		}
	    }
	  // finalłȂꍇ0܂nullɏ
	  // ̏̓mێɍŝŁAŎ{Kv͂Ȃ
	}
    }

  // \bh<clinit>sB
  method_info *clinit_method_info =
    find_declaring_method_info (cfile, CLINIT_METHOD_NAME,
				VOID_NOARG_METHOD_DESCRIPTOR);
  if (clinit_method_info != NULL)
    {
      // jobject static_data = get_static_data (cfile);
      invoke_method (current_frame, cfile, clinit_method_info);
    }

  // 10. Os
  // <clinit>sɃG[ꍇ
  if (old_exception != exception_occurred (current_frame))
    {
      set_static_data_status (static_instance, STATIC_DATA_ERROR);
      monitor_enter (static_instance);
      monitor_notify (static_instance, true);	// notifyAll
      monitor_exit (static_instance);

      // ExceptionInInitializerErrorthrow
      // AOutOfMemoryErroȑꍇɂ͐VCX^X𐶐ɁAɃ^[
      ClassFile *current_exception_class =
	get_ClassFile (current_frame->body->current_exception);
      if (!strcmp
	  (current_exception_class->this_class_name,
	   "java/lang/OutOfMemoryError"))
	{
	  return false;
	}

      // ONA
      jthrowable caused_exception = current_frame->body->current_exception;
      current_frame->body->current_exception = NULL;
      ClassFile *eiie_file = find_ClassFile (current_frame,
					     intern_utf8
					     ("java/lang/ExceptionInInitializerError"));
      if (eiie_file)
	{
	  jthrowable obj =
	    (jthrowable) alloc_object (eiie_file, current_frame);
	  if (obj == NULL)
	    {
	      return false;
	    }
	  PUSH_OBJECT (current_frame, obj);
	  PUSH_OBJECT (current_frame, caused_exception);
	  jmethodID mid = get_method_id (eiie_file,
					 INIT_METHOD_NAME,
					 intern_utf8
					 ("(Ljava/lang/Throwable;)V"));
	  method_info *minfo = get_method_info (eiie_file, mid);
	  invoke_method (current_frame, eiie_file, minfo);

	  current_frame->body->current_exception = obj;
	}

      return false;
    }
  else
    {
      // 9. ɏIƂmFAo^
      set_static_data_status (static_instance, STATIC_DATA_INITIALIZED);
      // 11.NXbNAXbhɒʒm
      monitor_enter (static_instance);
      monitor_notify (static_instance, true);	// notifyAll
      monitor_exit (static_instance);
    }
  return true;
}

/**
 * w肳ꂽNXɑΉclass_loader_element\̂
 * 񕪌s߁Aȉ̑OB
 *
 * 1. w肳ꂽO intern_utf8()"C^["ĂB
 * 2. elements̓e name ̃AhXɃ\[gĂ
 *
 * Ȃꍇ NULL ԂB
 */
static class_loader_element *
find_class_loader_element (class_loader * l, const java_utf8 * name)
{
assert(strchr(name, '.') == NULL);

  class_loader_element **elements = l->elements;
  const unsigned int count = l->elements_count;

  if (count)
    {
      // NX̃|C^Ń\[gĂ邽߁A񕪒T
      int start = 0;
      int end = count - 1;
      while (start <= end)
	{
	  int index = start + (end - start) / 2;
	  class_loader_element *elem = elements[index];
	  const java_utf8 *cname = elem->cfile->this_class_name;
	  if (cname > name)
	    {
	      end = index - 1;
	    }
	  else if (cname < name)
	    {
	      start = index + 1;
	    }
	  else
	    {
	      return elem;
	    }
	}
    }
  return NULL;
}

/**
 * w肳ꂽClassFileAclass_loader_element̔zɒǉ
 * ClassFile.this_class_name ̃|C^lɃ\[gĊi[邽
 * NX́uC^[vĂȂ΂ȂȂBA
 */
static void
add_ClassFile (class_loader * l, ClassFile * cfile)
{
  const unsigned int count = l->elements_count;
  class_loader_element **elements = l->elements;

  // vf̑}ʒu
  int start = 0;
  int end = count ? count - 1 : 0;
  if (count)
    {
      const java_utf8 *name = cfile->this_class_name;
      while (start <= end)
	{
	  int ind = start + (end - start) / 2;
	  class_loader_element *elem = elements[ind];
	  const java_utf8 *cname = elem->cfile->this_class_name;
	  if (cname > name)
	    {
	      end = ind - 1;
	    }
	  else if (cname < name)
	    {
	      start = ind + 1;
	    }
	  else
	    {
	      // łɑ݂Ă̂Ń^[
	      return;
	    }
	}
    }

  // }ʒuvZ
  int index = start + (end - start) / 2;

  // KvɉĔzg
  l->elements_count++;
  if (l->elements_count >= l->elements_size)
    {
      //l->elements_size += 64;	// 64₷
      l->elements_size += 256;	// 256₷
      l->elements = (class_loader_element **) realloc (l->elements,
						       sizeof
						       (class_loader_element
							*) *
						       l->elements_size);
      if (!l->elements)
	{
	  // mێs
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
    }

  // }ʒuP炷
  memmove (&l->elements[index + 1],
	   &l->elements[index],
	   sizeof (class_loader_element *) * (l->elements_count - index - 1));

  // vfi[
  class_loader_element *elem =
    (class_loader_element *) calloc (sizeof (class_loader_element), 1);
  if (!elem)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  elem->cfile = cfile;
  elem->class_object = NULL;

  l->elements[index] = elem;
}

/**
 * w肳ꂽNX[_[hNXɑ΂
 * xt@CKvǂԂB
 */
bool
needs_verify (class_loader * loader)
{
  return loader->verify;
}

/**
 * [he[ũCfbNXlvZ
 */
static inline int
get_loading_constraints_table_index (const java_utf8 * class_name)
{
  // AhX̉ʂRrbg͏ɂOî͂jȂ̂ŁARrbgVtgĂ
  // ]߂
  return ((int) class_name >> 3) % LOADING_CONSTRAINTS_TABLE_SIZE;
}

/**
 * [hǉB
 * [h̒ǉɎsꍇi[hς݃NX̊ԂłłɃ[h񂪔jĂꍇj
 * falseԂB
 */
static bool
add_loading_constraints (frame * frm, const java_utf8 * class_name,
			 class_loader * loader1, class_loader * loader2)
{
  EnterCriticalSection (&g_global_critical_section);

  int index = get_loading_constraints_table_index (class_name);
  loading_constraints *c = g_loading_constraints_table[index];
  bool exists = false;
  while (c)
    {
      if ((c->class_name == class_name)
	  && (c->loader1 == loader1 || c->loader2 == loader1)
	  && (c->loader1 == loader2 || c->loader2 == loader2))
	{
	  // Y̐񂪂łɑ݂
	  exists = true;
	  break;
	}
      c = c->next;
    }

  bool failed = false;
  ClassFile *cfile = NULL;
  // [h񂪂łɔjĂȂׂ
  class_loader_element *e1 = find_class_loader_element (loader1, class_name);
  class_loader_element *e2 = find_class_loader_element (loader2, class_name);
  if (e1 && e2
      && e1->initiating_loader && e2->initiating_loader
      && e1->cfile != e2->cfile)
    {
      // jĂ
      failed = true;
    }

  if (!failed && !exists)
    {
      // Vo^
      c = (loading_constraints *) calloc (sizeof (loading_constraints), 1);
      if (!c)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      c->class_name = class_name;
      c->loader1 = loader1;
      c->loader2 = loader2;
      c->next = g_loading_constraints_table[index];
      g_loading_constraints_table[index] = c;

      DBG (_T ("Loading constraints added class name="));
      DBG_UTF8 (class_name);
      DBG (_T ("\n"));
    }

  LeaveCriticalSection (&g_global_critical_section);

  if (failed)
    {
      // O throw 
      char *msg = (char *) malloc (128);
      if (!msg)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      _snprintf (msg,
		 128 - 1, "Class %s violates loader constraints", class_name);
      msg[127] = '\0';
      throw_exception (frm, "java/lang/LinkageError", msg);
      free (msg);
    }

  return !failed;
}

/**
 * w肳ꂽ[h̑Ó`FbNB
 */
static bool
check_loading_constraints (frame * frm, const java_utf8 * class_name,
			   class_loader * loader, ClassFile * cfile)
{
  // e[ũCfbNX߂
  int index = get_loading_constraints_table_index (class_name);
  loading_constraints *c = g_loading_constraints_table[index];
  bool failed = false;
  while (c)
    {
      if ((c->class_name == class_name)
	  && (c->loader1 == loader || c->loader2 == loader))
	{
	  class_loader *opposite =
	    (c->loader1 == loader) ? c->loader2 : c->loader1;
	  class_loader_element *elem =
	    find_class_loader_element (opposite, class_name);
	  if (elem && elem->initiating_loader && elem->cfile != cfile)
	    {
	      // `FbNs
	      failed = true;
	    }
	  break;
	}
      c = c->next;
    }
  if (failed)
    {
      char *msg = (char *) malloc (128);
      if (!msg)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      _snprintf (msg,
		 128 - 1, "Class %s violates loader constraints", class_name);
      msg[127] = '\0';
      throw_exception (frm, "java/lang/LinkageError", msg);
      free (msg);

    }
  return !failed;
}

/**
 * w肳ꂽtB[hfBXNv^Ɋւ郍[hǉ
 */
bool
add_loading_constraints_of_field (frame * current_frame,
				  ClassFile * declaring_ClassFile,
				  const java_utf8 * descriptor)
{
  class_loader *loader1 =
    get_defining_loader (current_frame->current_class_file);
  class_loader *loader2 = get_defining_loader (declaring_ClassFile);
  if (loader1 != loader2)
    {
      // tB[ȟ^ɂă[hǉ
      if (descriptor[0] == '[' || descriptor[0] == 'L')
	{
	  if (descriptor[0] == 'L')
	    {
	      descriptor =
		intern_utf8 (&descriptor[1], strlen (descriptor) - 2);
	    }
	  if (!add_loading_constraints (current_frame,
					descriptor, loader1, loader2))
	    {
	      // ̒ǉɎs
	      return false;
	    }
	}
    }
  return true;
}

/**
 * w肳ꂽ\bhfBXNv^Ɋւ郍[hǉ
 */
bool
add_loading_constraints_of_method (frame * current_frame,
				   class_loader * loader1,
				   class_loader * loader2,
				   const java_utf8 * descriptor)
{
  if (loader1 != loader2)
    {
      descriptor++;
      // \bḧƖ߂lɊւă[hǉ
      char c;
      while ((c = *descriptor++))
	{
	  switch (c)
	    {
	    case ')':
	      break;

	    case '[':
	      {
		const char *start = descriptor - 1;
		while (*descriptor == '[')
		  {
		    descriptor++;
		  }
		if (*descriptor == 'L')
		  {
		    // vf͎Qƌ^
		    while (*descriptor != ';')
		      {
			// NX̏I܂ŃXLbv
			descriptor++;
		      }
		    // ';' ܂߂Kv
		    descriptor++;

		  }
		else
		  {
		    // vf̓v~eBu^
		    descriptor++;
		  }
		const java_utf8 *param_name =
		  intern_utf8 (start, descriptor - start);
		if (!add_loading_constraints
		    (current_frame, param_name, loader1, loader2))
		  {
		    // ̒ǉɎs
		    return false;
		  }
	      }
	      break;

	    case 'L':
	      {
		const char *start = descriptor;
		while (*descriptor != ';')
		  {
		    descriptor++;
		  }
		const java_utf8 *param_name =
		  intern_utf8 (start, descriptor - start);
		if (!add_loading_constraints
		    (current_frame, param_name, loader1, loader2))
		  {
		    // ̒ǉɎs
		    return false;
		  }
	      }
	      break;

	    default:
	      break;
	    }
	}
    }
  return true;
}

/**
 * w肳ꂽ\bhfBXNv^Ɋւ郍[hǉ
 */
bool
add_loading_constraints_of_method (frame * current_frame,
				   ClassFile * declaring_ClassFile,
				   const java_utf8 * descriptor)
{
  if (!declaring_ClassFile)
    {
      // ǉ̕KvȂ
      return true;
    }
  class_loader *loader1 =
    get_defining_loader (current_frame->current_class_file);
  class_loader *loader2 = get_defining_loader (declaring_ClassFile);

  return add_loading_constraints_of_method (current_frame, loader1, loader2,
					    descriptor);
}

PermanentHeap *
get_heap_of_class_loader (class_loader * loader)
{
  return loader->heap;
}

/**
 * Returns annotation load level.
 */
bool
needs_visible_annotations (class_loader * loader)
{
  return g_annotation_mode != ANNOTATION_DISCARD;
}

bool
needs_all_annotations (class_loader * loader)
{
  return g_annotation_mode == ANNOTATION_ALL;
}
