/* FreeTypeFontMetrics.java -- Information about about a fonts display characteristics
   Copyright (C) 2008  Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package gnu.java.awt.peer.wce.font;

import gnu.java.awt.peer.wce.font.FreeTypeFace;

import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.font.FontRenderContext;
import java.awt.font.LineMetrics;
import java.awt.geom.Rectangle2D;
import java.text.CharacterIterator;

public class FreeTypeFontMetrics extends FontMetrics
{
  private final FreeTypeFace face;
  private final float[] fontMetrics;
  
  public FreeTypeFontMetrics(Font font, FreeTypeFace face)
  {
    super(font);
    this.face = face;
    this.fontMetrics = face.getFontMetrics(font.getSize2D());
  }

  /**
   * Returns the leading, or spacing between lines, for this font.
   *
   * @return The font leading.
   */
  public int getLeading()
  {
    return (int)this.fontMetrics[FreeTypeFace.LEADING_INDEX];
  }

  /**
   * Returns the ascent of the font, which is the distance from the base
   * to the top of the majority of characters in the set.  Some characters
   * can exceed this value however.
   *
   * @return The font ascent.
   */
  public int getAscent()
  {
    return (int)this.fontMetrics[FreeTypeFace.ASCENT_INDEX];
  }

  /**
   * Returns the descent of the font, which is the distance from the base
   * to the bottom of the majority of characters in the set.  Some characters
   * can exceed this value however.
   *
   * @return The font descent.
   */
  public int getDescent()
  {
    return (int)this.fontMetrics[FreeTypeFace.DESCENT_INDEX];
  }

  /**
   * Returns the height of a line in this font.  This will be the sum
   * of the leading, the ascent, and the descent.
   *
   * @return The height of the font.
   */
  public int getHeight()
  {
    return getAscent() + getDescent() + getLeading();
  }

  /**
   * Returns the maximum ascent value.  This is the maximum distance any
   * character in the font rised above the baseline.
   *
   * @return The maximum ascent for this font.
   */
  public int getMaxAscent()
  {
    return (int)this.fontMetrics[FreeTypeFace.MAXASCENT_INDEX];
  }

  /**
   * Returns the maximum descent value.  This is the maximum distance any
   * character in the font extends below the baseline.
   *
   * @return The maximum descent for this font.
   */
  public int getMaxDescent()
  {
    return (int)this.fontMetrics[FreeTypeFace.MAXDESCENT_INDEX];
  }

  /**
   * Returns the width of the widest character in the font.
   *
   * @return The width of the widest character in the font.
   */
  public int getMaxAdvance()
  {
    return (int) this.fontMetrics[FreeTypeFace.MAXADVANCE_INDEX];
  }

  /**
   * Returns the width of the specified character.
   *
   * @param ch The character to return the width of.
   *
   * @return The width of the specified character.
   */
  public int charWidth(char ch)
  {
    char[] tmp = new char[1];
    tmp[0] = ch;
    return charsWidth(tmp, 0, 1);
  }

  /**
   * Returns the total width of the specified character array.
   *
   * @param buf The character array containing the data.
   * @param offset The offset into the array to start calculating from.
   * @param len The total number of bytes to process.
   *
   * @return The width of the requested characters.
   */
  public int charsWidth(char[] buf, int offset, int len)
  {
    return (int) face.getCharsWidth(getFont().getSize2D(), buf, offset, len);
  }

  /**
   * Returns the total width of the specified byte array.
   *
   * @param buf The byte array containing the data.
   * @param offset The offset into the array to start calculating from.
   * @param len The total number of bytes to process.
   *
   * @return The width of the requested characters.
   */
  public int bytesWidth(byte[] buf, int offset, int len)
  {
    char[] chars = new char[len];
    for (int i = 0; i < len; i++)
    {
      chars[i] = (char) buf[offset + i];
    }

    return (int) charsWidth(chars, 0, chars.length);
  }

  /**
   * Returns the widths of the first 256 characters in the font.
   *
   * @return The widths of the first 256 characters in the font.
   */
  public int[] getWidths()
  {
    int[] result = new int[256];
    char[] chars = new char[256];
    for (char i = 0; i < 256; i++)
    {
      chars[i] = (char)i;
      result[i] = (int) charsWidth(chars, 0, 256);
    }
    return result;
  }

  /**
   * Returns a string representation of this object.
   *
   * @return A string representation of this object.
   */
  public String toString()
  {
    return (this.getClass() + "[font=" + font + ",ascent=" + getAscent()
            + ",descent=" + getDescent() + ",height=" + getHeight() + "]");
  }

  /**
   * Returns if the font has uniform line metrics.
   * @see Font#hasUniformLineMetrics()
   */
  public boolean hasUniformLineMetrics()
  {
    return font.hasUniformLineMetrics();
  }
}
