/* java_object.cpp
   Copyright (C) 2005 Free Software Foundation, Inc.

This file is part of Mysaifu JVM

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
*/

#include "StdAfx.h"
#include <tchar.h>
#include "java_object.h"
#include "class_loader.h"
#include "ClassFile.h"
#include "java_utf8.h"
#include "instruction.h"
#include "java_thread.h"
#include "common_funcs.h"
#include "console.h"

#ifdef DEBUG

/**
 * q[v̓vo͂t@C
 */
//      #define HEAP_STATISTICS_FILE    _T("\\heap_statistics.txt")

#ifdef HEAP_STATISTICS_FILE
#include <stdio.h>
FILE *g_heap_statistics_file;
#endif

#endif

/**
 * X[q[vgpꍇɂ͂̃}NɂOȊO̒l`
 */
#define SMALL_HEAP_ENABLED	1

/**
 * X[q[vubNTCY
 */
#define	SMALL_BLOCK_SIZE	64

/**
 * X[q[vubNJEg
 */
#define	SMALL_BLOCK_COUNT	256

/**
 * X[q[vGg
 */
struct small_heap_element
{
	/**
	 * 󂫗̈tO
	 */
  __int32 flags[SMALL_BLOCK_COUNT];

	/**
	 * f[^
	 */
  char data[SMALL_BLOCK_SIZE * SMALL_BLOCK_COUNT * 32];
};

/**
 * X[q[v̈
 */
static small_heap_element *g_heap_elements[4096];
static char g_heap_flags[4096];

/**
 * t@CiCU\
 */
struct finalizer
{
	/**
	 * sԂ𐧌䂷邽߂̃nh
	 */
  HANDLE hWaiting;

	/**
	 * ^[QbgIuWFNg̐
	 */
  unsigned int count;

	/**
	 * z̑傫
	 */
  unsigned int array_size;

	/**
	 * finalize()ĂяoΏۂƂȂIuWFNg̔z
	 */
  jobject *target_objects;

	/**
	 * t@CiCUAChԂǂ̃tO
	 */
  bool idling;

	/**
	 * ̍\̂̃oǂݏۂɎgp
	 * CRITICAL_SECTION
	 */
  LPCRITICAL_SECTION lock;
};

/**
 * t@CiCU
 */
static finalizer *g_finalizer;

/**
 * finalize()IuWFNg𐶐B
 * GCĂяo邽тɃNA
 */
static int g_finalize_objects_count;

/**
 * g_finalize_objects_count ̒萔𒴂ƁAIGC
 */
#define FINALIZE_OBJECTS_COUNT_THRESHOLD	512

/**
 * Kx[WRN^gptÕtH[}bg͈ȉ̂Ƃ
 *
 * rbg0-7 : ݖgpB͈ȉ̗prŎgp\B
 *			   AhXbNJEgBJEg1ȏ̏ꍇAKx[WRN^̓IuWFNg
 *            AhXړłȂ
 * rbg8   : GC}[NBmarktF[YŃZbgAsweeptF[YŃNAB
 * rbg9   : t@CiCYB\}[NBIuWFNgɃZbgAfinalize()sɃNA
 * rbg10  : ̃IuWFNgNXIuWFNgistatictB[h̒lێĂIuWFNgj
 *             ̏ꍇɃZbg
 * rbg11  : gpBVersion 0.2.3܂ł͈ȉ̈ӖŎgpĂB
 *			   IuWFNg̎ʂBӖ͈ȉ̂Ƃ
 *               0 z܂1v~eBuzB
 *               1 IuWFNgz܂͑zBievfɂ̓IuWFNg̎QƂj
 * rbg12-13 : gpBVersion 0.2.3܂ł͈ȉ̈ӖŎgpĂB
 *				zvf̃TCYBzłȂꍇ͈ӖȂ
 *               00 - 1oCg
 *               01 - 2oCg
 *               10 - 4oCg
 *               11 - 8oCg
 * rbg14:   ut@gv}[NBBł͂ȂAsBł͂ȂIuWFNg
 *@@@@@@}[NBmarktF[YŃZbgAsweeptF[YŃNAB
 * rbg15-16 : QƃIuWFNg̎ʂ\
 *               00 - QƃIuWFNgł͂Ȃ
 *               01 - SoftReference
 *               10 - WeakReference
 *               11 - PhantomReference
 * rbg17-31 : \rbg
 */

/**
 * Kx[WRN^}[N
 */
#define	GC_MARKED		(1L << 8)

/**
 * t@CiCUB\ȃIuWFNg
 */
#define	FINALIZE_MARK	(1L << 9)

/**
 * staticf[^
 */
#define STATIC_DATA	(1L << 10)

/**
 * ut@gv}[N
 */
#define REFERENT_MARK	(1 << 14)

/**
 * SoftReference/WeakReference/PhantomReferencetO
 */
#define SOFT_REFERENCE_FLAG		(1 << 15)
#define WEAK_REFERENCE_FLAG		(2 << 15)
#define PHANTOM_REFERENCE_FLAG	(3 << 15)

#define GET_FIELD(current_frame, address, descriptor, pop_stack) \
{\
	switch (descriptor[0]) { \
	case 'Z': \
	case 'B': \
		{\
			__int8 value = *(__int8*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_INT(current_frame, value); \
		} \
		break; \
	\
	case 'C': \
		{ \
			unsigned __int16 value = *(unsigned __int16*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_INT(current_frame, value); \
		} \
		break; \
	\
	case 'S': \
		{ \
			__int16 value = *(__int16*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_INT(current_frame, value); \
		} \
		break; \
	\
	case 'I': \
		{ \
			__int32 value = *(__int32*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_INT(current_frame, value); \
		} \
		break; \
	\
	case 'J': \
		{ \
			__int64 value = *(__int64*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_LONG(current_frame, value); \
		} \
		break; \
	\
	case 'F': \
		{ \
			float value = *(float*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_FLOAT(current_frame, value); \
		} \
		break; \
	\
	case 'D': \
		{ \
			double value = *(double*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_DOUBLE(current_frame, value); \
		} \
		break; \
	\
	case 'L': \
	case '[': \
		{ \
			jobject value = *(jobject*) address; \
			if (pop_stack) { \
				POP_DISCARD(current_frame);\
			}\
			PUSH_OBJECT(current_frame, value); \
		} \
		break; \
	\
	default: \
		assert(false); \
	} \
}

#define PUT_FIELD(current_frame, address, descriptor) \
{\
	switch (descriptor[0]) { \
	case 'Z': \
	case 'B': \
		{\
			__int32 value; \
			POP_INT(current_frame, value); \
			*(__int8*) address = (__int8) value; \
		} \
		break; \
	\
	case 'C': \
		{ \
			__int32 value; \
			POP_INT(current_frame, value); \
			*(unsigned __int16*) address = (unsigned __int16) value; \
		} \
		break; \
	\
	case 'S': \
		{ \
			__int32 value; \
			POP_INT(current_frame, value); \
			*(__int16*) address = (__int16) value; \
		} \
		break; \
	\
	case 'I': \
		{ \
			__int32 value; \
			POP_INT(current_frame, value); \
			*(__int32*) address = value; \
		} \
		break; \
	\
	case 'J': \
		{ \
			__int64 value; \
			POP_LONG(current_frame, value); \
			*(__int64*) address = value; \
		} \
		break; \
	\
	case 'F': \
		{ \
			float value; \
			POP_FLOAT(current_frame, value); \
			*(float*) address = value; \
		} \
		break; \
	\
	case 'D': \
		{ \
			double value; \
			POP_DOUBLE(current_frame, value); \
			*(double*) address = value; \
		} \
		break; \
	\
	case 'L': \
	case '[': \
		{ \
			jobject value; \
			POP_OBJECT(current_frame, value); \
			*(jobject*) address = value; \
		} \
		break; \
	\
	default: \
		assert(false); \
	} \
}

/**
 * -Xloggc:filenameIvVŎw肳ꂽt@C
 */
static _TCHAR *g_loggc_filename;

/**
 * -verbose:gc w肳Ă邩
 */
static boolean g_verbose_gc;

/**
 * ClassFile̔z
 */
static ClassFile **g_class_files;

/**
 * NXIuWFNg̃X^bN
 */
static frame *g_static_data_stack;

#define STATIC_DATA_STACK_SIZE	4096

/**
 * u[gvq[v̈ւ̃|C^
 */
static java_object *g_heap_root = NULL;

/**
 * őq[vTCY(oCgj
 */
static unsigned int g_max_heap_size = 2 * 1024 * 1024;

/**
 * ݊mۍς݃q[vTCY
 */
static unsigned int g_allocated_size;

/**
 * Kx[WRN^stO
 */
static bool g_gc_running = false;

/**
 * GCjava.lang.ref.Reference IuWFNgi[z̗vf
 */
static int g_references_count;

/**
 * GCjava.lang.ref.ReferenceIuWFNgi[z
 */
static jobject *g_references;

/**
 * java.lang.ref.Reference.lock IuWFNg
 * GC͂̃IuWFNg̃j^擾ĂB
 */
static jobject g_Reference_lock;

// --- staticϐ̒` ܂ ---


// --- static֐̐錾  ---
static inline bool is_non_primitive_array (java_object * obj);

static void gc_mark ();
static void gc_mark (frame * sframe);
static void gc_mark (jobject obj, int marktype);
static void gc_start_finalizer ();
static void gc_enqueue ();
static void gc_prepare_finalize ();

#ifndef USE_JAVA_HEAP
static void gc_sweep ();
#endif

static void gc_wakeup_finalizer (finalizer * f);
static bool is_idling (finalizer * f);

static int get_reference_type_flag (java_object * obj);

//--- static֐̐錾@܂

#ifdef USE_JAVA_HEAP

/**
 * `FbNpf[^
 */
#define CHECK_DATA     ((unsigned char) 0xEF)

class JavaHeap
{
private:
	/**
	 * rbg}XN
	 */
  static const __int32 BIT_MASK[32];

	/**
	 * IuWFNg̃ACg
	 */
  static const int OBJECT_ALIGN = 8;

	/**
	 * rbgtOz1vf̃rbg
	 */
  static const int NUM_BITS = 32;

	/**
	 * 蓖ĒAhXrbg
	 */
  __int32 *allocatedBits;

	/**
	 * q[v̊Jnʒu
	 */
  char *heapBase;

	/**
	 * CommitAhX̊Jnʒu
	 */
  char *startOfUncommitted;

	/**
	 * q[v~bg
	 */
  char *heapLimit;

	/**
	 * 󂫗̈̃NXg
	 */
  java_object *freeObjects;

	/**
	 * R~bg
	 */
  char *commitNextBlock ();

	/**
	 * w肳ꂽAhXɑΉrbgtOz̃CfbNX𓾂
	 */
  inline int getFlagIndex (java_object * addr)
  {
    assert (((unsigned int) addr % OBJECT_ALIGN) == 0);
    assert ((char *) addr >= this->heapBase
	    && (char *) addr < this->heapLimit);
    return ((unsigned int) addr -
	    (unsigned int) this->heapBase) / (OBJECT_ALIGN * NUM_BITS);
  }

	/**
	 * w肳ꂽAhXɑΉrbgtOz̃rbg}XN𓾂
	 */
  inline __int32 getFlagMask (java_object * addr)
  {
    assert (((unsigned int) addr % OBJECT_ALIGN) == 0);
    assert ((char *) addr >= this->heapBase
	    && (char *) addr < this->heapLimit);

    return
      BIT_MASK[(((unsigned int) addr -
		 (unsigned int) this->heapBase) / 8) % NUM_BITS];
  }

	/**
	 * w肳ꂽAhXʒuɃIuWFNg݂邩𔻒肷
	 * mark()tF[YŎgp
	 */
  inline bool isObject (java_object * addr)
  {
    assert (((unsigned int) addr % OBJECT_ALIGN) == 0);
    assert ((char *) addr >= this->heapBase
	    && (char *) addr < this->heapLimit);

    return (((unsigned int) addr % OBJECT_ALIGN) == 0)
      && this->allocatedBits[getFlagIndex (addr)] & getFlagMask (addr);
  }

	/**
	 * w肳ꂽAhXʒu allocatedBits ɋL^
	 * allocate()gp
	 */
  inline void setAllocatedBit (java_object * addr)
  {
    assert (((unsigned int) addr % OBJECT_ALIGN) == 0);
    assert ((char *) addr >= this->heapBase
	    && (char *) addr < this->heapLimit);

    this->allocatedBits[getFlagIndex (addr)] |= getFlagMask (addr);
  }

	/**
	 * w肳ꂽAhXʒuɑΉ allocatedBits NA
	 * sweep()Ɏgp
	 */
  inline void clearAllocatedBit (java_object * addr)
  {
    assert (((unsigned int) addr % OBJECT_ALIGN) == 0);
    assert ((char *) addr >= this->heapBase
	    && (char *) addr < this->heapLimit);

    this->allocatedBits[getFlagIndex (addr)] &= ~getFlagMask (addr);
  }

	/**
	 * `FbNf[^
	 */
  inline void writeCheckData (java_object * addr)
  {
    *(char *) ((char *) addr + addr->size - 1) = (char) CHECK_DATA;
  }

public:
	/**
	 * JavaHeap쐬
	 *
	 * @param	őq[vTCY
	 */
  JavaHeap (size_t maxSize);

  ~JavaHeap ();

	/**
	 * w肳ꂽTCỸJavaq[v犄蓖Ă
	 */
  java_object *allocate (size_t size);

	/**
	 * sweepsB
	 */
  void sweep ();
};

static JavaHeap *g_JavaHeap;

#endif

/**
 * q[v̈
 *
 * @param	max_size	q[v̍őTCY
 */
void
initialize_heap (unsigned int max_size)
{
  g_max_heap_size = max_size;
  g_allocated_size = 0;
  g_static_data_stack = alloc_root_frame (NULL, STATIC_DATA_STACK_SIZE);

  // X[q[ṽtO
  memset (g_heap_flags, 1, sizeof (g_heap_flags));

#ifdef HEAP_STATISTICS_FILE
  // q[vvo͂
  g_heap_statistics_file = _tfopen (HEAP_STATISTICS_FILE, _T ("w"));
#endif

#ifdef USE_JAVA_HEAP
  g_JavaHeap = new JavaHeap (max_size);
#endif
}

/**
 * Kx[WRN^
 */
bool
init_gc ()
{
  jobject lock_object = NULL;
  frame *root_frame = alloc_root_frame (NULL);
  ClassFile *reference_class = find_ClassFile (root_frame,
					       intern_utf8
					       ("java/lang/ref/Reference"));
  if (reference_class != NULL)
    {
      jfieldID lock =
	get_static_field_id (reference_class, intern_utf8 ("lock"),
			     intern_utf8 ("Ljava/lang/Object;"));
      get_static_field (root_frame, get_static_data (reference_class), lock);
      POP_OBJECT (root_frame, lock_object);
    }
  free_root_frame (root_frame);
  if (lock_object != NULL)
    {
      g_Reference_lock = lock_object;
      return true;
    }
  else
    {
      return false;
    }
}

/**
 * 64oCg̃mۂ
 */
static void *
alloc_64bytes (small_heap_element * elem)
{
  for (int i = 0; i < SMALL_BLOCK_COUNT; ++i)
    {
      __int32 flag = elem->flags[i];
      if (flag)
	{
	  // ԉE["1"ƂȂĂrbg̈ʒu߂AԉE["1"ƂȂĂrbg0ɂ
	  int pos = count_bits ((flag & (-flag)) - 1);
	  elem->flags[i] = flag & ((unsigned __int32) flag - 1);

	  // ItZbgvZ
	  unsigned int offset =
	    (i * SMALL_BLOCK_SIZE * 32) + (pos * SMALL_BLOCK_SIZE);
	  return elem->data + offset;
	}
    }
  return NULL;
}

/**
 * X[q[vGgmۂ
 */
static small_heap_element *
alloc_small_heap_element ()
{
//      small_heap_element* elem = (small_heap_element*) malloc(sizeof(small_heap_element));
  small_heap_element *elem =
    (small_heap_element *) calloc (sizeof (small_heap_element), 1);
  if (elem != NULL)
    {
      memset (elem->flags, 0xff, sizeof (elem->flags));
    }
  return elem;
}

/**
 * 64oCg̃mۂ
 */
static void *
alloc_64bytes ()
{
  void *result = NULL;
  for (int i = 0; i < sizeof (g_heap_elements) / sizeof (*g_heap_elements);
       ++i)
    {
      if (g_heap_flags[i])
	{
	  small_heap_element *elem = g_heap_elements[i];
	  if (elem == NULL)
	    {
	      elem = alloc_small_heap_element ();
	      if (elem == NULL)
		{
		  break;
		}
	      g_heap_elements[i] = elem;
	    }
	  result = alloc_64bytes (elem);
	  if (result != NULL)
	    {
	      break;
	    }
	  else
	    {
	      g_heap_flags[i] = 0;
	    }
	}
    }
  // [NA͗̈悪Jꂽ_ōsĂ邽߁A
  // ōsKv͂Ȃ
//      if (result != NULL) {
//              memset(result, 0, SMALL_BLOCK_SIZE);
//      }
  return result;
}

/**
 * 64oCg̃J
 */
static void
free_64bytes (small_heap_element * elem, void *p)
{
  unsigned int offset = (char *) p - elem->data;
  int index = offset / (SMALL_BLOCK_SIZE * 32);
  int pos = (offset - (index * SMALL_BLOCK_SIZE * 32)) / SMALL_BLOCK_SIZE;
  elem->flags[index] |= (1 << pos);
}

static void
free_64bytes (void *p)
{
  for (int i = 0; i < sizeof (g_heap_elements) / sizeof (*g_heap_elements);
       ++i)
    {
      small_heap_element *elem = g_heap_elements[i];
      if (elem != NULL)
	{
	  if (p >= elem->data && p < elem->data + sizeof (elem->data))
	    {
	      // gpɔă[NAĂ
	      memset (p, 0, SMALL_BLOCK_SIZE);
	      free_64bytes (elem, p);
	      g_heap_flags[i] = 1;	// ȂƂP̋󂫂
	      return;
	    }
	}
    }
  // ȃ|C^nꂽ
  assert (false);
  fatal_error (_T ("free_64bytes() failed."));
}

/**
 * q[v烁mۂ
 */
static void *
heap_alloc (size_t size)
{
#ifdef USE_JAVA_HEAP

  return g_JavaHeap->allocate (size);

#else

#if SMALL_HEAP_ENABLED
  // X[q[vTCY
  if (size <= SMALL_BLOCK_SIZE)
    {
      size = SMALL_BLOCK_SIZE;
    }
#endif
  assert (!g_gc_running);	// GCs͐VKq[vmۂłȂ

  if (g_allocated_size + size >= g_max_heap_size)
    {
      gc ();
      if (g_allocated_size + size >= g_max_heap_size)
	{
	  return NULL;
	}
    }

  // mۂA0ŏ
  java_object *elem;

#if SMALL_HEAP_ENABLED
  // X[q[vmۏs
  if (size == SMALL_BLOCK_SIZE)
    {
      elem = (java_object *) alloc_64bytes ();
    }
  else
    {
      elem = (java_object *) calloc (size, 1);
    }
#else
  elem = (java_object *) calloc (size, 1);
#endif

  if (!elem)
    {
      // GCsĂ݂
      gc ();
#if SMALL_HEAP_ENABLED
      // X[q[vmۏs
      if (size == SMALL_BLOCK_SIZE)
	{
	  elem = (java_object *) alloc_64bytes ();
	}
      else
	{
	  elem = (java_object *) calloc (size, 1);
	}
#else
      elem = (java_object *) calloc (size, 1);
#endif
      if (!elem)
	{
	  // MuAbv
	  return elem;
	}
    }

  // mۂTCYZ
  g_allocated_size += size;
  elem->size = size;

  if (g_heap_root == NULL)
    {
      // Xg擪vf쐬
      elem->next = NULL;
      g_heap_root = elem;
    }
  else
    {
      // Xg̐擪Ɋmۂq[v}
      elem->next = g_heap_root;
      g_heap_root = elem;
    }

#ifdef	HEAP_STATISTICS_FILE
  _TCHAR buff[128];
  _stprintf (buff, _T ("%d bytes allocated.\n"), size);
  _fputts (buff, g_heap_statistics_file);
#endif

  return elem;

#endif /* ! USE_JAVA_HEAP */
}

/**
 * q[v烁J
 */
#ifndef USE_JAVA_HEAP
static void
heap_free (java_object * elem)
{
  // JTCYZ
  int size = elem->size;
  g_allocated_size -= size;

#ifdef DEBUG
  // J̈ɃANZXĂ܂Ƃmł悤ɈĂ
  ClassFile *cfile = elem->class_file;
  memset (elem, 0xff, size);
  elem->class_file = cfile;
#endif

#if SMALL_HEAP_ENABLED
  if (size <= SMALL_BLOCK_SIZE)
    {
      free_64bytes (elem);
    }
  else
    {
      // ^CCuɃԋp
      free (elem);
    }
#else
  free (elem);
#endif

}
#endif

/**
 * w肳ꂽIuWFNgAv~eBuz񂩂Ԃ
 */
static inline bool
is_non_primitive_array (java_object * obj)
{
  java_utf8 *name = obj->class_file->this_class_name;
  return name[0] == '[' && (name[1] == 'L' || name[1] == '[');
}

/**
 * w肳ꂽNXstaticf[^IuWFNg쐬B
 * ̃IuWFNg̓Kx[WRNgȂ悤ɕی삳B
 *
 * @param	cfile	ClassFile\̂̃|C^B
 * @return	NXIuWFNgB쐬łȂꍇNULLB
 */
jobject
alloc_static_data (ClassFile * cfile)
{
  // NeBJZNVɓ
  EnterCriticalSection (&g_global_critical_section);

  // mۂ郁TCYvZ
  unsigned int static_fields_size = get_static_fields_size (cfile);
  size_t size = sizeof (java_object) + static_fields_size;

  java_object *data = (java_object *) heap_alloc (size);
  if (data != NULL)
    {
      data->class_file = cfile;
      data->count = static_fields_size;
      data->flags |= STATIC_DATA;
      if (!ensure_stack_capacity (g_static_data_stack))
	{
	  fatal_error (_T ("g_static_data_stack overflow"));
	}
      PUSH_OBJECT (g_static_data_stack, (jobject) data);
    }

  // NeBJZNV甲
  LeaveCriticalSection (&g_global_critical_section);

  return (jobject) data;
}

/**
 * w肳ꂽNX̃CX^Xf[^̈쐬
 * ̊֐̌Ăяoɐ旧āAw肳ꂽNXуX[p[NXAX[p[CX^X
 * ĂȂ΂ȂȂ
 *
 * @param	cfile	ClassFile\̂̃|C^B
 * @param	QƂpushX^bN
 * @return	IuWFNgւ̎QƁBmۂłȂꍇNULLB
 */
jobject
alloc_object (ClassFile * cfile, frame * stack)
{
  // NeBJZNVŕی삷
  EnterCriticalSection (&g_global_critical_section);

  assert (cfile->this_class_name[0] != '[');

  bool finalize_object = has_finalizer (cfile);
  if (finalize_object)
    {
      g_finalize_objects_count++;
      if (g_finalize_objects_count > FINALIZE_OBJECTS_COUNT_THRESHOLD)
	{
	  // finalize()IuWFNg̐臒l𒴂̂ŋIGC
	  gc ();
	}
    }

  // ̃NX̃tB[hTCY擾
  int fields_size = get_fields_size (cfile);
  assert (fields_size >= 0);

  // q[v̈悩烁mۂ
  size_t size = sizeof (java_object) + fields_size;
  java_object *data = (java_object *) heap_alloc (size);
  if (data == NULL)
    {
      // mۂłȂꍇ́AOutOfMemoryErrorthrow
      throw_OutOfMemoryError (stack);
    }
  else
    {

      // t@CiCUێĂ邩𒲂ׁAIuWFNgɈĂ
      if (finalize_object)
	{
	  data->flags |= FINALIZE_MARK;
	}

      // KvƂȂf[^
      data->class_file = cfile;
      data->count = fields_size;

      // X^bNpush
      PUSH_OBJECT (stack, (jobject) data);
    }
  // NeBJZNVI
  LeaveCriticalSection (&g_global_critical_section);

  return (jobject) data;
}

/**
 * w肳ꂽz̗̈쐬
 * ̊֐̌Ăяoɐ旧āAw肳ꂽNXуX[p[NXAX[p[CX^X
 * ĂȂ΂ȂȂ
 *
 * @param	cfile	ClassFile\̂̃|C^BzNX\ĂȂ΂ȂȂB
 * @param	count	z̗vfBłȂ΂ȂȂB
 * @return	z̓eێ̈BmۂłȂꍇNULLB
 */
jarray
alloc_array (ClassFile * cfile, __int32 count, frame * stack)
{
  // NeBJZNVɓ
  EnterCriticalSection (&g_global_critical_section);

  assert (count >= 0);
  assert (cfile->this_class_name[0] == '[');

  // q[v̈悩烁mۂ
  unsigned int component_size =
    get_array_component_size (cfile->this_class_name);
  size_t size = sizeof (java_object) + component_size * count;
  java_object *data = (java_object *) heap_alloc (size);
  if (data == NULL)
    {
      // s̏ꍇOutOfMemoryErrorthrow
      throw_OutOfMemoryError (stack);
    }
  else
    {
      // f[^
      data->class_file = cfile;

      // array.length  count ɓ
      data->count = count;

      // X^bNpush
      PUSH_OBJECT (stack, (jobject) data);
    }

  // NeBJZNVI
  LeaveCriticalSection (&g_global_critical_section);

  return (jarray) data;
}

/**
 * w肳ꂽUTF-8Javachar[]ɕϊ
 * w肳ꂽUTF-8UnicodeɕϊꂽAUnicodeێ
 * charz񂪍쐬
 *
 * @param	cfile	char[]\ClassFile\́B(NX"[C")
 * @param	utf8	UTF-8
 * @param	stack	쐬z̎QƂpushX^bNEt[B
 * @return	쐬z̎QƁB쐬łȂꍇNULL
 */
jcharArray
alloc_char_array (ClassFile * cfile, const java_utf8 * utf8, frame * stack)
{
  // NeBJZNVɓ
//      EnterCriticalSection(&g_global_critical_section);

  // Unicodeobt@쐬
  int unicode_length = strlen (utf8) * 2 + 1;
  TCHAR *unicode = (TCHAR *) malloc (sizeof (TCHAR) * unicode_length);
  if (unicode == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  jcharArray array_reference = NULL;
  // UTF-8UnicodeɃRo[g
  int len = convert_to_TCHAR (utf8, unicode, unicode_length);
  array_reference = alloc_char_array (cfile, unicode, len, stack);

  // 쐬Unicodeobt@J
  free (unicode);

  // NeBJZNV甲
//      LeaveCriticalSection(&g_global_critical_section);

  return array_reference;
}

/**
 * w肳ꂽQƂ̓e𕡐
 */
jobject
clone (frame * frm, jobject ref)
{
  ClassFile *cfile = get_ClassFile (ref);
  jobject result = NULL;
  if (is_array_class (cfile))
    {
      result = alloc_array (cfile, get_array_length ((jarray) ref), frm);
    }
  else
    {
      result = alloc_object (cfile, frm);
    }

  if (result != NULL)
    {
      java_object *dst = lock_address (result);
      void *dstptr = get_body_ptr (dst);

      java_object *src = lock_address (ref);
      void *srcptr = get_body_ptr (src);

      unsigned int bodysize = dst->size - sizeof (java_object);
      memcpy (dstptr, srcptr, bodysize);

      unlock_address (result);
      unlock_address (ref);
    }
  return result;
}

/**
 * w肳ꂽUnicodeJavachar[]ɕϊ
 *
 * @param	cfile	char[]\ClassFile\́B(NX"[C")
 * @param	unicode	Unicode
 * @param	length	Unicode̒B
 * @param	stack	쐬z̎QƂpushX^bNEt[B
 * @return	쐬z̎QƁB쐬łȂꍇNULL
 */
jcharArray
alloc_char_array (ClassFile * cfile, _TCHAR * unicode, int length,
		  frame * stack)
{
  // NeBJZNVɓ
//      EnterCriticalSection(&g_global_critical_section);

  // zIuWFNgmۂ
  jobject array_reference = alloc_array (cfile, length, stack);
  if (array_reference == NULL)
    {
      throw_OutOfMemoryError (stack);
    }
  else
    {
      // zvfUnicode𒼐ڃRs[
      java_object *obj = lock_address (array_reference);
      TCHAR *buff = (TCHAR *) get_body_ptr (obj);
      memcpy (buff, unicode, sizeof (TCHAR) * length);
      unlock_address (array_reference);
    }

  // NeBJZNV甲
//      LeaveCriticalSection(&g_global_critical_section);

  return (jcharArray) array_reference;
}

/**
 * w肳ꂽstatictB[hɒlݒ肷
 *
 * @param	cfile	ݒΏۂƂȂNX
 * @param	index	JgNXł̃CfbNX
 * @param	frame	ݒli[ꂽX^bNt[
 */
void
put_field (frame * frame, ClassFile * cfile, jfieldID fid)
{
  // tB[hTCYAX^bNɂQƂ̈ʒu߂
  unsigned int size = get_field_info (cfile, fid)->size;
  int offset;
  switch (size)
    {
    case 1:
    case 2:
    case 4:
      offset = 1;
      break;
    case 8:
      offset = 2;
      break;
    default:
      // ɂ͂Ȃ͂
      assert (false);
      return;
    }

  // QƂ擾
  // Kx[WRN^ɉĂ܂Ȃ悤ɁApop͂Ȃ
  __int32 data;
  get_stack_data (frame, offset, &data);
  jobject ref = (jobject) data;
  if (ref == NULL)
    {
      throw_exception (frame, "java/lang/NullPointerException");
      return;
    }

  // IuWFNg̃AhXbN
  java_object *obj = lock_address (ref);

  // tB[hɒlݒ肷
  field_info *finfo = get_field_info (obj->class_file, fid);
  unsigned char *address = (unsigned char *) get_body_ptr (obj);
  address += finfo->position;
  PUT_FIELD (frame, address, finfo->descriptor);

  // QƂpop
  POP_OBJECT (frame, ref);

  // IuWFNg̃AhXAbN
  unlock_address (ref);
}

/**
 * w肳ꂽstatictB[hɒlݒ肷
 *
 * @param	frame	ݒli[ꂽX^bNt[
 * @param       fieldref  ΏۃtB[hQƂĂCONSTANT_Fieldref_info
 */
void
put_field_by_position (frame * frame, const CONSTANT_Fieldref_info* fieldref)
{
  assert(fieldref->finfo);

  // tB[hTCYAX^bNɂQƂ̈ʒu߂
  u1 type = fieldref->field_descriptor[0];

  int offset;
  switch (type)
    {
    case 'J':
    case 'D':
      offset = 2;
      break;

    default:
      offset = 1;
      break;
    }

  // QƂ擾
  // Kx[WRN^ɉĂ܂Ȃ悤ɁApop͂Ȃ
  __int32 data;
  get_stack_data (frame, offset, &data);
  jobject ref = (jobject) data;
  if (ref == NULL)
    {
      throw_exception (frame, "java/lang/NullPointerException");
      return;
    }

  // IuWFNg̃AhXbN
  java_object *obj = lock_address (ref);

  // tB[hɒlݒ肷
  unsigned char *address = (unsigned char *) get_body_ptr (obj);
  address += get_field_position(fieldref);
  PUT_FIELD (frame, address, fieldref->field_descriptor);

  // QƂpop
  POP_OBJECT (frame, ref);

  // IuWFNg̃AhXAbN
  unlock_address (ref);
}

/**
 * w肳ꂽstatictB[hɒlݒ肷
 *
 * @param	ref		IuWFNg̎Q
 * @param	fid		tB[hID
 * @param	frame	ݒli[ꂽX^bNt[ւ̃|C^
 */
void
put_static_field (frame * frame, jobject ref, jfieldID fid)
{
  ClassFile *target_class_file = get_ClassFile (ref);
  field_info *finfo = get_static_field_info (target_class_file, fid);
  ClassFile *declaring_class_file = finfo->declaring_ClassFile;
  if (target_class_file != declaring_class_file)
    {
      ref = get_static_data (declaring_class_file);
    }

  // IuWFNg̃AhXbN
  java_object *obj = lock_address (ref);

  // tB[hɒlݒ肷
  unsigned char *address = (unsigned char *) get_body_ptr (obj);
  address += finfo->position;
  PUT_FIELD (frame, address, finfo->descriptor);

  // IuWFNg̃AhXAbN
  unlock_address (ref);
}

/**
 * w肳ꂽstatictB[h̒lԂ
 *
 * @param	ref		IuWFNgւ̎QƁBNXIuWFNgłȂ΂ȂȂ
 * @param	fid		tB[hID
 * @param	stack	tB[h̒li[X^bNEt[
 */
bool
get_static_field (frame * frame, jobject ref, jfieldID fid)
{
  assert(fid);

  ClassFile *target_class_file = get_ClassFile (ref);
  field_info *finfo = get_static_field_info (target_class_file, fid);
  ClassFile *declaring_class_file = finfo->declaring_ClassFile;
  if (target_class_file != declaring_class_file)
    {
      ref = get_static_data (declaring_class_file);
    }
  // IuWFNg̃AhXbN
  java_object *obj = lock_address (ref);
  unsigned char *address = (unsigned char *) get_body_ptr (obj);
  address += finfo->position;
  GET_FIELD (frame, address, finfo->descriptor, false);

  // IuWFNg̃AhXAbN
  unlock_address (ref);
  return true;
}

/**
 * w肳ꂽtB[h̒lԂ
 *
 * @param	cfile	NXt@C
 * @param	fid		tB[hID
 * @param	stack	tB[h̒li[X^bNEt[
 */
bool
get_field (frame * frame, ClassFile * cfile, jfieldID fid)
{
  jobject ref;
  get_stack_data (frame, 0, (__int32 *) & ref);

  // QƂnull𒲂ׁAnull̏ꍇNullPointerExceptionthrow
  if (ref == NULL)
    {
      throw_exception (frame, "java/lang/NullPointerException");
      return false;
    }


  // IuWFNg̃AhXbN
  java_object *obj = lock_address (ref);

  field_info *finfo = get_field_info (obj->class_file, fid);
  // tB[h̃AhXvZ
  unsigned int position = finfo->position;
  unsigned char *ptr = (unsigned char *) get_body_ptr (obj);
  ptr += position;

  // Kx[WRNg
//      jobject global_ref = new_global_reference(ref); 
//      POP_DISCARD(frame);
//      ref = global_ref;

  GET_FIELD (frame, ptr, finfo->descriptor, true);

  // AhXAbN
  unlock_address (ref);

  // Kx[WRNg\ɂ
//      delete_global_reference(global_ref);    

  return true;
}

/**
 * w肳ꂽtB[h̒lԂ
 *
 * @param	frame	tB[h̒li[t[
 * @param       fieldref   tB[hQƂĂCONSTANT_Fieldref_infoւ̃|C^
 */
void
get_field_by_position (frame * frame, const CONSTANT_Fieldref_info* fieldref)
{
  assert(fieldref->finfo);

  jobject ref;
  get_stack_data (frame, 0, (__int32 *) & ref);

  // QƂnull𒲂ׁAnull̏ꍇNullPointerExceptionthrow
  if (ref == NULL)
    {
      throw_exception (frame, "java/lang/NullPointerException");
      return;
    }

  // IuWFNg̃AhXbNAtB[h̃AhX擾
  java_object *obj = lock_address (ref);
  unsigned char *ptr = (unsigned char *) get_body_ptr (obj) + get_field_position(fieldref);

  // tB[h擾Aframepush
  GET_FIELD (frame,		              // ʂpushframe
	     ptr,		              // tB[h̃AhX
	     fieldref->field_descriptor,      // fBXNv^
	     true);		              // frameQƒlpop

  // AhXAbN
  unlock_address (ref);
}

/**
 * w肳ꂽstatictB[h̒lԂ
 *
 * @param	frame      tB[h̒li[t[
 * @param       fieldref   X^eBbNtB[hQƂĂCONSTANT_Fieldref_infoւ̃|C^
 */
void get_static_field_by_position(frame* frame, const CONSTANT_Fieldref_info* fieldref)
{
  assert(fieldref->finfo);
  ClassFile *declaring_class_file = fieldref->finfo->declaring_ClassFile;

  jobject ref = get_static_data (declaring_class_file);
  assert(ref);

  // IuWFNg̃AhXbNAtB[h̃AhX擾
  java_object *obj = lock_address (ref);
  unsigned char *ptr = (unsigned char *) get_body_ptr (obj) + get_field_position(fieldref);

  // tB[h擾Aframepush
  GET_FIELD (frame,		          // ʂpushframe
	     ptr,		          // tB[h̃AhX
             fieldref->field_descriptor,  // fBXNv^
	     false);		          // frameQƒlpopȂ

  // AhXAbN
  unlock_address (ref);
}

/**
 * w肳ꂽstatictB[hɒlݒ肷
 *
 * @param	frame             tB[h̒lݒ肳ꂽframe
 * @param       fieldref   X^eBbNtB[hQƂĂCONSTANT_Fieldref_infoւ̃|C^
 */
void put_static_field_by_position(frame* frame, const CONSTANT_Fieldref_info* fieldref)
{
  assert(fieldref->finfo);
  ClassFile *declaring_class_file = fieldref->finfo->declaring_ClassFile;
  jobject ref = get_static_data (declaring_class_file);
  assert(ref);

  // IuWFNg̃AhXbN
  java_object *obj = lock_address (ref);

  // tB[hɒlݒ肷
  unsigned char *address = (unsigned char *) get_body_ptr (obj);
  address += get_field_position(fieldref);
  PUT_FIELD (frame, address, fieldref->field_descriptor);

  // IuWFNg̃AhXAbN
  unlock_address (ref);
}


/**
 * IuWFNg̔zvf擾
 */
bool
load_object_array (jarray arrayref, __int32 index, jobject * pref)
{
  java_object *obj = lock_address (arrayref);
  if (index < 0 || index >= obj->count)
    {
      unlock_address (arrayref);
      return false;
    }
  jobject *body = (jobject *) get_body_ptr (obj);
  *pref = body[index];
  unlock_address (arrayref);
  return true;
}

/**
 * w肳ꂽz̗vfݒ肷
 *
 * @param	arrayref	zIuWFNg
 * @param	index		ݒ肷\vf̃CfbNX
 * @param	value		ݒl
 * @return	ɏłꍇtrueBCfbNXlz͈̔͊OłꍇfalseB
 */
bool
store_object_array (jarray arrayref, __int32 index, jobject value)
{
  java_object *obj = lock_address (arrayref);
  if (index < 0 || index >= obj->count)
    {
      unlock_address (arrayref);
      return false;
    }
  jobject *body = (jobject *) get_body_ptr (obj);
  body[index] = value;
  unlock_address (arrayref);
  return true;
}

/**
 * ̃IuWFNg̃j^Ԃ
 *
 */
monitor *
get_monitor (jobject ref)
{
  java_object *obj = lock_address (ref);
  monitor *monitor = obj->object_monitor;
  unlock_address (ref);
  return monitor;
}

/**
 * ̃IuWFNg̃j^ݒ肷
 *
 */
void
set_monitor (jobject ref, monitor * m)
{
  java_object *obj = lock_address (ref);
  obj->object_monitor = m;
  unlock_address (ref);
}

/**
 * Kx[WRN^N
 */
void
gc ()
{
  DBG (_T ("GC invoked. size="));
  DBG_UINT (g_allocated_size);
  DBG (_T ("\n"));

  // NeBJZNVɓ
  EnterCriticalSection (&g_global_critical_section);
  if (g_gc_running)
    {
      // ċAIɌĂяoƂ͂łȂ
      LeaveCriticalSection (&g_global_critical_section);
      DBG (_T ("GC recursive call, then return.\n"));
      return;
    }
  g_gc_running = true;

  // t@CiCYIuWFNgNA
  g_finalize_objects_count = 0;

  // s̃Xbh̃vCIeBۑ
  int org_priority = CeGetThreadPriority (GetCurrentThread ());

  // Kx[WRNV̏\Ȍ葁I点邽߁ADxグ
  CeSetThreadPriority (GetCurrentThread (), 248);

  // GCsÕq[vTCYƃptH[}XJE^l擾Ă
  unsigned int before_size = g_allocated_size;
  LARGE_INTEGER before_counter;
  QueryPerformanceCounter (&before_counter);

  // java.lang.ref.ReferencẽbN擾
  if (g_Reference_lock == NULL)
    {
      fatal_error (_T ("Object java.lang.ref.Reference is null"));
    }
  monitor_enter (g_Reference_lock);

  DBG (_T ("GC suspends all other threads\n"));
  // SXbhTXyh
  if (!suspend_all_other_threads ())
    {
      // XbhTXyhłȂBMuAbv
      DBG (_T ("FATAL: GC could not suspend all other threads.\n"));
      monitor_exit (g_Reference_lock);
      CeSetThreadPriority (GetCurrentThread (), org_priority);
      LeaveCriticalSection (&g_global_critical_section);
      return;
    }

  // }[N
  g_references_count = 0;
  g_references = NULL;
  DBG (_T ("GC marks objects.\n"));
  gc_mark ();

  DBG (_T ("GC resumes suspended threads.\n"));
  // XbhĊJłȂBɂ܂
  if (!resume_all_other_threads ())
    {
      monitor_exit (g_Reference_lock);
      CeSetThreadPriority (GetCurrentThread (), org_priority);
      LeaveCriticalSection (&g_global_critical_section);

      DBG (_T ("FATAL: Cannot resume all other threads.\n"));
      fatal_error (_T ("Cannot resume threads"));

      return;
    }

  // java.lang.ref.ReferencẽL[COs
  DBG (_T ("GC enqueue References.\n"));
  gc_enqueue ();

  // t@CiCY̏
  // ȉɃt@CiCYKvȃIuWFNgWj
  gc_prepare_finalize ();

  // XC[vB
  DBG (_T ("GC sweeps unused objects.\n"));

#ifdef USE_JAVA_HEAP
  g_JavaHeap->sweep ();
#else
  gc_sweep ();
#endif

  // t@CiCU̖ڂo܂B
  // t@CiCU𓯈XbhŎsĂ܂ƃfbhbN邽߁A
  // ut@CiCUXbhvŎs
  DBG (_T ("GC will wakeup finalize thread.\n"));
  finalizer *f = get_finalizer ();
  gc_start_finalizer ();

  // Reference.lock 
  monitor_exit (g_Reference_lock);
  DBG (_T ("GC Unlocked java.lang.ref.Reference.lock\n"));

  // Kx[WRN^s̃ptH[}XJE^擾
  LARGE_INTEGER after_counter;
  QueryPerformanceCounter (&after_counter);
  LARGE_INTEGER freq;
  QueryPerformanceFrequency (&freq);

  if (g_loggc_filename || g_verbose_gc)
    {
      _TCHAR buff[128];

      LONGLONG diff = after_counter.QuadPart - before_counter.QuadPart;
      double time = (double) diff / (double) freq.QuadPart;
      _stprintf (buff,
		 _T ("[FULL GC %uK -> %uK(%uK) %lf secs]\r\n"),
		 before_size / 1024,
		 g_allocated_size / 1024,
		 (g_max_heap_size - g_allocated_size) / 1024, time);
      if (g_loggc_filename)
	{
	  // -Xloggc:filename IvVw肳Ăꍇ
	  FILE *f = _tfopen (g_loggc_filename, _T ("a"));
	  if (f)
	    {
	      _fputts (buff, f);
	      fclose (f);
	    }
	}
      if (g_verbose_gc)
	{
	  console_write (buff);
	}
    }

  // Xbh̃vCIeBɖ߂
  CeSetThreadPriority (GetCurrentThread (), org_priority);

  // GCstONA
  g_gc_running = false;

  // NeBJZNV𔲂
  LeaveCriticalSection (&g_global_critical_section);

  DBG (_T ("GC finished. size="));
  DBG_UINT (g_allocated_size);
  DBG (_T ("\n"));
}

/**
 * }[Ns
 */
static void
gc_mark ()
{
  unsigned int root_count = get_root_frame_count ();
  for (unsigned int i = 0; i < root_count; ++i)
    {
      frame *root_frame = get_root_frame (i);
      if (root_frame)
	{
	  gc_mark (root_frame);
	}
    }
}

/**
 * w肳ꂽframe珇Ԃjava_objectmarkĂ
 */
static void
gc_mark (frame * root_frame)
{
  __int32 *data;
  __int32 *references;
  unsigned int count;
  jobject current_exception;
  if (get_root_frame_info
      (root_frame, &data, &references, &count, &current_exception))
    {
      for (unsigned int i = 0; i < count; ++i)
	{
	  __int32 ref = *references++;
	  if (*data++ == ref	// ݂Ƌʂ邽߂̔r
	      && ref		// null̓}[NȂ
	      && !(ref & 0x80000000))
	    {			// returnAddress^̓}[NȂ
	      // QƒlĂꍇ
	      gc_mark ((jobject) ref, GC_MARKED);
	    }
	}
      // O}[N
      if (current_exception != NULL)
	{
	  DBG (_T ("GC marks current exception\n"));
	  gc_mark (current_exception, GC_MARKED);
	}
    }
}

/**
 * w肳ꂽIuWFNgɁAw肳ꂽ}[N
 */
static void
gc_mark (jobject ref, int mark)
{
  if (ref == NULL)
    {
      return;
    }
  java_object *obj = lock_address (ref);
  if ((obj->flags & GC_MARKED) != 0)
    {
      // Błi[vj
      unlock_address (ref);
      return;
    }
  else if ((obj->flags & mark) != 0)
    {
      // Bł͂ȂAw肳ꂽ}[N͂łɂĂi[vj
      unlock_address (ref);
      return;
    }

  // }[N
  obj->flags |= mark;

  // ̃IuWFNgQƂĂIuWFNgAċAIɃ}[NĂ
  ClassFile *cfile = obj->class_file;
  const java_utf8 *cname = cfile->this_class_name;
  jobject *refs = (jobject *) get_body_ptr (obj);
  if (is_non_primitive_array (obj))
    {
      // z܂̓IuWFNg̔z̏ꍇAevf͎QƂɂȂ
      for (int i = 0; i < obj->count; ++i)
	{
	  gc_mark (refs[i], mark);
	}
    }
  else if (cname[0] != '[')
    {
      // z
      // Kx[WRN^́uQƃtOvɃ}[NsB
      // uQƃtOv́AeoCguQƒlǂv\oCgz
      // łAP̗vf̓IuWFNgf[^4oCgɑΉĂB
      // eQƒl͕K4oCgEɔzu邽߁AKx[WRN^
      // tB[ȟ^PP`FbNKvȂA4oCgPʂ
      // sƂ\ƂȂĂB
      bool *reference_flags;
      if ((obj->flags & STATIC_DATA) != 0)
	{
	  // staticIuWFNg̏ꍇ
	  reference_flags = get_static_field_reference_flags (cfile);
	}
      else
	{
	  // CX^XIuWFNg̏ꍇ
	  reference_flags = get_field_reference_flags (cfile);
	}
      int count = obj->count / 4;

      int startpos = 0;
      int type;
      if ((type = get_reference_type_flag (obj)) != 0)
	{
	  if (type != WEAK_REFERENCE_FLAG)
	    {
	      // WeakReferenceȊO̎QƃIuWFNg̏ꍇA擪̃tB[hireferent)
	      // ʂȃtO REFERENT_MARK 
	      jobject referent = refs[0];	// tB[h referent
	      if (referent != NULL)
		{
		  gc_mark (referent, REFERENT_MARK);
		}
	    }
	  startpos++;

	  // ŃL[COs߂ɁAzɎQƃIuWFNgB
	  g_references_count++;
	  g_references = (jobject *) realloc (g_references,
					      sizeof (jobject) *
					      g_references_count);
	  g_references[g_references_count - 1] = ref;
	}

      for (int i = startpos; i < count; ++i)
	{
	  // IuWFNg̗vfɃ}[NĂ
	  if (reference_flags[i])
	    {
	      jobject ref = refs[i];
	      if (ref != NULL)
		{
		  gc_mark (ref, mark);
		}
	    }
	}
    }
  unlock_address (ref);
}

/**
 * t@CiCY̏
 */
static void
gc_prepare_finalize ()
{
  // t@CiCU擾At@CiCUbN
  finalizer *f = get_finalizer ();
  EnterCriticalSection (f->lock);

  // [gGg珇ԂɒׂĂ
  for (java_object * elem = g_heap_root; elem != NULL; elem = elem->next)
    {
      if ((elem->flags & (GC_MARKED | REFERENT_MARK | FINALIZE_MARK)) ==
	  FINALIZE_MARK)
	{
	  // FINALIZE_MARK݂̂ĂIuWFNgA
	  // t@CiCUɓn
	  f->count++;
	  if (f->count >= f->array_size)
	    {
	      f->array_size += 128;	// 128₵Ă
	      f->target_objects =
		(jobject *) realloc (f->target_objects,
				     sizeof (jobject) * f->array_size);
	      if (!f->target_objects)
		{
		  fatal_error (FATAL_ERROR_NO_MEMORY);
		}
	    }
	  jobject target = make_reference (elem);
	  f->target_objects[f->count - 1] = target;

	  // t@CiCYOɉ̂h
	  // t@CiCYΏۂ̃IuWFNgюQƂĂIuWFNg
	  // ׂă}[NĂ
	  // ̕@̓_H
	  gc_mark (target, GC_MARKED);
	}
    }

  // t@CiCŨbN
  LeaveCriticalSection (f->lock);
}


/**
 * XC[vsB
 */
#ifndef USE_JAVA_HEAP
static void
gc_sweep ()
{
  // [gGg珇ԂɒׂĂ
  java_object *elem = g_heap_root;
  java_object *previous = NULL;
  while (elem != NULL)
    {
      java_object *next_elem = elem->next;
      bool marked = false;
      if ((elem->flags & REFERENT_MARK) != 0)
	{
	  // ut@gv}[NNA
	  elem->flags &= ~REFERENT_MARK;
	  marked = true;
	}
      if ((elem->flags & GC_MARKED) != 0)
	{
	  // GC_MARKEDNA
	  elem->flags &= ~GC_MARKED;
	  marked = true;
	}
      if ((elem->flags & FINALIZE_MARK) != 0)
	{
	  // ̎_ł FINALIZE_MARK ̓NAȂit@CiCUNAj
	  marked = true;
	}

      if (!marked)
	{
	  // }[NĂȂIuWFNg
	  // IuWFNg
	  jobject ref = make_reference (elem);
	  if (get_monitor (ref) != NULL)
	    {
	      // j^ۗLĂꍇɂ͍폜
	      delete_monitor (ref);
	    }

	  // XgΏۃGg菜
	  if (elem == g_heap_root)
	    {
	      g_heap_root = next_elem;
	    }
	  else
	    {
	      previous->next = next_elem;
	    }
	  heap_free (elem);
	}
      else
	{
	  // c
	  previous = elem;
	}
      elem = next_elem;
    }
}
#endif

/**
 * java.lang.ref.ReferencẽL[COs
 */
static void
gc_enqueue ()
{
  frame *frm = alloc_root_frame (NULL);
  for (int i = 0; i < g_references_count; ++i)
    {
      jobject o = g_references[i];
      int flags = 0;
      int reference_type = 0;

      {
	// AhXbN
	java_object *reference = lock_address (o);
	jobject *body = (jobject *) get_body_ptr (reference);

	// 擪̗vfireferent)̃}[N𒲂ׂ
	java_object *referent = lock_address (*body);
	if (referent != NULL)
	  {
	    flags = referent->flags;
	    reference_type = get_reference_type_flag (reference);
	  }

	// AhXAbN
	unlock_address (*body);
	unlock_address (o);
      }

      if ((flags & GC_MARKED) == 0)
	{
	  // Bł͂ȂȂĂꍇ
	  switch (reference_type)
	    {
	    case SOFT_REFERENCE_FLAG:
	    case WEAK_REFERENCE_FLAG:
	      {
		ClassFile *cfile = get_ClassFile (o);
		// clear()\bhĂяo
		jmethodID clear_mid = get_method_id (cfile,
						     intern_utf8 ("clear"),
						     intern_utf8 ("()V"));
		method_info *clear_minfo = get_method_info (cfile, clear_mid);
		ClassFile *declaring_cfile = clear_minfo->declaring_ClassFile;
		PUSH_OBJECT (frm, o);
		invoke_method (frm, o, declaring_cfile, clear_minfo);

		// enqueue()\bhĂяo
		jmethodID enqueue_mid = get_method_id (cfile,
						       intern_utf8
						       ("enqueue"),
						       intern_utf8 ("()Z"));
		method_info *enqueue_minfo =
		  get_method_info (declaring_cfile, enqueue_mid);
		declaring_cfile = enqueue_minfo->declaring_ClassFile;
		PUSH_OBJECT (frm, o);
		invoke_method (frm, o, declaring_cfile, enqueue_minfo);

		break;
	      }
	    case PHANTOM_REFERENCE_FLAG:
	      {
		ClassFile *cfile = get_ClassFile (o);
		// enqueue()\bhĂяo
		// ToDo : finalize()sɌĂяo悤ɏC
		jmethodID enqueue_mid = get_method_id (cfile,
						       intern_utf8
						       ("enqueue"),
						       intern_utf8 ("()Z"));
		method_info *enqueue_minfo =
		  get_method_info (cfile, enqueue_mid);
		ClassFile *declaring_cfile =
		  enqueue_minfo->declaring_ClassFile;
		PUSH_OBJECT (frm, o);
		invoke_method (frm, o, declaring_cfile, enqueue_minfo);

		break;
	      }
	    }
	}
    }
  free_root_frame (frm);
  free (g_references);
  g_references_count = 0;
}

/**
 * t@CiCYJn
 */
static void
gc_start_finalizer ()
{
  finalizer *f = get_finalizer ();
  EnterCriticalSection (f->lock);
  if (is_idling (f))
    {
      // t@CiCU̖ڂo܂
      gc_wakeup_finalizer (f);
    }
  else
    {
      // t@CiCUrW[ł
      // ToDo:t@CiCU𕡐XbhŎs\ɂ
      DBG (_T ("GC finalizer is busy.\n"));
      if (g_verbose_gc)
	{
	  console_write (_T ("[Finalize thread is busy.]\r\n"));
	}
    }
  LeaveCriticalSection (f->lock);
}

/**
 * IuWFNgf[^{̂Ăʒũ|C^bNB
 * ̊֐Ăяo_ŁAAhXʒubNAKx[WRNV
 * Ώۂł͂ȂȂB
 * pin_objectĂяôƓ unpin_object ֐Ăяo܂ŁA
 * bNꂽԂƂȂB
 */
void *
pin_object (jobject ref)
{
  java_object *obj = lock_address (ref);
  return get_body_ptr (obj);	// f[^{̂̃|C^ԂB
}

/**
 * w肳ꂽ|C^AbNB
 * ̊֐ɓn|C^́Apin_object()֐Ԃꂽ̂
 * gpȂ΂ȂȂB
 */
void
unpin_object (void *ptr)
{
  java_object *obj = (java_object *) ptr;
  obj--;			// IuWFNgwb_ʒuvZ
  jobject ref = make_reference (obj);
  unlock_address (ref);
}

/**
 * t@CiCUXbh
 */
bool
init_finalizer ()
{
  g_finalizer = (finalizer *) malloc (sizeof (finalizer));
  if (g_finalizer == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  g_finalizer->hWaiting = CreateEvent (NULL, FALSE, FALSE, NULL);
  g_finalizer->idling = true;
  g_finalizer->count = 0;
  g_finalizer->array_size = 0;	// TCY0
  g_finalizer->target_objects =
    (jobject *) malloc (sizeof (jobject) * g_finalizer->array_size);
  if (g_finalizer->target_objects == NULL)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }
  // bNp̃NeBJZNV쐬Ă
  g_finalizer->lock = (LPCRITICAL_SECTION) malloc (sizeof (CRITICAL_SECTION));
  InitializeCriticalSection (g_finalizer->lock);

  return true;
}

/**
 * t@CiCUԂ
 */
finalizer *
get_finalizer ()
{
  return g_finalizer;
}

/**
 * t@CiCU̖ڂo܂
 */
static void
gc_wakeup_finalizer (finalizer * f)
{
  EnterCriticalSection (f->lock);
  SetEvent (f->hWaiting);
  LeaveCriticalSection (f->lock);
}

/**
 * t@CiCYs
 */
void
gc_run_finalizer (frame * frm, finalizer * f)
{
  while (true)
    {
      // uAChOvtOݒ肷
      EnterCriticalSection (f->lock);
      f->idling = true;
      LeaveCriticalSection (f->lock);

      // CxgKx[WRN^ɂăVOiԂɂ̂҂
      WaitForSingleObject (f->hWaiting, INFINITE);

      // uAChOvtONA
      EnterCriticalSection (f->lock);
      f->idling = false;

      // t@CiCYΏۃIuWFNg̔zRs[
      const unsigned int count = f->count;
      // Kx[WRNgȂ悤frameɓĂ
      frame *targets = alloc_root_frame (NULL, count + 1);
      if (!targets)
	{
	  fatal_error (FATAL_ERROR_NO_MEMORY);
	}
      for (unsigned int i = 0; i < count; ++i)
	{
	  PUSH_OBJECT (targets, f->target_objects[i]);
	}
      // obt@̗LOɂĂ
      f->count = 0;

      // bN
      LeaveCriticalSection (f->lock);

      // ԂɃt@CiCUs
      DBG (_T ("GC starting finalize. target objects count="));
      DBG_INT (count);
      DBG (_T ("\n"));
      if (g_verbose_gc)
	{
	  _TCHAR msg[256];
	  _stprintf (msg, _T ("[Starting finalize (%d objects)]\r\n"), count);
	  console_write (msg);
	}
      jmethodID mid = NULL;
      for (unsigned int i = 0; i < count; ++i)
	{
	  jobject obj;
	  POP_OBJECT (targets, obj);
	  java_object *elem = lock_address (obj);

	  if (!mid)
	    {
	      mid = get_method_id (elem->class_file,
				   FINALIZE_METHOD_NAME,
				   VOID_NOARG_METHOD_DESCRIPTOR);
	    }

	  // IuWFNgQzɓ\邽߁A
	  // ōēx`FbN
	  if ((elem->flags & FINALIZE_MARK) != 0)
	    {
	      method_info *minfo = get_method_info (elem->class_file, mid);
	      ClassFile *declaring_class_file = minfo->declaring_ClassFile;

	      // IuWFNg̎QƂpush
	      PUSH_OBJECT (frm, obj);

	      // \bhĂяo
	      invoke_method (frm, obj, declaring_class_file, minfo);

	      // ONA
	      exception_clear (frm);

	      // finalize()ŝŁAFINALIZE_MARKNA
	      elem->flags &= ~FINALIZE_MARK;
	    }
	  unlock_address (elem);

	  // D揇ʂ̑̃XbhɃ^CXCX^isvHj
	  // Sleep(0);
	}

      // e|obt@J
      free_root_frame (targets);

      DBG (_T ("GC end finalize.\n"));
      if (g_verbose_gc)
	{
	  console_write (_T ("[End finalize]\r\n"));
	}
    }
}

/**
 * t@CiCUAChԂԂ
 */
static bool
is_idling (finalizer * f)
{
  return f->idling;
}

/**
 * z̃Rs[s
 */
void
array_copy (jobject src, unsigned int src_start, jobject dest,
	    unsigned int dest_start, int length)
{
  // IuWFNg̃AhXbN
  java_object *srcobj = lock_address (src);
  java_object *destobj = lock_address (dest);

  // ꂼ̊JnAhXvZ
  unsigned int compsize =
    get_array_component_size (srcobj->class_file->this_class_name);
  assert (compsize ==
	  get_array_component_size (destobj->class_file->this_class_name));
  unsigned char *srcaddr = (unsigned char *) get_body_ptr (srcobj);
  srcaddr += src_start * compsize;
  unsigned char *destaddr = (unsigned char *) get_body_ptr (destobj);
  destaddr += dest_start * compsize;

  // Rs[B̈悪dȂĂꍇ邽 memcpy() ł͂Ȃ memmove() gp
  memmove (destaddr, srcaddr, length * compsize);

  // IuWFNg̃AhXAbN
  unlock_address (src);
  unlock_address (dest);
}

/**
 * QƃIuWFNgł邱ƂKx[WRN^ɒʒmB
 * SoftReference/WeakReference/PhantomReferencẽCX^X́AK̊֐p
 * Kx[WRN^ɓo^ĂKvB
 * Kx[WRN^́Å֐œo^ꂽIuWFNgʈB
 * typeɎw肷鐔l͈ȉ̂ƂB
 */
void
set_reference_type (jobject obj, int reference_type)
{
  java_object *o = lock_address (obj);
  int flag = 0;
  switch (reference_type)
    {
    case SOFT_REFERENCE:
      flag = SOFT_REFERENCE_FLAG;
      break;

    case WEAK_REFERENCE:
      flag = WEAK_REFERENCE_FLAG;
      break;

    case PHANTOM_REFERENCE:
      flag = PHANTOM_REFERENCE_FLAG;
      break;
    }

  o->flags |= flag;

  unlock_address (obj);
}

/**
 * Qƃ^CvtO([SOFT|WEAK|PHANTOM]_REFERENCE_TYPE_FLAG)ԂB
 */
static int
get_reference_type_flag (java_object * obj)
{
  return obj->flags & (3 << 15);
}

/**
 * Kx[WRN^Ot@Cݒ肷
 */
void
set_loggc_filename (const _TCHAR * filename)
{
  if (filename)
    {
      g_loggc_filename =
	(_TCHAR *) malloc (sizeof (_TCHAR) * (_tcslen (filename) + 1));
      _tcscpy (g_loggc_filename, filename);
    }
}

/**
 * -verbose:gc IvVݒ肷
 */
void
gc_set_verbose (bool verbose)
{
  g_verbose_gc = verbose;
}

/**
 * 󂫃ʂԂ
 */
unsigned int
gc_get_free_memory ()
{
  return g_max_heap_size - g_allocated_size;
}

/**
 * q[vŜ̃TCYԂ
 */
unsigned int
gc_get_total_memory ()
{
  return g_max_heap_size;
}

/**
 * ő僁ʂԂ
 */
unsigned int
gc_get_max_memory ()
{
  return g_max_heap_size;
}


#ifdef USE_JAVA_HEAP

// ݂Exact GCConservative GCɕύX邽߂̏


/**
 * ؂グ
 */
#define ALIGN_UP(p, s) ((((p) % (s)) == 0) ? (p) : ((p) + ((s) - ((p) % (s)))))

/**
 * MEM_COMMITۂ̃ACg
 */
#define	COMMIT_ALIGN	(64*1024)

#define BIT_0_ON	1
#define BIT_1_ON	(1<<1)
#define BIT_2_ON	(1<<2)
#define BIT_3_ON	(1<<3)
#define BIT_4_ON	(1<<4)
#define BIT_5_ON	(1<<5)
#define BIT_6_ON	(1<<6)
#define BIT_7_ON	(1<<7)
#define BIT_8_ON	(1<<8)
#define BIT_9_ON	(1<<9)
#define BIT_10_ON	(1<<10)
#define BIT_11_ON	(1<<11)
#define BIT_12_ON	(1<<12)
#define BIT_13_ON	(1<<13)
#define BIT_14_ON	(1<<14)
#define BIT_15_ON	(1<<15)
#define BIT_16_ON	(1<<16)
#define BIT_17_ON	(1<<17)
#define BIT_18_ON	(1<<18)
#define BIT_19_ON	(1<<19)
#define BIT_20_ON	(1<<20)
#define BIT_21_ON	(1<<21)
#define BIT_22_ON	(1<<22)
#define BIT_23_ON	(1<<23)
#define BIT_24_ON	(1<<24)
#define BIT_25_ON	(1<<25)
#define BIT_26_ON	(1<<26)
#define BIT_27_ON	(1<<27)
#define BIT_28_ON	(1<<28)
#define BIT_29_ON	(1<<29)
#define BIT_30_ON	(1<<30)
#define BIT_31_ON	(1<<31)

__int32 const
  JavaHeap::BIT_MASK[32] = {
  BIT_0_ON,
  BIT_1_ON,
  BIT_2_ON,
  BIT_3_ON,
  BIT_4_ON,
  BIT_5_ON,
  BIT_6_ON,
  BIT_7_ON,
  BIT_8_ON,
  BIT_9_ON,
  BIT_10_ON,
  BIT_11_ON,
  BIT_12_ON,
  BIT_13_ON,
  BIT_14_ON,
  BIT_15_ON,
  BIT_16_ON,
  BIT_17_ON,
  BIT_18_ON,
  BIT_19_ON,
  BIT_20_ON,
  BIT_21_ON,
  BIT_22_ON,
  BIT_23_ON,
  BIT_24_ON,
  BIT_25_ON,
  BIT_26_ON,
  BIT_27_ON,
  BIT_28_ON,
  BIT_29_ON,
  BIT_30_ON,
  BIT_31_ON,
};

/**
 * JavaHeap쐬
 *
 * @param	őq[vTCY
 */
JavaHeap::JavaHeap (size_t maxSize)
{
  // rbgzɕKvȃʂvZ

  maxSize = ALIGN_UP (maxSize, COMMIT_ALIGN);

  unsigned int numBits = maxSize / OBJECT_ALIGN;
  assert ((numBits % 32) == 0);
  size_t bitsSize = numBits / 8;

  this->allocatedBits = (__int32 *) malloc (bitsSize);

  // zAhXԂ\
  this->heapBase =
    (char *) VirtualAlloc (NULL, (DWORD) maxSize, MEM_RESERVE, PAGE_NOACCESS);
  if (NULL == this->heapBase)
    {
      fatal_error (FATAL_ERROR_NO_MEMORY);
    }

  // oϐ
  this->heapLimit = this->heapBase + maxSize;
  this->startOfUncommitted = this->heapBase;
  this->freeObjects = NULL;
}

/**
 * JavaHeapJ
 */
JavaHeap::~JavaHeap ()
{
  char *address = this->heapBase;
  while (address < this->startOfUncommitted)
    {
      // R~bg
      VirtualFree (address, COMMIT_ALIGN, MEM_DECOMMIT);
      address += COMMIT_ALIGN;
    }
  // zAhXԂJ
  VirtualFree (this->heapBase, 0, MEM_RELEASE);

  free (this->allocatedBits);
}

/**
 * VirtualAlloc()ŃR~bg
 */
char *
JavaHeap::commitNextBlock ()
{
  char *result =
    (char *) VirtualAlloc (this->startOfUncommitted, COMMIT_ALIGN, MEM_COMMIT,
			   PAGE_READWRITE);
  ZeroMemory (result, COMMIT_ALIGN);
  this->startOfUncommitted += COMMIT_ALIGN;
  return result;
}

/**
 * w肳ꂽTCỸmۂ
 *
 * @param	size	mۂTCYBjava_object̃TCY܂
 */
java_object *
JavaHeap::allocate (size_t size)
{
  assert (size >= sizeof (java_object));

  // Ɂu`FbNpf[^v쐬邽߁A1oCgZ
  size_t allocation_size = ALIGN_UP (size + 1, OBJECT_ALIGN);

  bool retry = false;

RETRY:
  java_object * result = NULL;

  if (NULL == this->freeObjects)
    {
      java_object *freeObj = (java_object *) commitNextBlock ();
      if (NULL == freeObj)
	{
	  // 󂫗̈Ȃ
	  if (retry)
	    {
	      goto END;
	    }
	  gc ();
	  retry = TRUE;
	  goto RETRY;
	}
      freeObj->size = COMMIT_ALIGN;
      freeObj->next = NULL;
      writeCheckData (freeObj);
      this->freeObjects = freeObj;
    }

  // 󂫗̈̌Jn
  java_object *obj = this->freeObjects;
  java_object *prev_obj = NULL;

  while (true)
    {
      if (allocation_size <= obj->size)
	{
	  // 󂫗̈攭
	  if (allocation_size <= (obj->size - sizeof (java_object)))
	    {
	      // q[v𕪊
	      java_object *div_obj =
		(java_object *) (((char *) obj) + allocation_size);

	      // IuWFNg̃wb_AR~bg̈܂ł܂ꍇ
	      // ̗̈commitĂiHj
	      if (((char *) div_obj + sizeof (java_object)) >=
		  this->startOfUncommitted)
		{
		  if (!commitNextBlock ())
		    {
		      // gC
		      if (retry)
			{
			  goto END;
			}
		      gc ();
		      retry = true;
		      goto RETRY;
		    }

		  // IuWFNg̃TCYɂCOMMIT_ALIGNZ
		  div_obj->size = obj->size - allocation_size + COMMIT_ALIGN;

		}
	      else
		{
		  // IuWFNg̃TCYݒ肷
		  div_obj->size = obj->size - allocation_size;
		  assert (div_obj->size >= sizeof (java_object));
		}
	      div_obj->next = obj->next;
	      obj->next = div_obj;

	    }
	  else
	    {
	      // Ȃꍇ allocation_size 𒲐
	      allocation_size = obj->size;
	    }
	  // t[XgXV
	  if (obj == this->freeObjects)
	    {
	      // 擪u
	      assert (!prev_obj);
	      this->freeObjects = obj->next;
	    }
	  else
	    {
	      // NȂ
	      assert (prev_obj);
	      prev_obj->next = obj->next;
	    }
	  // IuWFNgTCY𒲐
	  obj->size = allocation_size;
	  obj->next = NULL;
	  writeCheckData (obj);

	  if (((char *) obj + obj->size) > this->heapLimit)
	    {
	      // q[v~bgщzĂ܂(=maxSize𒴂j
	      break;
	    }
	  // [vEo
	  result = obj;
	  break;
	}

      if (!obj->next)
	{
	  // ̃ubNR~bgĂ݂
	  char *nextBlock = commitNextBlock ();
	  if (!nextBlock)
	    {
	      // Ȃ
	      break;
	    }
	  if (((char *) obj + obj->size) == nextBlock)
	    {
	      // ݈ʒuƎubNAĂꍇA[v̐擪ɖ߂
	      obj->size += COMMIT_ALIGN;
	      continue;

	    }
	  else
	    {
	      // sAłꍇɂ́AIuWFNg̃wb_ݒ肵ナN
	      java_object *newobj = (java_object *) nextBlock;
	      newobj->size = COMMIT_ALIGN;
	      newobj->next = NULL;
	      obj->next = newobj;
	      writeCheckData (obj);
	    }
	}
      prev_obj = obj;
      obj = obj->next;
    }

  if (!result)
    {
      // Kx[WRN^NAă`W
      if (retry)
	{
	  goto END;
	}
      gc ();
      retry = true;
      goto RETRY;
    }

  // allocatedBitstO𗧂Ă
  setAllocatedBit (result);

  // `FbNpf[^ł
  // iŒ1oCg͗]ɊmۂĂ邽߁AރXy[Xj
  writeCheckData (result);

END:
  return result;
}

void
JavaHeap::sweep ()
{
  // 擪珇ԂɒׂĂ
  java_object *elem = (java_object *) this->heapBase;
  this->freeObjects = NULL;
  java_object *previous = NULL;
  java_object *limit =
    (java_object *) min (this->startOfUncommitted, this->heapLimit);
  while (elem < limit)
    {
      java_object *next_elem = (java_object *) ((char *) elem + elem->size);

      if (*(char *) ((char *) elem + elem->size - 1) != (char) CHECK_DATA)
	{
	  // `FbNf[^ȂBG[
	  DBG (_T ("Heap check failed. Class name="));
	  DBG_UTF8 (elem->class_file->this_class_name);
	  DBG (_T ("\n"));
	}

      bool marked = false;
      if ((elem->flags & REFERENT_MARK) != 0)
	{
	  // ut@gv}[NNA
	  elem->flags &= ~REFERENT_MARK;
	  marked = true;
	}
      if ((elem->flags & GC_MARKED) != 0)
	{
	  // GC_MARKEDNA
	  elem->flags &= ~GC_MARKED;
	  marked = true;
	}
      if ((elem->flags & FINALIZE_MARK) != 0)
	{
	  // ̎_ł FINALIZE_MARK ̓NAȂit@CiCUNAj
	  marked = true;
	}

      if (!marked)
	{
	  // }[NĂȂIuWFNg
	  jobject ref = make_reference (elem);
	  if (get_monitor (ref) != NULL)
	    {
	      // j^ۗLĂꍇɂ͍폜
	      delete_monitor (ref);
	    }

	  if (this->freeObjects == NULL)
	    {
	      assert (previous == NULL);
	      this->freeObjects = elem;
	      previous = elem;
	      previous->next = NULL;

	    }
	  else if (((char *) previous + previous->size) == (char *) elem)
	    {
	      // Ö̗ƌ
	      previous->size += elem->size;
	      previous->next = NULL;
	    }
	  else
	    {
	      // t[XgAĂ
	      previous->next = elem;
	      previous = elem;
	    }
	  // [NA
	  size_t size = previous->size;
	  memset (previous, 0, size);
	  previous->size = size;
	  writeCheckData (previous);
	}
      elem = next_elem;
    }
}

#endif
