/* WCEClip.java
   Copyright (C) 2006 Mysaifu.com

This file is part of Mysaifu JVM.

Mysaifu JVM is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; the version 2.

Mysaifu JVM is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package gnu.javax.sound.sampled.wce;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioPermission;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Clip;
import javax.sound.sampled.Control;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.sound.sampled.LineUnavailableException;

public class WCEClip extends WCEDataLine implements Clip {
	/**
	 * TEhf[^
	 */
	private byte[] soundData;

	/**
	 * o͐
	 */
	private final WCESourceDataLine sourceLine;
	
	/**
	 * ĐpXbh
	 */
	private Thread playbackThread;
	
	/**
	 * Jnʒu
	 */
	private int startFrame;
	
	/**
	 * [vJnʒu
	 */
	private int loopStart;
	
	/**
	 * [vIʒu
	 */
	private int loopEnd;
	
	/**
	 * [vJEg
	 */
	private int loopCount;
	
	/**
	 * Đ
	 */
	private volatile boolean playbackCompleted;
	
	public WCEClip(DataLine.Info info) {
		super(info);
		this.sourceLine = new WCESourceDataLine(info);
		
		// START/STOPCxg_CNg邽߂
		// Xiݒ肷
		this.sourceLine.addLineListener(new LineListener() {
			public void update(LineEvent e) {
				if (e.getType() == LineEvent.Type.START) {
					notifyLineEvent(new LineEvent(WCEClip.this,
										 LineEvent.Type.START,
										 AudioSystem.NOT_SPECIFIED));
				} else if (e.getType() == LineEvent.Type.STOP) {
					notifyLineEvent(new LineEvent(WCEClip.this,
										 LineEvent.Type.STOP,
										 AudioSystem.NOT_SPECIFIED));
				}
			}
		});
	}
	
	/**
	 * 0oCg̃NbvI[v
	 */
	public void open() throws LineUnavailableException {
		open(getDefaultFormat(), new byte[0], 0, 0);
	}
	
	/**
	 * w肳ꂽoCgzŕ\鉹f[^I[v
	 */
	public void open(AudioFormat format,
	                 byte[] data,
	                 int offset,
	                 int bufferSize)
	                 	throws LineUnavailableException {

		// obt@Ƀf[^Rs[
		this.soundData = new byte[bufferSize];
		System.arraycopy(data, offset, this.soundData, 0, bufferSize);

		open(format);
	}
	
	/**
	 * w肳ꂽXg[f[^ǂݍ݁AClipI[v
	 */
	public void open(AudioInputStream stream)
						throws LineUnavailableException, IOException {
		// obt@Ƀf[^ǂݍ
		byte[] buff = new byte[1024];
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		int len;
		while ((len = stream.read(buff)) != -1) {
			baos.write(buff, 0, len);
		}
		baos.close();
		this.soundData = baos.toByteArray();
		
		open(stream.getFormat());
	}
	
	private void open(AudioFormat format) throws LineUnavailableException {
		// ANZX`FbN
		checkAccess();
		
		// tH[}bg`FbN
		WCESourceDataLine.checkFormat(format);

		// I[v
		super.open();

		// tH[}bgݒ
		setFormat(format);

		// ʒu
		setFramePosition(0);
		setLoopPoints(0, -1);
		
		// [v񐔂
		loop(0);

		// o̓foCXI[v
		this.sourceLine.open(format);


		// Cxgʒm
		notifyLineEvent(new LineEvent(this,
							 LineEvent.Type.OPEN,
							 AudioSystem.NOT_SPECIFIED));
	}
	
	/**
	 * ̃Nbv
	 */
	public void close() {
		if (! isOpen()) {
			return;
		}
		stop();
		super.close();
		
		// ubNĂXbhɒʒm
		synchronized (this) {
			// Đ
			this.playbackCompleted = true;
			notifyAll();
		}

		if (this.playbackThread != null) {
			// XbhÎ҂
			try {
				this.playbackThread.join();
			} catch (InterruptedException ie) {
				ie.printStackTrace();
			}
		}
		this.playbackThread = null;
		
		// TEhf[^NA
		this.soundData = null;
		
		// o͐N[Y
		this.sourceLine.close();
		
		// Cxgʒm
		notifyLineEvent(new LineEvent(this,
							 LineEvent.Type.CLOSE,
							 AudioSystem.NOT_SPECIFIED));
	}
	
	public Control[] getControls() {
		// ToDo: 
		return new Control[0];
	}
	
	public boolean isControlSupported(Control.Type control) {
		// ToDo: 
		return false;
	}
	
	public Control getControl(Control.Type control) {
		// ToDo: 
		throw new IllegalArgumentException("Not supported:" + control);
	}
	
	public void drain() {
		if (! isOpen()) {
			return;
		}
		synchronized (this) {
			while (! this.playbackCompleted) {
				try {
					wait();
				} catch (InterruptedException ie) {
					ie.printStackTrace();
				}
			}
		}
		this.sourceLine.drain();
	}
	
	public void flush() {
		if (! isOpen()) {
			return;
		}
		this.sourceLine.flush();
	}
	
	public synchronized void start() {
		if (isRunning()) {
			return;
		}
		super.start();
		this.sourceLine.start();
		
		// ĐXbhJn
		if (this.playbackThread == null) {
			this.playbackThread = new Thread() {
				public void run() {
					playbackLoop();
				}
			};
			this.playbackThread.start();
		}
		
		// ĐtONA
		this.playbackCompleted = false;
		
		// startƂĐXbhɒʒm
		notifyAll();
	}
	
	/**
	 * Clip~
	 */
	public synchronized void stop() {
		if (! isRunning()) {
			return;
		}
		super.stop();
		this.sourceLine.stop();
		
		if (this.playbackThread != null) {
			notifyAll();
		}
	}
	
	public boolean isActive() {
		return this.sourceLine.isActive();
	}
	
	public int getBufferSize() {
		return this.sourceLine.getBufferSize();
	}
	
	public int available() {
		// 0Ԃ
		return 0;
	}
	
	public int getFramePosition() {
		// ToDo: 
		return AudioSystem.NOT_SPECIFIED;
	}
	
	public long getLongFramePosition() {
		// ToDo: 
		return AudioSystem.NOT_SPECIFIED;
	}

	public long getMicrosecondPosition() {
		// ToDo: 
		return AudioSystem.NOT_SPECIFIED;
	}
	
	public float getLevel() {
		// ToDo: 
		return AudioSystem.NOT_SPECIFIED;
	}
	
	/**
	 * fBÃt[Ԃ
	 */
	public int getFrameLength() {
		if (! isOpen()) {
			return 0;
		}
		return this.soundData.length / getFormat().getFrameSize();
	}
	
	/**
	 * fBA̒}CNbPʂŕԂB
	 */
	public long getMicrosecondLength() {
		if (! isOpen()) {
			return 0L;
		}
		return ((long) getFrameLength() * 1000000L) / (long) getFormat().getFrameRate();
	}
	
	/**
	 * ĐJnʒuݒ肷B
	 */
	public void setFramePosition(int frames) {
		if (frames > getFrameLength()) {
			throw new IllegalArgumentException("frames=" + frames);
		}
		this.startFrame = frames;
	}
	
	/**
	 * ĐJnʒu}CNbPʂŐݒ肷B
	 */
	public void setMicrosecondPosition(long microseconds) {
		if (getMicrosecondLength() < microseconds) {
			throw new IllegalArgumentException("microseconds=" + microseconds);
		}
		
		this.startFrame =  (int) (((long) getFormat().getFrameRate() * microseconds) / 1000000L);
	}
	
	/**
	 * [v̊Jn^Iʒuݒ肷
	 */
	public void setLoopPoints(int start, int end) {
		if (! isOpen()) {
			return;
		}
		
		if ((start < 0)
				|| (end != -1 && end >= getFrameLength())
				|| (end != -1 && (start > end))) {
			throw new IllegalArgumentException("start=" + start + " end=" + end);
		}
		
		synchronized (this) {
			this.loopStart = start;
			this.loopEnd = (end == -1) ? (getFrameLength() - 1) : end;
		}
	}
	
	/**
	 * [vĐJn
	 */
	public synchronized void loop(int count) {
		this.loopCount = count;
	}
	
	/**
	 * ANZX`FbNs
	 */
	private void checkAccess() throws SecurityException {
		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
			// "play" `FbN
			sm.checkPermission(new AudioPermission("play"));
		}
	}
	
	/**
	 * Đ[v
	 */
	private void playbackLoop() {
		this.playbackCompleted = false;
		while (isOpen()) {
			synchronized (this) {
				while (isOpen() && (! isRunning() || this.playbackCompleted)) {
					// stopĂԂ܂͍ĐĂꍇ
					// 炩̃Cxg܂őҋ@
					try {
						wait();
					} catch (InterruptedException ie) {
						ie.printStackTrace();
						break;
					}
				}
				if (! isOpen()) {
					// CN[Yꂽ
					break;
				}
			}

			// xwrite()钷vZ
			final int writeUnitLength = (int) getFormat().getFrameRate();
			final int frameSize = getFormat().getFrameSize();
			
			// JnʒuƏIʒuvZ
			int nextFrame = this.startFrame;
			int endFrame = (this.loopCount != 0) ? this.loopEnd : getFrameLength() - 1;
			

			// Đ[vJn
			while (isRunning() && (nextFrame < endFrame)) {
				// ރf[^vZ
				int framesWrite = ((nextFrame + writeUnitLength) > endFrame)
							? endFrame - nextFrame
							: writeUnitLength;
				int offset = nextFrame * frameSize;
				int bytesWrite = framesWrite * frameSize;

				// o͏
				sourceLine.write(this.soundData,
						 		 offset,
								 bytesWrite);
								 
				// ̃t[ʒuvZ
				synchronized (this) {
					endFrame = (this.loopCount != 0) ? loopEnd : (getFrameLength() - 1);
					if ((this.loopCount != 0) && (nextFrame + framesWrite) >= endFrame) {
						// [vꍇ́AloopStartɖ߂
						nextFrame = this.loopStart;
						if (this.loopCount != Clip.LOOP_CONTINUOUSLY) {
							// [vJEgZ
							this.loopCount--;
						}
					} else {
						// [vʒuɓBĂȂꍇ
						nextFrame += framesWrite;
					}
				}
			}
			synchronized (this) {
				// ĐƂʒm
				this.playbackCompleted = true;
				notifyAll();
			}
		}
		
	}
}
