/*
 * listheap.c
 *      Functions to list memory blocks on heap area.
 *  1993/08     V.1.0   by kmatsui
 *  1996/09     V.2.0   by kmatsui
 *      Updated in accordance with memory.c V.2.0.
 *      Implemented list_heap() for BSD_MALLOC, DB_MALLOC, DMALLOC
 *          and BORLANDC.
 *  1997/03     V.2.1   by kmatsui
 *      Implemented calling-count information.
 *  1998/05     V.2.2   by kmatsui
 *      Added implementation for Win32 and phk-malloc.
 *  1998/11     V.2.3   by kmatsui
 *      Enabled to specify the output filename by an environment variable.
 *      Made WIN32 version always output to "memlist" unless specified
 *          otherwise by an environment variable.
 *      Added list_heap() for DLMALLOC.
 *  2003/06     V.2.4   by kmatsui
 *      Fixed the bug of output-file-setting on WIN32.
 *  2003/09     V.2.5   by kmatsui
 *      Re-fixed the problem of output-file-setting on WIN32.
 *      Added implementation for Visual C++.
 *
 *  One of the following 8 macros may be defined on command line --
 *  KMMALLOC    :   for the kmatsui's version of memory allocating routines.
 *  BSD_MALLOC  :   for the malloc() of 4.[34]BSD (and GNU C / libc)
 *                  compiled with -DMSTATS (maybe also -DDEBUG -DRCHECK).
 *  PHK_MALLOC  :   for the malloc() of Poul-Henning Kamp (and FreeBSD)
 *                  compiled with -DMALLOC_EXTRA_SANITY and -DMALLOC_STATS.
 *  DLMALLOC    :   for the malloc() of Doug Lea.
 *  DB_MALLOC   :   for the debugging malloc() by Conor P. Cahill.
 *  DMALLOC     :   for the debug malloc() by Gray Watson.
 *  BORLANDC    :   for the malloc() of Borland C.
 *  VISUALC     :   for the malloc() of Visual C++.
 *
 *  LARGE       :   large data memory model of MS-DOS.
 */

#include    <stdio.h>
#include    "errno.h"

/* kmallocgp */
#define	KMMALLOC		1
#undef	DMALLOC
#define NO_STDOUT		1

#if     KMMALLOC    /* for kmatsui's version of malloc() & others   */

/*
 * list_heap() is called from free() and malloc() on _MEM_DEBUG version.
 * It maybe also called from user program.
 * print_memerror(), memerr_list() or merr_list_exit() are used usually to
 * set it's pointer to 'merrfunc' which is executed by malloc() or free()
 * on heap error.
 */

#include    <string.h>
#include    "yalloc.h"

int     list_heap();

FILE    *memfp = NULL;          /* Output stream    */

static int      recursion;      /* Count of recursive calls of the function */
static int      reopened;       /* 0: before the first output, 1: after     */

#if     _MEM_DEBUG && _GETENV
extern FILE     *set_memfp( void);
#endif

static char     *merr[] = {
    "Illegal argument to free()",
    "Corrupted free block",
    "Corrupted allocated block",
    "Free block written on",
    "Trailing area written on",
};

/* Print message of heap error to 'memfp'   */
void    print_memerror( error)
    int     error;  /* Code of heap error detected by malloc() or free()    */
{
    int     num;

    if (recursion++) {
        recursion--;
        return;
    }
    if (! reopened) {
        reopened = 1;
#if     _MEM_DEBUG && _GETENV
        memfp = set_memfp();
#endif
        if (memfp == NULL)
#if     NO_STDOUT
            memfp = fopen( "memlist", "a");
#else
            memfp = stdout;
#endif
    }

    switch (error) {
    case EFREEP     :   num = 0;    break;
    case EFREEBLK   :   num = 1;    break;
    case EALLOCBLK  :   num = 2;    break;
    case EFREEWRT   :   num = 3;    break;
    case ETRAILWRT  :   num = 4;    break;
    }
    fprintf( memfp, "Heap error: %s\n", merr[ num]);
    recursion--;
}

void    memerr_list( err)
    int     err;
{
    print_memerror( err);
    list_heap( 1);
}

void    merr_list_exit( err)
    int     err;
{
    memerr_list( err);
    exit( err);
}

/*
 * List blocks on heap.
 * On success return 0, on error return the values defined in <errno.h>
 * without setting 'errno', on no heap return -1.
 */
int     list_heap( output)
    int     output;                         /* Output or not        */
{
extern char     *_heapbase;
#if     __TURBOC__ && ! __WIN32__
extern _ptr_t   _brklvl;
#else
extern char     *_brklvl;
#endif
extern _Header  _base;
static char     *format = "  %s";

#if     _MEM_DEBUG
static char
    *countinfo = "This is the %u'th call of malloc() or such.\n",
    *where = " %6u  %s:%d";
#endif

#if     defined _LARGE
static char
    *heaparea = "_heapbase:%lx, _brklvl:%lx, basep:%lx\n",
    *freeinfo = "%8lx  %8lx  %8lx",
    *allocinfo = "%8lx  %8lx  (in use)",
    *totalsize = "Total size of %d %s blocks is 0x%lx bytes.\n";
#else
static char
    *heaparea = "_heapbase:%x, _brklvl:%x, basep:%x\n",
    *freeinfo = "%8x  %8x  %8x",
    *allocinfo = "%8x  %8x  (in use)",
    *totalsize = "Total size of %d %s blocks is 0x%x bytes.\n";
#endif

    _ptr_t  freesize, allocsize, fsize;
    int     error, fcount, acount;
    char    _HUGE *heapbase;
    _Header _HUGE *freep, _HUGE *next_free, _HUGE *allocp, _HUGE *end_alloc,
            _HUGE *basep;

    if ((char *)_brklvl == _heapbase)       /* No heap yet          */
        return -1;              /* Do not use stdio which may use malloc()  */
    if (recursion++) {  /* Recursively called from stdio library function   */
        recursion--;
        return 0;       /* Prevent recursive call of stdio function */
    }

    if (! reopened && output) {
        reopened = 1;
#if _MEM_DEBUG && _GETENV
        /*
         * set_memfp() may calls fopen() whose implementation may calls
         * malloc().
         */
        memfp = set_memfp();
#endif
        if (memfp == NULL)
#if     NO_STDOUT
    /*
     * This macro should be defined by makefile for Windows (not console)
     * program.
     */
            memfp = fopen( "memlist", "a");
#else
            memfp = stdout;
#endif
    }

    if (output)
        freesize = allocsize = fcount = acount = 0;
    error = 0;
    heapbase = _heapbase;
    freep = basep = &_base;
    if (output) {
        /*
         * fputc() may call malloc() to initialize memfp.  If list_heap() is
         * called from within malloc(), this may causes recursion.  This
         * recursive call of malloc(), however, does not cause problem except
         * unifying of _memcount, _memfile and _memline of the last two calls.
         * fputc() is necessary to initialize memfp and possibly update
         * _brklvl prior to use of its value in fprintf().
         */
        fputc( '\n', memfp);
        fprintf( memfp, heaparea, heapbase, _brklvl, basep);
#if     _MEM_DEBUG
        /* Called from malloc(), free(), realloc() or calloc()  */
        fprintf( memfp, countinfo, _memcount);
#endif
        fputs(
        "        << Blocks on heap >>\n  header      size      next",
                memfp);
#if     _MEM_DEBUG
        fputs( "  count  filename:linenum", memfp);
#endif
        fputc( '\n', memfp);
    }

    do {                                    /* Until to wrap around */
        next_free = freep->_next;
        fsize = freep->_size;
        allocp = (_Header *)((char _HUGE *)freep + fsize);
        if (freep == basep)
            allocp = (_Header *)heapbase;   /* From the top of heap */
        if (next_free == basep)
            end_alloc = (_Header *)_brklvl; /* To the end of heap   */
        else
            end_alloc = next_free;
        if (freep != basep) {
            if (output) {
                fprintf( memfp, freeinfo, freep, fsize, next_free);
#if     _MEM_DEBUG
                fprintf( memfp, where,
                        freep->_count, freep->_file, freep->_line);
#endif
            }
            if ((_ptr_t)freep & _ODD || fsize & _ODD
                    || (char _HUGE *)_brklvl < (char _HUGE *)freep
                    || (char _HUGE *)freep < heapbase
                    || (char _HUGE *)end_alloc < (char _HUGE *)freep + fsize
#if     defined _LARGE
                    || 0xA00000L <= fsize
#endif
                    ) {
                error = EFREEBLK;   /* Corrupted free block, can't continue */
                if (output) {
                    fprintf( memfp, format, merr[ 1]);
                    fputc( '\n', memfp);
                }
                recursion--;
                return error;
            }
            if (
#if     ! _MALLOC0_NULL
                freep->_size > _CELLSIZE &&
#endif
                    (*((char _HUGE *)freep + _CELLSIZE) & 0xFF) != _FMAGIC) {
                error = EFREEWRT;   /* Free area has been written on    */
                if (output)
                    fprintf( memfp, format, merr[ 3]);
            }
            if (output) {
                fcount++;
                freesize += fsize;
                fputc( '\n', memfp);
            }
        }

        while (allocp < end_alloc) {        /* Blocks between the   */
            _ptr_t  asize;                  /*      2 freed blocks  */
            _Header _HUGE   *next_alloc;

            asize = allocp->_size;
            next_alloc = (_Header *)((char _HUGE *)allocp + asize);
            if (output) {
                fprintf( memfp, allocinfo, allocp, asize);
#if     _MEM_DEBUG
                fprintf( memfp, where,
                        allocp->_count, allocp->_file, allocp->_line);
#endif
            }
            if (! _ALLOCATED( allocp->_next) || asize & _ODD
                    || end_alloc < next_alloc || next_alloc <= allocp
#if     defined _LARGE
                    || asize >= ~(size_t)0
#endif
                    ) {
                error = EALLOCBLK;
                if (output) {
                    fprintf( memfp, format, merr[ 2]);
                    fputc( '\n', memfp);
                }
                break;
            }
            if (allocp->_next != (_Header _HUGE *)_PMAX
                    && (*((char _HUGE *)next_alloc
                        - ~(_ptr_t)allocp->_next) & 0xFF) != _FMAGIC) {
                error = ETRAILWRT;
                if (output)
                    fprintf( memfp, format, merr[ 4]);
            }
            allocp = next_alloc;
            if (output) {
                acount++;
                allocsize += asize;
                fputc( '\n', memfp);
            }
        }
        if (error && !output)
            break;
    } while ((freep = next_free) != basep);

    if (output) {
        fprintf( memfp, totalsize, acount, "allocated", allocsize);
        fprintf( memfp, totalsize, fcount, "free", freesize);
    }
    recursion--;
    return error;
}

#elif   BSD_MALLOC

int     list_heap( char *mes)
{
    mstats( mes);
    return 0;
}

#elif   PHK_MALLOC

int     list_heap( FILE *fp)
{
    malloc_dump( *fp);
    return 0;
}

#elif   DLMALLOC

int     list_heap( FILE *fp)
{
    struct mallinfo     mallinfo( void);
    void                malloc_stats( void);
    struct mallinfo {
        int arena;      /* total space allocated from system */
        int ordblks;    /* number of non-inuse chunks */
        int smblks;     /* unused -- always zero */
        int hblks;      /* number of mmapped regions */
        int hblkhd;     /* total space in mmapped regions */
        int usmblks;    /* unused -- always zero */
        int fsmblks;    /* unused -- always zero */
        int uordblks;   /* total allocated space */
        int fordblks;   /* total non-inuse space */
        int keepcost;   /* top-most, releasable (via malloc_trim) space */
    } info;

    info = mallinfo();
    fprintf( fp, "arena:%#x, ordblks:%#x, hblks:%#x, hblkhd:%#x,\n"
            "uordblks:%#x, fordblks:%#x, keepcost:%#x\n",
            info.arena, info.ordblks, info.hblks, info.hblkhd,
            info.uordblks, info.fordblks, info.keepcost);
    malloc_stats();
    return 0;
}

#elif   DB_MALLOC

#include    "malloc.h"

int     list_heap( FILE *fp)
{
    malloc_dump( fileno( fp));
    return 0;
}

#elif   DMALLOC

#include    "dmalloc.h"

int     list_heap( FILE *fp)
{
    dmalloc_log_heap_map();
        /* Output to the log-file specified by environment-variable */
    return 1;
}

#elif   LEAK

list_heap( fp)
    FILE    *fp;
{
    extern void     leak_dump( void);
    leak_dump();
    return 0;
}

#elif   BORLANDC

#include    <alloc.h>

int     list_heap( FILE *fp)
{
#if     ! LARGE
#define _heapbase   __heapbase
#define _brklvl     __brklvl
#endif
#if     ! __WIN32__
    extern char     *_heapbase, *_brklvl;
#endif
    static char     *stat[] = { "_BADVALUE", "_BADNODE", "_HEAPCORRUPT",
            "", "_HEAPEMPTY", "_HEAPOK", "_FREEENTRY", "_USEDENTRY",
            "_HEAPEND", };
    struct heapinfo     hi = { NULL, };
    int     acount, fcount;
    long    allocsize, freesize;

    acount = fcount = allocsize = freesize = 0;
#if     ! __WIN32__
    fprintf( fp, "heapcheck:%s, _heapbase:%p, _brklvl:%p\n",
            stat[ heapcheck() - _BADVALUE], _heapbase, _brklvl);
#endif
    fputs( "   ptr       size  use   stat\n", fp);

    while (heapwalk( &hi) != _HEAPEND) {
        int     status;

        status = heapchecknode( hi.ptr);
        fprintf( fp, "%9p %8lx  %d  %s\n",
                hi.ptr, (long) hi.size, hi.in_use, stat[ status - _BADVALUE]);
        if (status < _HEAPOK)
            return status;
        if (hi.in_use) {
            allocsize += hi.size;
            acount++;
        } else {
            freesize += hi.size;
            fcount++;
        }
    }
    fprintf( fp, "Total %ld bytes size of %d free blocks.\n",
            freesize, fcount);
    fprintf( fp, "Total %ld bytes size of %d blocks in use.\n",
            allocsize, acount);
    return _HEAPEND;
}

#elif   VISUALC

#include    <malloc.h>

int     list_heap( FILE * fp)
{
    static char     *stat[] = { "_HEAPBADPTR", "_HEAPEND", "_HEAPBADNODE",
            "_HEAPBADBEGIN", "_HEAPOK", "_HEAPEMPTY", };
    struct _heapinfo    hi = { NULL, };
    int     status;
    int     acount, fcount;
    long    allocsize, freesize;

    acount = fcount = allocsize = freesize = 0;
    fputs( "\n   ptr        size   use\n", fp);

    while ((status = _heapwalk( &hi)) != _HEAPEND) {
        fprintf( fp, "%9p %8x  %s\n",
                hi._pentry, hi._size,
                status == _HEAPOK ? hi._useflag ? "used" : "free" :
                stat[ status - _HEAPBADPTR]);
        if (status != _HEAPOK)
            return status;
        if (hi._useflag) {
            allocsize += hi._size;
            acount++;
        } else {
            freesize += hi._size;
            fcount++;
        }
    }
    fprintf( fp, "Total %ld bytes size of %d free blocks (net).\n",
            freesize, fcount);
    fprintf( fp, "Total %ld bytes size of %d blocks in use (net).\n",
            allocsize, acount);
    return _HEAPEND;
}

#endif
