/******************************************************************************/
/*                                                                            */
/* src/lib/libmk/LibMkTimer.c                                                 */
/*                                                                 2019/07/28 */
/* Copyright (C) 2018-2019 Mochi.                                             */
/*                                                                            */
/******************************************************************************/
/******************************************************************************/
/* インクルード                                                               */
/******************************************************************************/
/* 標準ヘッダ */
#include <stdint.h>

/* ライブラリヘッダ */
#include <MLib/MLib.h>

/* カーネルヘッダ */
#include <kernel/timer.h>


/******************************************************************************/
/* グローバル関数宣言                                                         */
/******************************************************************************/
/******************************************************************************/
/**
 * @brief       スリープ
 * @details     指定した時間スリープする。
 *
 * @param[in]   usec  スリープ時間(マイクロ秒)
 * @param[out]  *pErr エラー内容
 *                  - MK_ERR_NONE        エラー無し
 *                  - MK_ERR_NO_RESOURCE リソース不足
 *
 * @return      処理結果を返す。
 * @retval      MK_RET_SUCCESS 成功
 * @retval      MK_RET_FAILURE 失敗
 *
 * @attention   カーネルのtick時間よりも短いスリープ時間を設定した場合、tick時
 *              間に丸められる。
 */
/******************************************************************************/
MkRet_t LibMkTimerSleep( uint32_t usec,
                         MkErr_t  *pErr )
{
    volatile MkTimerParam_t param;

    /* パラメータ設定 */
    param.funcId = MK_TIMER_FUNCID_SLEEP;
    param.ret    = MK_RET_FAILURE;
    param.err    = MK_ERR_NONE;
    param.usec   = usec;

    /* カーネルコール */
    __asm__ __volatile__ ( "mov esi, %0\n"
                           "int %1"
                           :
                           : "a" ( &param         ),
                             "i" ( MK_TIMER_INTNO )
                           : "esi"                   );

    /* エラー内容設定 */
    MLIB_SET_IFNOT_NULL( pErr, param.err );

    return param.ret;
}


/******************************************************************************/
