//
// Utils.hpp
//
#pragma once

#include <string>
#include <tuple>
#include <boost/format.hpp>

namespace network {
    namespace Utils {
        std::string Encode(const std::string&);
        std::string Decode(const std::string&);

        std::string ByteStuffingEncode(const std::string&);
        std::string ByteStuffingDecode(const std::string&);

        std::string Base64Encode(const std::string&);
        std::string Base64Decode(const std::string&);

        std::string SnappyCompress(const std::string&);
        std::string SnappyUncompress(const std::string&);

        std::string ToHexString(const std::string&);

        bool IsPrivateAddress(const std::string&);

        constexpr bool little_endian_check(int i)
        {
            return *reinterpret_cast<char*>(&i);
        }

        constexpr bool little_endian()
        {
            return little_endian_check(1);
        }

        inline std::string ConvertEndian(const std::string& in)
        {
            if (little_endian()) {
                std::string out = in;
                std::reverse(out.begin(), out.end());
                return out;
            } else {
                return in;
            }
        }

        // Serialize
        template<class T>
        inline std::string GetSerializedValue(const T& t)
        {
            return ConvertEndian(std::string(reinterpret_cast<const char*>(&t), sizeof(t)));
        }

        template<>
        inline std::string GetSerializedValue(const std::string& t)
        {
            size_t size = t.size();
            return ConvertEndian(std::string(reinterpret_cast<const char*>(&size), sizeof(size_t))) +
                    std::string(t);
        }

        inline std::string Serialize() {
            return std::string();
        }

        template<class First, class... Rest>
        inline std::string Serialize(const First& first, const Rest&... rest)
        {
            return GetSerializedValue(first) + Serialize(rest...);
        }

        // Deserialize

        template<class First>
        inline First GetDeserializedValue(std::string& buffer)
        {
            const First value = *reinterpret_cast<const First*>(ConvertEndian(buffer.substr(0, sizeof(First))).data());
            buffer.erase(0, sizeof(First));
            return value;
        }

        template<>
        inline std::string GetDeserializedValue<std::string>(std::string& buffer)
        {
            size_t size = *reinterpret_cast<const size_t*>(ConvertEndian(buffer.substr(0, sizeof(size_t))).data());
            buffer.erase(0, sizeof(size_t));
            std::string data(buffer.data(), size);
            buffer.erase(0, size);
            return data;
        }

        template<class First>
        inline void GetDeserializedItems(std::string& buffer, First* first)
        {
            *first = GetDeserializedValue<First>(buffer);
        }

        template<class First, class Second, class... Rest>
        inline void GetDeserializedItems(std::string& buffer, First* first, Second* second, Rest*... rest)
        {
            GetDeserializedItems<First>(buffer, first);
            GetDeserializedItems<Second, Rest...>(buffer, second, rest...);
        }

        template<class... Types>
        inline size_t Deserialize(const std::string& data, Types... types)
        {
            size_t original_size = data.size();
            std::string buffer(data);
            GetDeserializedItems(buffer, types...);
            return original_size - buffer.size();
        }

        constexpr static char DELIMITOR = 0x7e;
    }
}
