//
// Card.hpp
//

#pragma once

#include <string>
#include <vector>
#include <map>
#include <set>
#include <memory>
#include <v8.h>
#include <boost/timer.hpp>
#include "ui/include.hpp"
#include "ManagerAccessor.hpp"
#include "InputManager.hpp"
#include "ScriptEnvironment.hpp"

class CardManager;
typedef std::weak_ptr<CardManager> CardManagerWeakPtr;

class Player;
typedef std::shared_ptr<Player> PlayerPtr;

struct TimerEvent {
        v8::Persistent<v8::Function> function;
        std::vector<v8::Persistent<v8::Value>> args;
        int delay;
        bool interval;
        boost::timer timer;
};
typedef std::shared_ptr<TimerEvent> TimerEventPtr;

class Card {
    public:
        Card(const ManagerAccessorPtr& manager_accessor,
             std::string source_folder,
             std::string name,
             std::string author,
             std::string caption,
             std::string icon,
             std::vector<std::string> scripts,
             bool group,
             bool autorun);

        ~Card();

        void Run();
        void Execute(const std::string& script, const std::string& filename = "",
                     const V8ValueCallBack& callback = V8ValueCallBack());

        void Update();
        void Draw();
        void ProcessInput(InputManager* input);

        bool HasInputEvent();
        void onEnter(const std::string& text);

        std::string author() const;
        bool autorun() const;
        std::string caption() const;
        bool group() const;
        std::string icon() const;
        std::string name() const;
        std::vector<std::string> scripts() const;
        std::string source_folder() const;
        int icon_handle();
        bool running() const;
        bool close_flag() const;
        std::string input_message() const;

        int focus_index() const;

        static void set_max_local_storage_size(int size);

    private:
        static Card* GetPointerFromObject(const Handle<Object>& object);
        v8::Handle<v8::ObjectTemplate> GenerateGlobalTemplate();
        void SetFunctions();

        void LoadStorage();
        void SaveStorage();

        std::string EscapeString(const std::string& str);
        void CheckFunctionKey(int keynum, const std::string& name, const InputManager& input);

    public:
        void OnReceiveJSON(const std::string& info_json, const std::string& msg_json);
        void OnLogin(const PlayerPtr& player);
        void OnLogout(const PlayerPtr& player);
        void OnClose();

    private:
        ManagerAccessorPtr manager_accessor_;

        std::string source_folder_;
        std::string name_;
        std::string author_;
        std::string caption_;
        std::string icon_;
        std::vector<std::string> scripts_;
        bool group_;
        bool autorun_;

        ScriptEnvironment script_;

        v8::Persistent<v8::Context> context_;
        std::map<int, TimerEventPtr> timer_events_;
        int event_id_;

        Persistent<Object> ui_board_obj_;
        Persistent<Object> local_storage_;
        Persistent<Function> network_on_receive_json_;

        struct {
            Persistent<Function> on_login_;
            Persistent<Function> on_logout_;
        } player;

        struct {
            Persistent<Function> on_enter_;
            bool enable_;
            std::string tabname_;
            std::string message_;
        } inputbox;

        int icon_handle_;

        bool running_;
        bool close_flag_;

        static bool force_gc_flag;

    private:
        static char STORAGE_DIR[];
        static char SCRIPT_PATH[];

        static int max_local_storage_size;
        static std::set<Card*> ptr_set;
};

typedef std::shared_ptr<Card> CardPtr;
typedef std::weak_ptr<Card> CardWeakPtr;

