//
// Session.hpp
//

#pragma once

#include <boost/asio.hpp>
#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <boost/timer.hpp>
#include <stdint.h>
#include <string>
#include <queue>
#include <memory>
#include "Command.hpp"
#include "Encrypter.hpp"

namespace network {

    using boost::asio::ip::tcp;
    using boost::asio::ip::udp;
    typedef std::function<void(Command)> CallbackFunc;
    typedef std::shared_ptr<CallbackFunc> CallbackFuncPtr;
    typedef long UserID;

    class Session : public boost::enable_shared_from_this<Session> {
        public:
            Session(boost::asio::io_service& io_service, tcp::resolver::iterator endpoint_iterator);
            Session(boost::asio::io_service& io_service);
            virtual ~Session();

            virtual void Start() = 0;
            void Close();
            void Send(const Command&);
            void SyncSend(const Command&);
            void UDPSend(const Command&);

            void EnableEncryption();

            void LogIn(UserID id);
            void LogOut();

            double GetReadByteAverage() const;
            double GetWriteByteAverage() const;

            tcp::socket& socket();
            Encrypter& encrypter();

            void set_on_receive(CallbackFuncPtr);

            UserID id() const;
            bool login() const;
            bool online() const;

            std::string global_ip() const;
            uint16_t udp_port() const;
            void set_global_ip(const std::string& global_ip);
            void set_udp_port(uint16_t udp_port);

            int account_revision() const;
            void set_account_revision(int revision);

            int serialized_byte_sum() const;
            int compressed_byte_sum() const;

            bool operator==(const Session&);
            bool operator!=(const Session&);

        protected:
            void UpdateReadByteAverage();
            void UpdateWriteByteAverage();

            std::string Serialize(const Command& command);
            Command Deserialize(const std::string& msg);

            void Receive(const boost::system::error_code& error);
            void DoWrite(const std::string);
            void Write(const boost::system::error_code& error);
            void Fetch(const std::string&);

            void FatalError();

        protected:
            // ソケット
            boost::asio::io_service& io_service_;
            tcp::socket socket_;

            // 暗号化通信
            Encrypter encrypter_;
            bool encryption_;

            // 送受信のためのバッファ
            boost::asio::streambuf receive_buf_;
            std::queue<std::string> send_queue_;

            CallbackFuncPtr on_receive_;

            // UDPパケット送信の宛先
            std::string global_ip_;
            uint16_t udp_port_;

            bool online_;
            bool login_;
            int account_revision_;

            time_t read_start_time_, write_start_time_;
            int read_byte_sum_, write_byte_sum_;
            int serialized_byte_sum_, compressed_byte_sum_;

            UserID id_;

        private:
            constexpr static unsigned int BYTE_AVERAGE_REFRESH_SECONDS = 10;
            constexpr static unsigned int COMPRESSED_FLAG = 0x00010000;
            constexpr static unsigned int COMPRESS_MIN_LENGTH = 100;
    };

    typedef boost::weak_ptr<Session> SessionWeakPtr;
    typedef boost::shared_ptr<Session> SessionPtr;

}
