//
// Encrypter.cpp
//

#include <openssl/rand.h>
#include <boost/format.hpp>
#include "Encrypter.hpp"
#include "Utils.hpp"

namespace network {

Encrypter::Encrypter()
{
    RAND_bytes(common_key_, sizeof(common_key_));
    RAND_bytes(common_key_iv_, sizeof(common_key_iv_));

    EVP_CIPHER_CTX_init(&ctx_encrypt_);
    EVP_CIPHER_CTX_init(&ctx_decrypt_);
    EVP_EncryptInit_ex(&ctx_encrypt_, EVP_aes_128_cbc(), NULL, common_key_, common_key_iv_);
    EVP_DecryptInit_ex(&ctx_decrypt_, EVP_aes_128_cbc(), NULL, common_key_, common_key_iv_);

    rsa_key_ = RSA_generate_key(RSA_KEY_LENGTH * 8, RSA_F4, NULL, NULL);
}

Encrypter::~Encrypter()
{
    EVP_CIPHER_CTX_cleanup(&ctx_encrypt_);
    EVP_CIPHER_CTX_cleanup(&ctx_decrypt_);
    RSA_free(rsa_key_);
}

std::string Encrypter::Encrypt(const std::string& in)
{
     unsigned char outbuf[in.size()];
     int outlen, tmplen;

     EVP_EncryptInit_ex(&ctx_encrypt_, NULL, NULL, NULL, NULL);
     if(!EVP_EncryptUpdate(&ctx_encrypt_, outbuf, &outlen, (unsigned char*)in.data(), in.size()) ||
             !EVP_EncryptFinal_ex(&ctx_encrypt_, outbuf + outlen, &tmplen)) {

         return std::string();
     }

     outlen += tmplen;

     return std::string((const char*)outbuf, outlen);
}

std::string Encrypter::Decrypt(const std::string& in)
{
     unsigned char outbuf[in.size()];
     int outlen, tmplen;

     EVP_DecryptInit_ex(&ctx_decrypt_, NULL, NULL, NULL, NULL);
     if(!EVP_DecryptUpdate(&ctx_decrypt_, outbuf, &outlen, (unsigned char*)in.data(), in.size()) ||
             !EVP_DecryptFinal_ex(&ctx_decrypt_, outbuf + outlen, &tmplen)) {

         return std::string();
     }

     outlen += tmplen;

     return std::string((const char*)outbuf, outlen);
}

std::string Encrypter::GetPublicKey()
{
    unsigned char key[RSA_KEY_LENGTH];
    BN_bn2bin(rsa_key_->n, key);
    return std::string((const char*)key, sizeof(key));
}

void Encrypter::SetPublicKey(const std::string& in)
{
    BN_bin2bn((unsigned char*)in.data(), in.size(), rsa_key_->n);
}

std::string Encrypter::GetPrivateKey()
{
    unsigned char key[RSA_KEY_LENGTH];
    BN_bn2bin(rsa_key_->d, key);
    return std::string((const char*)key, sizeof(key));
}

void Encrypter::SetPrivateKey(const std::string& in)
{
    BN_bin2bn((unsigned char*)in.data(), in.size(), rsa_key_->d);
}

void Encrypter::SetPairKey(const std::string& pub, const std::string& pri)
{
    BN_bin2bn((unsigned char*)pub.data(), pub.size(), rsa_key_->n);
    BN_bin2bn((unsigned char*)pri.data(), pri.size(), rsa_key_->d);
}

std::string Encrypter::GetCryptedCommonKey()
{
    return PublicEncrypt(GetCommonKey());
}

void Encrypter::SetCryptedCommonKey(const std::string& in)
{
    std::string key = PublicDecrypt(in);
    key.copy((char*)common_key_, sizeof(common_key_));
    key.copy((char*)common_key_iv_, sizeof(common_key_iv_), sizeof(common_key_));

    EVP_EncryptInit_ex(&ctx_encrypt_, EVP_aes_128_cbc(), NULL, common_key_, common_key_iv_);
    EVP_DecryptInit_ex(&ctx_decrypt_, EVP_aes_128_cbc(), NULL, common_key_, common_key_iv_);
}

std::string Encrypter::PublicEncrypt(const std::string& in)
{
    int outlen = RSA_size(rsa_key_);
    unsigned char outbuf[outlen];

    RSA_public_encrypt(in.size(), (unsigned char *)in.data(),
            outbuf, rsa_key_, RSA_PKCS1_OAEP_PADDING);

    return std::string((const char*)outbuf, outlen);
}

std::string Encrypter::PublicDecrypt(const std::string& in)
{
    int outlen = RSA_size(rsa_key_);
    unsigned char outbuf[outlen];

    RSA_private_decrypt(in.size(), (unsigned char *)in.data(),
            outbuf, rsa_key_, RSA_PKCS1_OAEP_PADDING);

    return std::string((const char*)outbuf, outlen);
}

std::string Encrypter::GetPublicKeyFingerPrint()
{
    return GetHash(GetPublicKey());
}

std::string Encrypter::GetHash(const std::string& in)
{
    int outlen = SHA_LENGTH;
    unsigned char outbuf[outlen];

    static const std::string key = Utils::Base64Decode(SHA_HMAC_KEY).data();
    HMAC(EVP_sha512(), key.data(), key.size(), (unsigned char*)in.data(), in.size(), outbuf, NULL);

    return std::string((char*)outbuf, outlen);
}

std::string Encrypter::GetTrip(const std::string& in)
{
    static const unsigned char trip_chars[] =
            "abcdefghijklmnopqrstuvwxyz"
            "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
            ".:;@#$%&_()=*{}~+-!?[]()^/";

    int pattern_size = sizeof(trip_chars) - 1;

    auto buffer = GetTripHash(in).substr(0, TRIP_LENGTH);
    std::string out;

    for (const unsigned char & c : buffer) {
        out += trip_chars[c % pattern_size];
    }

    return out;
}

std::string Encrypter::GetTripHash(const std::string& in)
{
    int outlen = SHA_LENGTH;
    unsigned char outbuf[outlen];

    static const std::string key = Utils::Base64Decode(TRIP_SHA_HMAC_KEY).data();
    HMAC(EVP_sha512(), key.data(), key.size(), (unsigned char*)in.data(), in.size(), outbuf, NULL);

    return std::string((char*)outbuf, outlen);
}

bool Encrypter::CheckKeyPair()
{
    // 復号化できるかチェック
    std::string test_data("test");
    bool result = (PublicDecrypt(PublicEncrypt(test_data)).substr(0, 4) == test_data);
    return result;
}

std::string Encrypter::GetCommonKey()
{
    return std::string((const char*)common_key_, sizeof(common_key_)) +
            std::string((const char*)common_key_iv_, sizeof(common_key_iv_));
}

}

