//
// Downloader.cpp
//

#include "Downloader.hpp"
#include <boost/regex.hpp>
#include <boost/filesystem.hpp>
#include <boost/format.hpp>
#include <sstream>
#include "../Logger.hpp"

Downloader::Downloader(const std::string& url,
        unsigned int max_length) :
resolver_(io_service_),
socket_(io_service_),
max_length_(max_length),
status_(STANDBY)
{
    boost::regex reg("http://([\\w\\.]+)(/[\\w\\./]+)");
    boost::smatch result;

    // HTTPアドレスかどうかチェック
    if (boost::regex_match(url, result, reg)) {
        server_ = result.str(1);
        path_ = result.str(2);
        Logger::Info(server_);
        Logger::Info(path_);

    } else {
        Logger::Error("Invalid URL");
        status_ = FAILED;
    }
}

void Downloader::SaveToFile(const std::string& filename)
{
    if (status_ != DOWNLOADING) {
        status_ = DOWNLOADING;

        std::ostream request_stream(&request_);
        request_stream << "GET " << path_ << " HTTP/1.0\r\n";
        request_stream << "Host: " << server_ << "\r\n";
        request_stream << "Accept: */*\r\n";
        request_stream << "Connection: close\r\n\r\n";

        boost::asio::ip::tcp::resolver::query query(server_, "http");
        resolver_.async_resolve(query,
            boost::bind(&Downloader::handle_resolve, this,
              boost::asio::placeholders::error,
              boost::asio::placeholders::iterator));

        save_filename_ = filename;

        io_service_.run();
    } else {
        Logger::Error("Downloading already started.");
    }
}

Downloader::Status Downloader::status() const
{
    return status_;
}

void Downloader::handle_resolve(const boost::system::error_code& err,
        boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
    if (!err) {
      boost::asio::async_connect(socket_, endpoint_iterator,
          boost::bind(&Downloader::handle_connect, this,
            boost::asio::placeholders::error));
    } else {
        Logger::Error(err.message());
        status_ = FAILED;
    }
}

void Downloader::handle_connect(const boost::system::error_code& err)
{
    if (!err) {
      // The connection was successful. Send the request.
      boost::asio::async_write(socket_, request_,
          boost::bind(&Downloader::handle_write_request, this,
            boost::asio::placeholders::error));
    } else {
        Logger::Error(err.message());
        status_ = FAILED;
    }
}

void Downloader::handle_write_request(const boost::system::error_code& err)
{
    if (!err) {
      boost::asio::async_read_until(socket_, response_, "\r\n",
          boost::bind(&Downloader::handle_read_status_line, this,
            boost::asio::placeholders::error));
    } else {
        Logger::Error(err.message());
        status_ = FAILED;
    }
}

void Downloader::handle_read_status_line(const boost::system::error_code& err)
{
    if (!err) {
      // Check that response is OK.
      std::istream response_stream(&response_);
      std::string http_version;
      response_stream >> http_version;
      unsigned int status_code;
      response_stream >> status_code;
      std::string status_message;
      std::getline(response_stream, status_message);
      if (!response_stream || http_version.substr(0, 5) != "HTTP/") {
          Logger::Error("Invalid response");
          return;
      }
      if (status_code != 200 && status_code != 301) {
          Logger::Info((boost::format("Response returned with status code %d") % status_code).str());
          return;
      }

      // Read the response headers, which are terminated by a blank line.
      boost::asio::async_read_until(socket_, response_, "\r\n\r\n",
          boost::bind(&Downloader::handle_read_headers, this,
            boost::asio::placeholders::error));
    } else {
        Logger::Error(err.message());
        status_ = FAILED;
    }
}

void Downloader::handle_read_headers(const boost::system::error_code& err)
{
    if (!err) {
      // Process the response headers.
      std::istream response_stream(&response_);
      std::string header;
      while (std::getline(response_stream, header) && header != "\r") {
          Logger::Debug(header);
      }

      if (save_filename_.size() > 0) {
          ofs_.open(save_filename_);
      }

      // Write whatever content we already have to output.
      auto response_size = response_.size();
      if (response_size > 0 && ofs_.is_open()) {
          ofs_ << &response_;
          response_.consume(response_size);
      }

          Logger::Info("Receive: %d", response_size);

      if (ofs_.tellp() > max_length_) {
          Logger::Error("Too large file size");
          status_ = TOOLARGE;
          return;
      }

      // Start reading remaining data until EOF.
      boost::asio::async_read(socket_, response_,
          boost::asio::transfer_at_least(1),
          boost::bind(&Downloader::handle_read_content, this,
            boost::asio::placeholders::error));
    } else {
        Logger::Error(err.message());
        status_ = FAILED;
    }
}

void Downloader::handle_read_content(const boost::system::error_code& err)
{
    if (!err) {
      // Write all of the data that has been read so far.
        auto response_size = response_.size();
        if (response_size > 0 && ofs_.is_open()) {
            ofs_ << &response_;
            response_.consume(response_size);
        }

        Logger::Info("Receive: %d", response_size);

        if (ofs_.tellp() > max_length_) {
            Logger::Error("Too large file size");
            status_ = TOOLARGE;
            return;
        }

      // Continue reading remaining data until EOF.
      boost::asio::async_read(socket_, response_,
          boost::asio::transfer_at_least(1),
          boost::bind(&Downloader::handle_read_content, this,
            boost::asio::placeholders::error));
    } else if (err == boost::asio::error::eof) {
        ofs_.close();
        status_ = COMPLETE;
    } else {
        Logger::Error(err.message());
        status_ = FAILED;
    }
}
