//
// SystemLog.cpp
//

#include "SystemLog.hpp"
#include <boost/format.hpp>
#include <boost/thread.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

SystemLog::SystemLog(const std::string& logfile) :
thread_active_(true),
db_(nullptr),
stmt_log_(nullptr)
{
    int result = sqlite3_open(logfile.c_str(), &db_);
    if (result != SQLITE_OK){
        std::cout << "ERROR : Cannot open system log." << std::endl;
        sqlite3_close(db_);
        db_ = nullptr;
    } else {

        const char query[] =
        u8R"(
                CREATE TABLE IF NOT EXISTS "systemlog" (
                    "date" TEXT,
                    "type" TEXT,
                    "message" TEXT
                );
                VACUUM;
        )"; // ヒアドキュメント

        // テーブルを作成
        char *error = nullptr;
        int result = sqlite3_exec(db_, query, nullptr, nullptr, &error);
        if (result != SQLITE_OK) {
            std::cout << error << std::endl;
        }

        const char query_log[] =
        u8R"(
            INSERT INTO `systemlog`
            (`date`, `type`, `message`)
            VALUES(?,?,?)
        )"; // ヒアドキュメント
        sqlite3_prepare(db_, query_log, strlen(query_log), &stmt_log_, nullptr);
    }

    sql_thread_ = boost::thread([this](){
        while (thread_active_) {
            LogTuple log;
            bool loaded = false;

            {
                boost::mutex::scoped_lock lk(mutex_);
                if (!wait_logs_.empty()) {
                    log = wait_logs_.front();
                    wait_logs_.pop();
                    loaded = true;
                }
            }

            if (loaded) {
                sqlite3_reset(stmt_log_);

                sqlite3_bind_text(stmt_log_, 1,  std::get<0>(log).c_str(), std::get<0>(log).size(), SQLITE_STATIC);
                sqlite3_bind_text(stmt_log_, 2,  std::get<1>(log).c_str(), std::get<1>(log).size(), SQLITE_STATIC);
                sqlite3_bind_text(stmt_log_, 3,  std::get<2>(log).c_str(), std::get<2>(log).size(), SQLITE_STATIC);

                // stmtのSQLを実行
                int result, loop=0;
                while (SQLITE_DONE != (result = sqlite3_step(stmt_log_)) && loop++<1000){}
                if (result != SQLITE_DONE) {
                    std::cout << "System Log Error : " << sqlite3_errmsg(db_) << std::endl;
                }
            }
        }
    });
}

SystemLog::~SystemLog()
{
    thread_active_ = false;
    sql_thread_.join();

    if (db_) {
        sqlite3_finalize(stmt_log_);
        stmt_log_ = nullptr;

        sqlite3_close(db_);
        db_ = nullptr;
    }
}

void SystemLog::Info(const std::string& msg)
{
    Log(LOG_INFO, msg);
}

void SystemLog::Debug(const std::string& msg)
{
    Log(LOG_DEBUG, msg);
}

void SystemLog::Warning(const std::string& msg)
{
    Log(LOG_WARNING, msg);
}

void SystemLog::Error(const std::string& msg)
{
    Log(LOG_ERROR, msg);
}

void SystemLog::Log(Type type, const std::string& msg)
{
    using boost::posix_time::ptime;
    using boost::posix_time::second_clock;

    ptime now = second_clock::local_time();
    std::string date_str = boost::posix_time::to_iso_extended_string(now);

    std::string type_str;
    switch (type) {
    case LOG_INFO:
        type_str = u8"Info";
        break;
    case LOG_DEBUG:
        type_str = u8"Debug";
        break;
    case LOG_WARNING:
        type_str = u8"Warning";
        break;
    case LOG_ERROR:
        type_str = u8"Error";
        break;
    }

    std::cout << date_str << " " << type_str << " : " << msg << std::endl;
    {
        boost::mutex::scoped_lock lk(mutex_);
        wait_logs_.push(LogTuple(date_str, type_str, msg));
    }

}
