//
// CardLibrary.cpp
//

#include "CardLibrary.hpp"
#include "../network/Encrypter.hpp"
#include "../network/Utils.hpp"
#include <iostream>
#include <string.h>
#include <boost/regex.hpp>
#include "../Logger.hpp"

CardLibrary::CardLibrary(const std::string& logfile) :
db_(nullptr),
stmt_register_(nullptr),
stmt_get_current_revision_(nullptr),
stmt_get_revision_patch_(nullptr),
stmt_apply_revision_patch_(nullptr),
stmt_get_waiting_url_(nullptr),
stmt_accept_url_(nullptr),
stmt_reject_url_(nullptr),
stmt_remove_(nullptr),
stmt_clean_removed_(nullptr),
stmt_get_size_(nullptr)
{
    Initialize(logfile);
}

void CardLibrary::Initialize(const std::string& logfile)
{
    int result = sqlite3_open(logfile.c_str(), &db_);
    if (result != SQLITE_OK){
        Logger::Error("Cannot open card data.");
        sqlite3_close(db_);
        db_ = nullptr;
    } else {
        const char query[] =
        u8R"(
                CREATE TABLE IF NOT EXISTS "card" (
                    "id" INTEGER PRIMARY KEY,
                    "user_id" INTEGER,
                    "name" TEXT,
                    "note_json" TEXT,
                    "url" TEXT,
                    "finger_print" TEXT,
                    "size" INTEGER,
                    "revision" INTEGER,
                    "status" INT DEFAULT 0
                );
                CREATE UNIQUE INDEX IF NOT EXISTS name_index ON Card(name);
                VACUUM;
                REINDEX;
        )"; // ヒアドキュメント

        // テーブルを作成
        char *error = nullptr;
        int result = sqlite3_exec(db_, query, nullptr, nullptr, &error);
        if (result != SQLITE_OK) {
            Logger::Error(error);
        }

        // stmt_register_
        {
            const char query[] =
            u8R"(
                INSERT INTO `card`
                (`user_id`, `name`, `note_json`, `url`, `revision`)
                VALUES(?,?,?,?,ifnull((SELECT MAX(revision) FROM card) + 1,1));
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_register_, nullptr);
        }

        // stmt_get_current_revision_
        {
            const char query[] =
            u8R"(
                SELECT revision FROM `card`
                ORDER BY revision DESC
                LIMIT 1
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_get_current_revision_, nullptr);
        }

        // stmt_get_revision_patch_
        {
            const char query[] =
            u8R"(
                SELECT * FROM `card`
                WHERE revision > ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_get_revision_patch_, nullptr);
        }

        // stmt_apply_revision_patch_
        {
            const char query[] =
            u8R"(
                INSERT OR REPLACE INTO `card`
                (`id`, `user_id`, `name`, `note_json`, `url`, `finger_print`, `size`, `revision`, `status`)
                VALUES(?,?,?,?,?,?,?,?,?);
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_apply_revision_patch_, nullptr);
        }

        // stmt_get_waiting_url_
        {
            const char query[] =
            u8R"(
                SELECT id, url FROM `card`
                WHERE status = 0
                LIMIT 1
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_get_waiting_url_, nullptr);
        }

        // stmt_accept_url_
        {
            const char query[] =
            u8R"(
                UPDATE `card`
                SET status = 1,
                finger_print = ?,
                size = ?,
                revision = (SELECT MAX(revision) FROM card) + 1
                WHERE id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_accept_url_, nullptr);
        }

        // stmt_reject_url_
        {
            const char query[] =
            u8R"(
                UPDATE `card`
                SET status = 2,
                revision = (SELECT MAX(revision) FROM card) + 1
                WHERE id = ?
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_reject_url_, nullptr);
        }

        // stmt_remove_
        {
            const char query[] =
            u8R"(
                UPDATE `card`
                SET status = 3,
                user_id = 0,
                name = '',
                note_json = '',
                url = '',
                finger_print = '',
                size = 0,
                revision = (SELECT MAX(revision) FROM card) + 1
                WHERE id = ?
                AND revision != (SELECT MAX(revision) FROM card)
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_remove_, nullptr);
        }

        // stmt_clean_removed_
        {
            const char query[] =
            u8R"(
                DELETE FROM `card`
                WHERE status = 3
                AND revision != (SELECT MAX(revision) FROM card)
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_clean_removed_, nullptr);
        }

        // stmt_get_size_
        {
            const char query[] =
            u8R"(
                SELECT COUNT(*) FROM `card`
            )"; // ヒアドキュメント

            sqlite3_prepare_v2(db_, query, strlen(query), &stmt_get_size_, nullptr);
        }
    }
}

CardLibrary::~CardLibrary()
{
    if (db_) {
        sqlite3_finalize(stmt_register_);
        sqlite3_finalize(stmt_get_current_revision_);
        sqlite3_finalize(stmt_get_revision_patch_);
        sqlite3_finalize(stmt_apply_revision_patch_);
        sqlite3_finalize(stmt_get_waiting_url_);
        sqlite3_finalize(stmt_accept_url_);
        sqlite3_finalize(stmt_reject_url_);
        sqlite3_finalize(stmt_remove_);
        sqlite3_finalize(stmt_clean_removed_);
        sqlite3_finalize(stmt_get_size_);

        stmt_register_ = nullptr;
        stmt_get_current_revision_ = nullptr;
        stmt_get_revision_patch_ = nullptr;
        stmt_apply_revision_patch_ = nullptr;
        stmt_get_waiting_url_ = nullptr;
        stmt_accept_url_ = nullptr;
        stmt_reject_url_ = nullptr;
        stmt_remove_ = nullptr;
        stmt_clean_removed_ = nullptr;
        stmt_get_size_ = nullptr;

        sqlite3_close(db_);
        db_ = nullptr;
    }
}

int CardLibrary::Register(const std::string& name,
                   const std::string& note,
                   const std::string& url,
                   int user_id)
{
    int return_code = 0;
    {

        sqlite3_reset(stmt_register_);

        sqlite3_bind_int(stmt_register_, 1, static_cast<int>(user_id));
        sqlite3_bind_text(stmt_register_, 2, name.c_str(), name.size(), SQLITE_STATIC);
        sqlite3_bind_text(stmt_register_, 3, note.c_str(), note.size(), SQLITE_STATIC);
        sqlite3_bind_text(stmt_register_, 4, url.c_str(), url.size(), SQLITE_STATIC);

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_register_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
            return_code = -1;
        }

    }
    return return_code;
}

int CardLibrary::GetCurrentRevision()
{
    int revision = 0;
    {
        sqlite3_reset(stmt_get_current_revision_);

        int result;
        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_current_revision_))){
            revision = sqlite3_column_int(stmt_get_current_revision_, 0);
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return revision;
}

std::string CardLibrary::GetRevisionPatch(int revision)
{
    std::string patch;

    sqlite3_reset(stmt_get_revision_patch_);
    sqlite3_bind_int(stmt_get_revision_patch_, 1, revision);

    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_get_revision_patch_))){
        int id = sqlite3_column_int(stmt_get_revision_patch_, 0);
        int user_id = sqlite3_column_int(stmt_get_revision_patch_, 1);

        std::string name(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 2)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 2));

        std::string note(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 3)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 3));

        std::string url(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 4)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 4));

        std::string finger_print(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_revision_patch_, 5)),
                sqlite3_column_bytes(stmt_get_revision_patch_, 5));

        int size = sqlite3_column_int(stmt_get_revision_patch_, 6);

        int revision = sqlite3_column_int(stmt_get_revision_patch_, 7);
        char status = sqlite3_column_int(stmt_get_revision_patch_, 8);

        patch += network::Utils::Serialize(id, user_id, name, note, url, finger_print, size, revision, status);
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }

    return patch;
}

void CardLibrary::ApplyRevisionPatch(const std::string& patch)
{
    std::string buffer(patch);

    while (buffer.size() > 0) {

        std::tuple<int, int,std::string, std::string, std::string, std::string, int, int, char> result_tuple;
        size_t readed = network::Utils::Deserialize(buffer, &result_tuple);

        buffer.erase(0, readed);

        int id, user_id;
        std::string name, note, url, finger_print;
        int size, revision;
        char status;

        id = std::get<0>(result_tuple);
        user_id = std::get<1>(result_tuple);
        name = std::get<2>(result_tuple);
        note = std::get<3>(result_tuple);
        url = std::get<4>(result_tuple);
        finger_print = std::get<5>(result_tuple);
        size = std::get<6>(result_tuple);
        revision = std::get<7>(result_tuple);
        status = std::get<8>(result_tuple);

        sqlite3_reset(stmt_apply_revision_patch_);
        sqlite3_bind_int(stmt_apply_revision_patch_, 1, id);
        sqlite3_bind_int(stmt_apply_revision_patch_, 2, user_id);
        sqlite3_bind_text(stmt_apply_revision_patch_, 3, name.data(), name.size(), SQLITE_TRANSIENT);
        sqlite3_bind_text(stmt_apply_revision_patch_, 4, note.data(), note.size(), SQLITE_TRANSIENT);
        sqlite3_bind_text(stmt_apply_revision_patch_, 5, url.data(), url.size(), SQLITE_TRANSIENT);
        sqlite3_bind_text(stmt_apply_revision_patch_, 6, finger_print.data(), finger_print.size(), SQLITE_TRANSIENT);
        sqlite3_bind_int(stmt_apply_revision_patch_, 7, size);
        sqlite3_bind_int(stmt_apply_revision_patch_, 8, revision);
        sqlite3_bind_int(stmt_apply_revision_patch_, 9, status);

        int result, loop=0;
        while (SQLITE_DONE != (result = sqlite3_step(stmt_apply_revision_patch_)) && loop++<1000){}
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }

    // 削除フラグの付いたデータを消去
    sqlite3_reset(stmt_clean_removed_);

    int result, loop=0;
    while (SQLITE_DONE != (result = sqlite3_step(stmt_clean_removed_)) && loop++<1000){}
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

std::pair<int, std::string> CardLibrary::GetWaitingUrl()
{
    int id = -1;
    std::string url;
    {
        sqlite3_reset(stmt_get_waiting_url_);

        int result;
        while (SQLITE_ROW == (result = sqlite3_step(stmt_get_waiting_url_))){
            id = sqlite3_column_int(stmt_get_waiting_url_, 0);
            url = std::string(reinterpret_cast<const char*>(sqlite3_column_text(stmt_get_waiting_url_, 1)),
                    sqlite3_column_bytes(stmt_get_waiting_url_, 1));
        }
        if (result != SQLITE_DONE) {
            Logger::Error(sqlite3_errmsg(db_));
        }
    }
    return std::pair<int, std::string>(id, url);
}

void CardLibrary::AcceptUrl(int id, const std::string& finger_print, int size)
{
    sqlite3_reset(stmt_accept_url_);

    sqlite3_bind_text(stmt_accept_url_, 1, finger_print.c_str(), finger_print.size(), SQLITE_STATIC);
    sqlite3_bind_int(stmt_accept_url_, 2, size);
    sqlite3_bind_int(stmt_accept_url_, 3, id);

    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_accept_url_))){
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

void CardLibrary::RejectUrl(int id)
{
    sqlite3_reset(stmt_reject_url_);
    sqlite3_bind_int(stmt_reject_url_, 1, id);

    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_reject_url_))){
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}

void CardLibrary::Remove(int id)
{
    sqlite3_reset(stmt_remove_);
    sqlite3_bind_int(stmt_remove_, 1, id);

    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_remove_))){
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
}


int CardLibrary::Size()
{
    sqlite3_reset(stmt_get_size_);
    int size = -1;

    int result;
    while (SQLITE_ROW == (result = sqlite3_step(stmt_get_size_))){
        size = sqlite3_column_int(stmt_get_size_, 0);
    }
    if (result != SQLITE_DONE) {
        Logger::Error(sqlite3_errmsg(db_));
    }
    return size;
}
