//
// PlayerManager.cpp
//

#include "PlayerManager.hpp"
#include "CommandManager.hpp"
#include "CardManager.hpp"
#include "WorldManager.hpp"
#include "../common/network/Utils.hpp"
#include "../common/network/Command.hpp"
#include "../common/Logger.hpp"
#include "unicode.hpp"

#include "3d/CharacterDataProvider.hpp"
#include "3d/CharacterManager.hpp"
#include "3d/Timer.hpp"
#include "3d/model.hpp"
#include "3d/PlayerCharacter.hpp"

#include <list>

PlayerManager::PlayerManager(const ManagerAccessorPtr& manager_accessor) :
revision_(0),
timer_(std::make_shared<Timer>()),
manager_accessor_(manager_accessor),
charmgr_(std::make_shared<CharacterManager>())
{
}

void PlayerManager::Init()
{
    AddCharacter<FieldPlayer>(0, 0);
    charmgr_->set_my_character_id(0);
}

void PlayerManager::ProcessInput(InputManager* input)
{

}

void PlayerManager::Update()
{
    timer_->Next();

    // 自分自身のキャラクターIDは常に0, それ以外はアカウントのID
    // プレイヤーの位置情報を更新
    if (auto player = GetMyself()) {
        const VECTOR& pos = char_data_providers_[charmgr_->my_character_id()]->position();
        const float theta = char_data_providers_[charmgr_->my_character_id()]->theta();
        player->set_position({pos.x, pos.y, pos.z, theta});
    }

    // 位置情報を送信
    static int count = 0;
    if (count % 30 == 0) {
        const VECTOR& pos = char_data_providers_[charmgr_->my_character_id()]->position();
        const float theta = char_data_providers_[charmgr_->my_character_id()]->theta();
        if (auto command_manager = manager_accessor_->command_manager().lock()) {
            command_manager->Write(network::ServerUpdatePlayerPosition(pos.x, pos.y, pos.z, theta));
        }
    }
    count++;

    for (auto pair : players_) {
        pair.second->Update();
    }
}

void PlayerManager::Draw()
{
    // TODO: モデルの高さを取得する必要あり
    for (auto pair : players_) {
        pair.second->Draw();
        if (char_data_providers_.find(pair.second->id()) != char_data_providers_.end()) {
            const VECTOR& pos = char_data_providers_[pair.second->id()]->position();
            const float theta = char_data_providers_[pair.second->id()]->theta();
            pair.second->set_position({pos.x, pos.y, pos.z, theta});
        }
    }
}

int PlayerManager::GetCurrentRevision()
{
    return revision_;
}

void PlayerManager::ApplyRevisionPatch(const std::string& patch)
{
    std::string buffer(patch);
    std::list<unsigned int> logout_user_ids;

    while (buffer.size() > 0) {
        std::tuple<unsigned int, std::string, std::string, std::string, uint16_t, char, int> result_tuple;
        size_t readed = network::Utils::Deserialize(buffer, &result_tuple);
        buffer.erase(0, readed);

        unsigned int user_id;
        std::string name, trip;
        std::string ip_address;
        uint16_t udp_port;
        char login;
        int revision;

        user_id = std::get<0>(result_tuple);
        name = std::get<1>(result_tuple);
        trip = std::get<2>(result_tuple);
        ip_address = std::get<3>(result_tuple);
        udp_port = std::get<4>(result_tuple);
        login = std::get<5>(result_tuple);
        revision = std::get<6>(result_tuple);

        if (login) {
            // プレイヤーを追加
            if (players_.find(user_id) == players_.end()) {
                auto player = std::make_shared<Player>(user_id);
                player->set_name(name);
                player->set_trip(trip);
                player->set_revision(revision);
                player->set_login(login);
                player->set_ip_address(ip_address);
                player->set_udp_port(udp_port);

                players_[user_id] = player;

                auto command_manager = manager_accessor_->command_manager().lock();
                if (user_id != command_manager->user_id()) {
                    AddCharacter<PlayerCharacter>(user_id, 1);
                }

                auto card_manager = manager_accessor_->card_manager().lock();
                card_manager->OnLogin(player);
            }
        } else {
            logout_user_ids.push_back(user_id);
        }

        if (players_.find(user_id) != players_.end()) {
            auto player = players_[user_id];
            player->set_name(name);
            player->set_trip(trip);
            player->set_revision(revision);
            player->set_login(login);
            player->set_ip_address(ip_address);
            player->set_udp_port(udp_port);
        }

        revision_ = std::max(revision_, revision);
    }

    for (unsigned int id : logout_user_ids) {
        if (players_.find(id) != players_.end()) {
            auto card_manager = manager_accessor_->card_manager().lock();
            card_manager->OnLogout(players_[id]);
            charmgr_->Remove(id);
            players_.erase(id);
            Logger::Debug("char_data_providers::Remove : %d", char_data_providers_.erase(id));
        }
    }
}

PlayerPtr PlayerManager::GetMyself()
{
    if (auto command_manager = manager_accessor_->command_manager().lock()) {
        return GetFromId(command_manager->user_id());
    } else {
        return PlayerPtr();
    }
}

PlayerPtr PlayerManager::GetFromId(unsigned int user_id)
{
    if (players_.find(user_id) != players_.end()) {
        return players_[user_id];
    } else {
        return PlayerPtr();
    }
}

std::vector<PlayerPtr> PlayerManager::GetAll()
{
    std::vector<PlayerPtr> players;
    for (auto pair : players_) {
        if (pair.second->login()) {
            players.push_back(pair.second);
        }
    }
    return players;
}

void PlayerManager::UpdatePlayerPosition(unsigned int user_id, const Player::Position& pos)
{
    if (players_.find(user_id) != players_.end()) {
        char_data_providers_[user_id]->set_target_position({pos.x, pos.y, pos.z});
    }
}

std::shared_ptr<CharacterManager> PlayerManager::charmgr() const
{
    return charmgr_;
}

TimerPtr PlayerManager::timer() const
{
    return timer_;
}

std::map<unsigned int, std::unique_ptr<CharacterDataProvider>>& PlayerManager::char_data_providers()
{
    return char_data_providers_;
}

// 指定されたキャラIDとモデル番号を持つキャラをcharmgrに追加する
template <typename CharacterType>
void PlayerManager::AddCharacter(unsigned int character_id, int model_number)
{
    auto world_manager = manager_accessor_->world_manager().lock();
    std::unique_ptr<CharacterDataProvider> cdp_ptr(new CharacterDataProvider());
    auto& cdp = *cdp_ptr;
    char_data_providers_[character_id] = move(cdp_ptr);
    cdp.set_id(character_id);
    cdp.set_model(model_number);
    auto character = std::make_shared<CharacterType>(cdp, world_manager->stage(), timer_);
    charmgr_->Add(character_id, character);
}
