//
// Core.cpp
//

#include "Core.hpp"
#include "../common/Logger.hpp"

Core::Core() : accessor_(new CoreAccessor(this))
{
}

int Core::Run()
{
    const char* configfile = CONFIG_PATH;
    config_.Load(configfile);

    if (SetUpDxLib() == -1) {
        return -1;
    }

    LoadFont();
    SceneInit();

    MainLoop();

    DxLib_End();
    return 0;
}

void Core::MainLoop()
{
    while(ProcessMessage() == 0 &&
            ScreenFlip() == 0 &&
            ClearDrawScreen() == 0 &&
            current_scene_) {
        InputManager::Update();
        InputManager input;
        if (input.GetKeyCount(InputManager::KEYBIND_EXIT) > 0) {
            break;
        }

        current_scene_->Update();
        current_scene_->Draw();
    }

    if (current_scene_)
    {
        current_scene_->End();
    }
}

int Core::SetUpDxLib()
{
    //TCHAR title[] = WINDOW_TITILE;
    const TCHAR* title = WINDOW_TITILE;
    SetMainWindowText(title);

    SetWindowIconID(100);

    if (config_.fullscreen() == 0) {
        ChangeWindowMode(TRUE);
    }

    SetAlwaysRunFlag(TRUE);

    // 多重起動許可（デバッグ専用）
    SetDoubleStartValidFlag(TRUE);

    SetGraphMode(config_.screen_width(), config_.screen_height(), 32);

    if (config_.antialias() == 1) {
        SetFullSceneAntiAliasingMode(4, 2);
    }

    if( DxLib_Init() == -1 )
        return -1;

    SetDrawScreen(DX_SCREEN_BACK) ;

    return 0;
}

void Core::SceneInit()
{
    SetScene(scene::BasePtr(new scene::Main(*accessor_)));
}

void Core::SetScene(scene::BasePtr scene)
{
    if (current_scene_)
    {
        current_scene_->End();
    }

    current_scene_ = std::move(scene);

    if (current_scene_)
    {
        current_scene_->Begin();
    }
}

void Core::LoadFont()
{
    //TCHAR font_path[] = FONT_FILE_PATH;
    const TCHAR* font_path = FONT_FILE_PATH;
    if (AddFontResourceEx(font_path, FR_PRIVATE, nullptr) == 0) {
        Logger::Error("Cannot load font: %s", font_path);
    }
}

void Core::CoreAccessor::SetScene(scene::BasePtr scene)
{
    core_->SetScene(std::move(scene));
}

const Config& Core::CoreAccessor::config()
{
    return core_->config_;
}
