//
// Config.cpp
//

#include "Config.hpp"
#include "../common/Logger.hpp"

Config::Config() :
    fullscreen_(0),
    screen_width_(800),
    screen_height_(600),
    host_("0.0.0.0"),
    port_(39390),
    model_dir_("./model"),
    max_script_execution_time_(5000),
    max_local_storage_size_(512000),
    upnp_(true),
    udp_port_(39391)
{

}

void Config::Load(const std::string& filename)
{

    std::ifstream ifs(filename);
    std::string source_str((std::istreambuf_iterator<char>(ifs)),
            std::istreambuf_iterator<char>());

    ScriptEnvironment script;
    script.ParseJSON(source_str,
            [&](const Handle<Value>& result, const std::string& error) {
                if (result.IsEmpty() || !result->IsObject()) {
                    Logger::Error("%s の読み込みに失敗しました。", filename);
                    return;
                }

                auto array = result->ToObject();
                Get(array, "fullscreen", &fullscreen_);
                Get(array, "screen_width", &screen_width_);
                Get(array, "screen_height", &screen_height_);
                Get(array, "antialias", &antialias_);
                Get(array, "host", &host_);
                Get(array, "port", &port_);
                Get(array, "model_dir", &model_dir_);
                Get(array, "max_script_execution_time", &max_script_execution_time_);
                Get(array, "max_local_storage_size", &max_local_storage_size_);
                Get(array, "upnp", &upnp_);
                Get(array, "udp_port", &udp_port_);
            });

}

void Config::Get(const Handle<Object>& object, const std::string& name, int* value)
{
    Handle<String> key = String::New(name.c_str());
    if (object->Has(key)) {
        *value = object->Get(key)->ToInteger()->Value();
    }
}

void Config::Get(const Handle<Object>& object, const std::string& name, bool* value)
{
    Handle<String> key = String::New(name.c_str());
    if (object->Has(key)) {
        *value = object->Get(key)->ToBoolean()->Value();
    }
}

void Config::Get(const Handle<Object>& object, const std::string& name,
        std::string* value)
{
    Handle<String> key = String::New(name.c_str());
    if (object->Has(key)) {
        *value = std::string(
                *v8::String::Utf8Value(object->Get(key)->ToString()));
    }
}

bool Config::fullscreen() const
{
    return fullscreen_;
}

int Config::screen_width() const
{
    return screen_width_;
}

int Config::screen_height() const
{
    return screen_height_;
}

bool Config::antialias() const
{
    return antialias_;
}

std::string Config::host() const
{
    return host_;
}

int Config::port() const
{
    return port_;
}

std::string Config::model_dir() const
{
    return model_dir_;
}

int Config::max_script_execution_time() const
{
    return max_script_execution_time_;
}

int Config::max_local_storage_size() const
{
    return max_local_storage_size_;
}

bool Config::upnp() const
{
    return upnp_;
}

int Config::udp_port() const
{
    return udp_port_;
}
