//
// CommandManager.cpp
//

#include "CommandManager.hpp"
#include "CardManager.hpp"
#include "PlayerManager.hpp"
#include "../common/Logger.hpp"
#include "../common/network/Client.hpp"

CommandManager::CommandManager(const ManagerAccessorPtr& manager_accessor) :
manager_accessor_(manager_accessor)
{
}

void CommandManager::Update()
{
    if (client_ && !client_->command_empty())
    {
        auto command = client_->PopCommand();
        unsigned int header =  command.header();

        CardManagerPtr card_manager;
        PlayerManagerPtr player_manager;

        card_manager = manager_accessor_->card_manager().lock();
        player_manager = manager_accessor_->player_manager().lock();

        Logger::Debug("Receive: 0x%08x %d byte", header, command.body().size());

        switch (header) {
        using namespace network::header;

        case ClientStartEncryptedSession:
        {
            Logger::Info("Request card database update data...");
            Logger::Info("Request account database update data...");
            client_->Write(network::ServerRequestedCardRevisionPatch(card_manager->GetCurrentRevision()));
            client_->Write(network::ServerRequestedAccountRevisionPatch(player_manager->GetCurrentRevision()));
        }
            break;

        // チャットメッセージ受信
//        case ClientReceiveChatMessage:
//        // case ClientReceiveChatLog:
//        {
//            std::tuple<unsigned int, std::string, std::string> result_tuple;
//            network::Utils::Deserialize(command.body(), &result_tuple);
//
//            card_manager->OnChatReceive(
//                    std::get<0>(result_tuple),
//                    std::get<1>(result_tuple),
//                    std::get<2>(result_tuple));
//        }
//            break;

        case ClientReceiveJSON:
        // case ClientReceiveChatLog:
        {
            std::tuple<std::string, std::string> result_tuple;
            network::Utils::Deserialize(command.body(), &result_tuple);

            Logger::Info("Receive JSON: %s %s",
                    std::get<0>(result_tuple), std::get<1>(result_tuple));

            card_manager->OnReceiveJSON(
                    std::get<0>(result_tuple),
                    std::get<1>(result_tuple));
        }
            break;

        // プレイヤー位置更新
        case ClientUpdatePlayerPosition:
        {
            std::tuple<unsigned int, float, float, float, float> result_tuple;
            network::Utils::Deserialize(command.body(), &result_tuple);
            Player::Position pos = {
                    std::get<1>(result_tuple),
                    std::get<2>(result_tuple),
                    std::get<3>(result_tuple),
                    std::get<4>(result_tuple)
            };

            player_manager->UpdatePlayerPosition(std::get<0>(result_tuple), pos);
            // auto data = network::ClientUpdatePlayerPosition(command);
            // char_data_providers_[1]->set_position(VGet(data.x, data.y, data.z));
            // char_data_providers_[1]->set_theta(data.theta);
        }
            break;

        case ClientReceiveCardRevisionUpdateNotify:
        {
            Logger::Info("Receive card database update notify");
            int server_revision;
            assert(command.body().size() == sizeof(server_revision));
            network::Utils::Deserialize(command.body(), &server_revision);
            auto current_revision = card_manager->GetCurrentRevision();
            if (server_revision > current_revision) {
                client_->Write(network::ServerRequestedCardRevisionPatch(current_revision));
            }
        }
            break;

        case ClientReceiveAccountRevisionUpdateNotify:
        {
            Logger::Info("Receive account database update notify");
            int server_revision;
            assert(command.body().size() == sizeof(server_revision));
            network::Utils::Deserialize(command.body(), &server_revision);
            auto current_revision = player_manager->GetCurrentRevision();
            if (server_revision > current_revision) {
                client_->Write(network::ServerRequestedAccountRevisionPatch(current_revision));
            }
        }
            break;

        case ClientReceiveCardRevisionPatch:
        {
            Logger::Info("Receive card database update data");
            assert(command.body().size() > 0);
            card_manager->ApplyRevisionPatch(command.body());
        }
            break;

        case ClientReceiveAccountRevisionPatch:
        {
            Logger::Info("Receive account database update data");
            assert(command.body().size() > 0);
            player_manager->ApplyRevisionPatch(command.body());
        }
            break;

        default:
            break;
        }
    }
}

void CommandManager::set_client(ClientUniqPtr client)
{
    client_= std::move(client);
}


void CommandManager::Write(const network::Command& command)
{
    if (client_) {
        client_->Write(command);
    }
}

unsigned int CommandManager::user_id()
{
    if (client_) {
        return client_->id();
    } else {
        return 0;
    }
}

void CommandManager::SendJSON(const std::string& msg) {
    if (client_ && msg.size() > 0) {
        client_->Write(network::ServerReceiveJSON(msg));
    }
}

