//
// Card.cpp
//

#include "Card.hpp"
#include <boost/filesystem.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <DxLib.h>
#include <Psapi.h>
#include <fstream>
#include "CardManager.hpp"
#include "InputManager.hpp"
#include "CommandManager.hpp"
#include "PlayerManager.hpp"
#include "Player.hpp"
#include "ResourceManager.hpp"
#include "unicode.hpp"
#include "../common/network/Command.hpp"

char Card::STORAGE_DIR[] = "storage";
char Card::SCRIPT_PATH[] = "resources/js";
int Card::max_local_storage_size;
std::set<Card*> Card::ptr_set;

bool Card::force_gc_flag;

Card::Card(
    const ManagerAccessorPtr& manager_accessor,
    std::string source_folder,
    std::string name,
    std::string author,
    std::string caption,
    std::string icon,
    std::vector<std::string> scripts,
    bool group,
    bool autorun) :
    manager_accessor_(manager_accessor),
    source_folder_(source_folder),
    name_(name),
    author_(author),
    caption_(caption),
    icon_(icon),
    scripts_(scripts),
    group_(group),
    autorun_(autorun),
    event_id_(0),
    icon_handle_(-1),
    running_(false),
    close_flag_(false)
{

    using namespace v8;
    icon_handle_ = ResourceManager::LoadCachedGraph((source_folder_ + "/" + icon_).c_str());

    // 入力ウィンドウのセットアップ
    inputbox.enable_ = true;
    inputbox.tabname_ = name_;

    // Cardへのポインタをコンテキストに保持
    script_.With([&](const Handle<Context>& context){
        Handle<ObjectTemplate> script_template = ObjectTemplate::New();
        script_template->SetInternalFieldCount(1);
        auto script_object = script_template->NewInstance();
        script_object->SetPointerInInternalField(0, this);

        // デバッグ用にコンテキストに登録したポインタを記録
        ptr_set.insert(this);

        context->Global()->Set(String::New("Network"),  script_object->Clone());
        context->Global()->Set(String::New("Player"),  script_object->Clone());
        context->Global()->Set(String::New("Account"), script_object->Clone());
        context->Global()->Set(String::New("InputBox"),   script_object->Clone());
        context->Global()->Set(String::New("Card"),    script_object->Clone());
        context->Global()->Set(String::New("Screen"),  script_object->Clone());
        context->Global()->Set(String::New("UI"),      script_object->Clone());
    });

    // 関数を登録
    SetFunctions();

    std::cout << "Load MMO js" << std::endl;

    // 補助スクリプトをロード
    script_.Load("mmo.js");

    // ストレージを読み込み
    LoadStorage();

    // UIBoardをセット
    script_.Execute("(new UI.Board)", "",
            [&](const Handle<Value>& value, const std::string error) {
                ui_board_obj_ = Persistent<Object>::New(value->ToObject());
            });

    assert(!ui_board_obj_.IsEmpty() && ui_board_obj_->IsObject());
}

Card::~Card()
{
    OnClose();
    SaveStorage();
    ui_board_obj_.Dispose();
    local_storage_.Dispose();
    network_on_receive_json_.Dispose();
    inputbox.on_enter_.Dispose();

    ptr_set.erase(this);
}

void Card::SetFunctions()
{

    /**
    * @module global
    */

    script_.SetProperty("global",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                return Context::GetCurrent()->Global();
            }, (AccessorSetter)0);

    /**
    * スクリプト
    *
    * @class Script
    * @static
    */

//    /**
//     * 強制的にガーベジコレクションを実行します
//     *
//     * @method forceGC
//     * @static
//     */
//    script_.SetFunction("Script.forceGC", [](const Arguments& args) -> Handle<Value> {
//        Card::force_gc_flag = true;
//        return Undefined();
//    });

    /**
     * ローカルデータへのアクセサ
     *
     * このオブジェクトに設定された文字列、数値、ブール値はスクリプトやクライアントが終了してもローカルに保存され、次回実行時に復元されます。
     *
     * 配列やオブジェクトはそのまま保存できないため、JSON.stringifyなどを使う必要があります。
     *
     * @property localStorage
     * @type Object
     * @example
     *     // 保存
     *     localStorage['savedata'] = "でーた";
     *
     *     // JSONで保存
     *     localStorage['json'] = JSON.stringify({"data": 1234});
     *
     * @static
     */
    script_.SetProperty("Card.localStorage",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                assert(ptr_set.find(self) != ptr_set.end());

                if (self->local_storage_.IsEmpty()) {
                    self->local_storage_ = Persistent<Object>::New(Object::New());
                }
                return self->local_storage_;
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                assert(ptr_set.find(self) != ptr_set.end());

                self->local_storage_ = Persistent<Object>::New(value->ToObject());
            });

    /**
    * ネットワーク
    *
    * @class Network
    * @static
    */

    /**
     * サーバーへの接続状態を返します
     *
     * @method status
     * @return {String} 接続状態
     * @static
     */
    script_.SetFunction("Network.online",
            [](const Arguments& args) -> Handle<Value> {
                return String::New("online");
            });

    script_.SetFunction("Network._sendJSONAll",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (args.Length() > 0) {
                    HandleScope handle;
                    v8::String::Utf8Value utf8(args[0]);

                    if (auto command_manager = self->manager_accessor_->command_manager().lock()) {
                        command_manager->SendJSON(*utf8);
                    }
                }
                return Undefined();
            });

    script_.SetProperty("Network._onReceiveJSON",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                assert(ptr_set.find(self) != ptr_set.end());

                return self->network_on_receive_json_;
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                if (value->IsFunction()) {
                    assert(info.Holder()->InternalFieldCount() > 0);
                    auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                    assert(ptr_set.find(self) != ptr_set.end());

                    self->network_on_receive_json_ = Persistent<Function>::New(value.As<Function>());
                }
            });

    /**
     * プレイヤー
     *
     * @class Player
     * @static
     */

    /**
     * ログインしているプレイヤーの一覧を返します
     *
     * @method all
     * @return {Array} Playerオブジェクトの配列
     * @static
     */
    script_.SetFunction("Player.all",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                HandleScope handle;
                auto array = Array::New();

                if (auto player_manager = self->manager_accessor_->player_manager().lock()) {
                    auto players = player_manager->GetAll();
                    int i = 0;
                    for (auto player : players) {
                        array->Set(i, player->GetJSObject());
                        i++;
                    }
                }
                return array;
            });

    /**
    * プレイヤーのIDからプレイヤーオブジェクトを取得します
    *
    * 存在しないプレイヤーやログインしていないプレイヤーの場合は Undefined を返します
    *
    * @method getFromId
    * @param {Integer} プレイヤーのID
    * @return {Player} プレイヤーオブジェクト
    */
    script_.SetFunction("Player.getFromId",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (args.Length() > 0) {
                    if (auto player_manager = self->manager_accessor_->player_manager().lock()) {
                        if (auto player = player_manager->GetFromId(args[0]->ToUint32()->Uint32Value())) {
                            return player->GetJSObject();
                        }
                    }
                }
                return Undefined();
            });

    /**
     * プレイヤーがログインした時に呼ばれます
     *
     * @event onLogin
     * @static
     */
    script_.SetProperty("Player.onLogin",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                return self->player.on_login_;
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                if (value->IsFunction()) {
                    assert(info.Holder()->InternalFieldCount() > 0);
                    auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                    self->player.on_login_ = Persistent<Function>::New(value.As<Function>());
                }
            });

    /**
     * プレイヤーがログアウトした時に呼ばれます
     *
     * @event onLogout
     * @static
     */
    script_.SetProperty("Player.onLogout",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                return self->player.on_logout_;
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                if (value->IsFunction()) {
                    assert(info.Holder()->InternalFieldCount() > 0);
                    auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                    self->player.on_logout_ = Persistent<Function>::New(value.As<Function>());
                }
            });

    /**
     * アカウント
     *
     * @class Account
     * @static
     */

    /**
     * ユーザーIDを返します
     *
     * @method id
     * @return {Integer} ユーザーID
     *
     * @static
     */
    script_.SetFunction("Account.id",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (auto command_manager = self->manager_accessor_->command_manager().lock()) {
                    return Integer::NewFromUnsigned(command_manager->user_id());
                }

                return Undefined();
            });

    /**
     * 現在のニックネームを返します
     *
     * @method name
     * @return {String} ニックネーム
     *
     * @static
     */
    script_.SetFunction("Account.name",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                int id = 0;
                if (auto command_manager = self->manager_accessor_->command_manager().lock()) {
                    id = command_manager->user_id();
                }
                if (auto player_manager = self->manager_accessor_->player_manager().lock()) {
                    if (auto player = player_manager->GetFromId(id)) {
                        return String::New(player->name().c_str());
                    }
                }

                return Undefined();
            });

    /**
     * ニックネームを新しく設定します
     *
     * @method updateName
     * @param {String} name 名前
     *
     * @static
     */
    script_.SetFunction("Account.updateName",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (args.Length() >= 1 && args[0]->IsString())
                if (auto command_manager = self->manager_accessor_->command_manager().lock()) {
                    auto name = std::string(*String::Utf8Value(args[0]->ToString()));
                    command_manager->Write(network::ServerUpdatePlayerName(name));
                }

                return Undefined();
            });

    /**
     * 現在設定されているトリップを返します
     *
     * トリップは20文字のASCII文字列です
     *
     * @method trip
     * @return {String} トリップ
     *
     * @static
     */
    script_.SetFunction("Account.trip",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                int id = 0;
                if (auto command_manager = self->manager_accessor_->command_manager().lock()) {
                    id = command_manager->user_id();
                }
                if (auto player_manager = self->manager_accessor_->player_manager().lock()) {
                    if (auto player = player_manager->GetFromId(id)) {
                        return String::New(player->trip().c_str());
                    }
                }

                return Undefined();
            });

    /**
     * トリップを新しく設定します
     *
     * @method updateTrip
     * @param {String} password トリップパスワード
     * @example
     *     Account.updateTrip('秘密の文字列');
     *
     * @static
     */
    script_.SetFunction("Account.updateTrip",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (args.Length() >= 1 &&
                        args[0]->IsString() &&
                        args[0]->ToString()->Length() > 0) {

                    if (auto command_manager = self->manager_accessor_->command_manager().lock()) {
                        auto trip = std::string(*String::Utf8Value(args[0]->ToString()));
                        command_manager->Write(network::ServerUpdatePlayerTrip(trip));
                    }
                }

                return Undefined();
            });

    /**
     * カード
     *
     * @class Card
     * @static
     */

    /**
     * スクリプトを停止します
     *
     * @method close
     * @static
     */
    script_.SetFunction("Card.close",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                self->close_flag_ = true;
                self->running_ = false;
                return Undefined();
            });

    /**
     * カードライブラリの登録数を返します
     *
     * @method size
     * @return {Integer} カードライブラリの登録数
     * @static
     */
    script_.SetFunction("Card.size",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (auto card_manager = self->manager_accessor_->card_manager().lock()) {
                    return Integer::NewFromUnsigned(card_manager->Size());
                }

                return Integer::New(0);
            });

    script_.SetProperty("Card.board",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {

                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                assert(ptr_set.find(self) != ptr_set.end());

                return self->ui_board_obj_;
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {

                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                assert(ptr_set.find(self) != ptr_set.end());

                self->ui_board_obj_ = Persistent<Object>::New(value->ToObject());
            });

    /**
     * スクリーン情報
     *
     * @class Screen
     * @static
     */

    /**
     * クライアントウィンドウの幅を返します
     *
     * @method width
     * @return {Integer} クライアントウィンドウの幅(px)
     * @static
     */
    script_.SetFunction("Screen.width",
            [](const Arguments& args) -> Handle<Value> {
                int width;
                GetScreenState(&width, nullptr, nullptr);
                return Integer::New(width);
            });

    /**
     * クライアントウィンドウの高さを返します
     *
     * @method height
     * @return {Integer} クライアントウィンドウの高さ(px)
     * @static
     */
    script_.SetFunction("Screen.height",
            [](const Arguments& args) -> Handle<Value> {
                int height;
                GetScreenState(nullptr, &height, nullptr);
                return Integer::New(height);
            });

    /**
     * マウスのクライアントX座標を返します
     *
     * @method mouse_x
     * @return {Integer} マウスX座標(px)
     * @static
     */
    script_.SetFunction("Screen.mouse_x",
            [](const Arguments& args) -> Handle<Value> {
                int x;
                GetMousePoint(&x, nullptr);
                return Integer::New(x);
            });

    /**
     * マウスのクライアントY座標を返します
     *
     * @method mouse_y
     * @return {Integer} マウスY座標(px)
     * @static
     */
    script_.SetFunction("Screen.mouse_y",
            [](const Arguments& args) -> Handle<Value> {
                int y;
                GetMousePoint(&y, nullptr);
                return Integer::New(y);
            });

    /**
     * プレイヤーをアクティブにして、操作できるようにします
     *
     * @method player_focus
     * @static
     */
    script_.SetFunction("Screen.player_focus",
            [](const Arguments& args) -> Handle<Value> {
                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));

                if (auto card_manager = self->manager_accessor_->card_manager().lock()) {
                    card_manager->FocusPlayer();
                }
                return Undefined();
            });

//    /**
//     * ワールド座標をスクリーン座標に変換します
//     *
//     * @method worldToScreen
//     * @param {Float} x
//     * @param {Float} y
//     * @param {Float} z
//     * @retrun {Array} スクリーン座標
//     * @static
//     */
//    script_.SetFunction("Screen.worldToScreen",
//            [](const Arguments& args) -> Handle<Value> {
//                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));
//
//                return Undefined();
//            });

    /**
     * 入力ウィンドウへのアクセスを提供するクラス
     *
     * @class InputBox
     * @static
     */

    /**
     * テキストボックスの内容が確定された時に呼ばれます
     *
     * @event onEnter
     * @return {String} text テキストボックスの内容
     *
     * @static
     */
    script_.SetProperty("InputBox.onEnter",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                return self->inputbox.on_enter_;
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                if (value->IsFunction()) {
                    assert(info.Holder()->InternalFieldCount() > 0);
                    auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                    self->inputbox.on_enter_ = Persistent<Function>::New(value.As<Function>());
                }
            });

    /**
     * テキストボックスに入力できるかどうか
     *
     * このプロパティに trueが設定されていて、
     * 且つinputbox.onEnterに関数がセットされている場合に入力ウィンドウにタブが表示されます。
     *
     * @property enable
     * @type Boolean
     * @default true
     * @static
     */
    script_.SetProperty("InputBox.enable",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                return Boolean::New(self->inputbox.enable_);
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                self->inputbox.enable_ = value->ToBoolean()->BooleanValue();
            });

    /**
     * テキストボックスに表示するメッセージ
     *
     * @property message
     * @type String
     * @static
     */
    script_.SetProperty("InputBox.message",
            [](Local<String> property, const AccessorInfo &info) -> Handle<Value> {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                return String::New(self->inputbox.message_.c_str());
            },
            [](Local<String> property, Local<Value> value, const AccessorInfo& info) {
                assert(info.Holder()->InternalFieldCount() > 0);
                auto self = static_cast<Card*>(info.Holder()->GetPointerFromInternalField(0));
                self->inputbox.message_ = *String::Utf8Value(value->ToString());
            });

    /**
     * 指定されたファンクションキーが押された時に実行される関数を指定します
     *
     * @class FunctionKey
     * @static
     */

    /**
     * @property {Function} F1
     * @static
     */
    /**
     * @property {Function} F2
     * @static
     */
    /**
     * @property {Function} F3
     * @static
     */
    /**
     * @property {Function} F4
     * @static
     */
    /**
     * @property {Function} F5
     * @static
     */
    /**
     * @property {Function} F6
     * @static
     */
    /**
     * @property {Function} F7
     * @static
     */
    /**
     * @property {Function} F8
     * @static
     */
    /**
     * @property {Function} F9
     * @static
     */
    /**
     * @property {Function} F10
     * @static
     */
    /**
     * @property {Function} F11
     * @static
     */
    /**
     * @property {Function} F12
     * @static
     */
    script_.SetConstant("FunctionKey.F1",  Undefined());
    script_.SetConstant("FunctionKey.F2",  Undefined());
    script_.SetConstant("FunctionKey.F3",  Undefined());
    script_.SetConstant("FunctionKey.F4",  Undefined());
    script_.SetConstant("FunctionKey.F5",  Undefined());
    script_.SetConstant("FunctionKey.F6",  Undefined());
    script_.SetConstant("FunctionKey.F7",  Undefined());
    script_.SetConstant("FunctionKey.F8",  Undefined());
    script_.SetConstant("FunctionKey.F9",  Undefined());
    script_.SetConstant("FunctionKey.F10", Undefined());
    script_.SetConstant("FunctionKey.F11", Undefined());
    script_.SetConstant("FunctionKey.F12", Undefined());

    /**
     * GUI
     *
     * @class UI
     * @static
     */

    script_.With(
            [&](const Handle<Context>& context) {
                v8::Handle<v8::ObjectTemplate> object_template = v8::ObjectTemplate::New();

                UIBase::SetObjectTemplate<UIButton>("Button", &object_template);
                UIBase::SetObjectTemplate<UIBoard>("Board", &object_template);
                UIBase::SetObjectTemplate<UILabel>("Label", &object_template);
                UIBase::SetObjectTemplate<UIList>("List", &object_template);
                UIBase::SetObjectTemplate<UIGroup>("Group", &object_template);
                auto script_object = object_template->NewInstance();
                script_object->SetPointerInInternalField(0, this);
                context->Global()->Set(String::New("UI"), script_object);
            });

    // TODO: GetPointerFromObject

    /**
     * @property DOCKING_LEFT
     * @type Integer
     * @static
     * @final
    **/
    script_.SetConstant("UI.DOCKING_LEFT", UIBase::DOCKING_LEFT);

    /**
     * @property DOCKING_TOP
     * @type Integer
     * @static
     * @final
    **/
    script_.SetConstant("UI.DOCKING_TOP", UIBase::DOCKING_TOP);

    /**
     * @property DOCKING_BOTTOM
     * @type Integer
     * @static
     * @final
    **/
    script_.SetConstant("UI.DOCKING_BOTTOM", UIBase::DOCKING_BOTTOM);

    /**
     * @property DOCKING_RIGHT
     * @type Integer
     * @static
     * @final
    **/
    script_.SetConstant("UI.DOCKING_RIGHT", UIBase::DOCKING_RIGHT);


//    /**
//     * GUIツリーにオブジェクトを追加します
//     *
//     * @method addChildren
//     * @param {GUI} object GUIオブジェクト
//     * @static
//     */
//    script_.SetFunction("UI.addChildren",
//            [](const Arguments& args) -> Handle<Value> {
//                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));
//
//                if (args.Length() >= 1 && args[0]->IsObject()) {
//                    v8::HandleScope handle;
//                    self->children_.push_back(std::make_shared<gui::GUIHolder<gui::Object>>(args[0]));
//                }
//                return v8::Undefined();
//            });

//    /**
//    * GUIツリーからオブジェクトを削除します
//    *
//    * @method removeChildren
//    * @param {GUI} object GUIオブジェクト
//    * @static
//    */
//    script_.SetFunction("GUI.removeChildren",
//            [](const Arguments& args) -> Handle<Value> {
//                auto self = static_cast<Card*>(args.Holder()->GetPointerFromInternalField(0));
//
//                if (args.Length() >= 1 && args[0]->IsObject()) {
//                    v8::HandleScope handle;
//                    auto it = std::remove(
//                            self->children_.begin(),
//                            self->children_.end(),
//                            std::make_shared<gui::GUIHolder<gui::Object>>(args[0]));
//                    self->children_.erase(it);
//                }
//                return v8::Undefined();
//            });

}

void Card::Run()
{
    using namespace boost::filesystem3;
    path source_path(source_folder_);

    if (is_directory(source_path)) {
        for (const std::string& script_name : scripts_) {
            auto script_path = source_path / path(script_name);
            if (exists(script_path)) {
                std::ifstream ifs(script_path.string());
                std::string script((std::istreambuf_iterator<char>(ifs)),
                        std::istreambuf_iterator<char>());
                script_.Execute(script, script_path.string(),
                        [](const Handle<Value>& v, const std::string e) {
                    std::cout << "Error: " << e << std::endl;
                });

            } else {
                std::cout << "Error: " << script_path.string()
                        << " No such file." << std::endl;
            }
        }
    }

    running_ = true;
}

void Card::Execute(const std::string& script, const std::string& filename,
             const V8ValueCallBack& callback)
{
    auto compiled_code = script_.CompileCoffeeScript(script);
    script_.Execute(compiled_code, filename, callback);
}

void Card::Update()
{
    if (force_gc_flag) {
        PROCESS_MEMORY_COUNTERS pmc = { 0 };
        DWORD dwProcessID = GetCurrentProcessId();
        HANDLE hProcess;

        if ( (hProcess = OpenProcess(PROCESS_QUERY_INFORMATION,FALSE,dwProcessID)) != NULL ){
            v8::V8::LowMemoryNotification();
            if ( GetProcessMemoryInfo(hProcess,&pmc,sizeof(pmc)) ){
                std::cout << "Using memory: " << pmc.WorkingSetSize << std::endl;
            }
            CloseHandle( hProcess );
        }

        force_gc_flag = false;
    }

    static int cnt = 0;
    if (!ui_board_obj_.IsEmpty() && ui_board_obj_->IsObject()) {
        auto ptr = *static_cast<UIBasePtr*>(ui_board_obj_->GetPointerFromInternalField(0));
        ptr->Update();
        if (cnt % 2 == 0) {
            ptr->AsyncUpdate();
        }
    }
    cnt++;
}

void Card::Draw()
{
    if (!ui_board_obj_.IsEmpty() && ui_board_obj_->IsObject()) {
        auto ptr = *static_cast<UIBasePtr*>(ui_board_obj_->GetPointerFromInternalField(0));
        ptr->Draw();
    }
}

void Card::ProcessInput(InputManager* input)
{
    CheckFunctionKey(KEY_INPUT_F1,  "F1",  *input);
    CheckFunctionKey(KEY_INPUT_F2,  "F2",  *input);
    CheckFunctionKey(KEY_INPUT_F3,  "F3",  *input);
    CheckFunctionKey(KEY_INPUT_F4,  "F4",  *input);
    CheckFunctionKey(KEY_INPUT_F5,  "F5",  *input);
    CheckFunctionKey(KEY_INPUT_F6,  "F6",  *input);
    CheckFunctionKey(KEY_INPUT_F7,  "F7",  *input);
    CheckFunctionKey(KEY_INPUT_F8,  "F8",  *input);
    CheckFunctionKey(KEY_INPUT_F9,  "F9",  *input);
    CheckFunctionKey(KEY_INPUT_F10, "F10", *input);
    CheckFunctionKey(KEY_INPUT_F11, "F11", *input);
    CheckFunctionKey(KEY_INPUT_F12, "F12", *input);

    if (!ui_board_obj_.IsEmpty() && ui_board_obj_->IsObject()) {
        auto ptr = *static_cast<UIBasePtr*>(ui_board_obj_->GetPointerFromInternalField(0));
        ptr->ProcessInput(input);
    }
}

void Card::CheckFunctionKey(int keynum, const std::string& name, const InputManager& input)
{
    if (input.GetKeyCount(keynum) == 1) {
        script_.TimedWith([&](const Handle<Context>& context){
            auto obj = context->Global()->Get(String::New("FunctionKey"));
            if (!obj.IsEmpty() && obj->IsObject()) {
                auto func = obj->ToObject()->Get(String::New(name.c_str()));
                if (!func.IsEmpty() && func->IsFunction()) {
                    func.As<Function>()->CallAsFunction(context->Global(), 0, nullptr);
                }
            }
        });
    }
}

bool Card::HasInputEvent()
{
    bool flag = false;
    script_.With([&](const Handle<Context>& context)
    {
        flag = !inputbox.on_enter_.IsEmpty() && inputbox.on_enter_->IsFunction();
    });
    return flag & inputbox.enable_;
}

void Card::onEnter(const std::string& text)
{
    script_.TimedWith([&](const Handle<Context>& context)
    {
        if (!inputbox.on_enter_.IsEmpty() && inputbox.on_enter_->IsFunction()) {
            Handle<Value> args[1];
            args[0] = String::New(text.data(), text.size());

            inputbox.on_enter_->Call(context->Global(), 1, args);
        }
    });
}

void Card::OnReceiveJSON(const std::string& info_json, const std::string& msg_json)
{
    if (!network_on_receive_json_.IsEmpty() && network_on_receive_json_->IsFunction()) {
        script_.TimedWith(
                [&](const Handle<Context>& context)
                {
                    HandleScope handle;

                    /*
                    v8::Handle<v8::Value> player = v8::Undefined();
                    if (auto player_manager = manager_accessor_->player_manager().lock()) {
                        if (auto player_ptr = player_manager->GetFromId(user_id)) {
                            player = player_ptr->GetJSObject();
                        }
                    }
                    */

                    Handle<Value> args[2];
                    args[0] = String::New(info_json.data(), info_json.size());
                    args[1] = String::New(msg_json.data(), msg_json.size());

                    network_on_receive_json_->Call(context->Global(), 2, args);
                });
    }
}

void Card::OnLogin(const PlayerPtr& player_ptr)
{
    if (!player.on_login_.IsEmpty() && player.on_login_->IsFunction()) {
        script_.TimedWith(
                [&](const Handle<Context>& context)
                {
                    HandleScope handle;
                    Handle<Value> arg = player_ptr->GetJSObject();
                    player.on_login_->Call(context->Global(), 1, &arg);
                });
    }
}

void Card::OnLogout(const PlayerPtr& player_ptr)
{
    if (!player.on_logout_.IsEmpty() && player.on_logout_->IsFunction()) {
        script_.TimedWith(
                [&](const Handle<Context>& context)
                {
                    HandleScope handle;
                    Handle<Value> arg = player_ptr->GetJSObject();
                    player.on_logout_->Call(context->Global(), 1, &arg);
                });
    }
}

void Card::OnClose()
{
    script_.TimedWith([&](const Handle<Context>& context)
    {
        HandleScope handle;
        auto global = context->Global();
        auto chat = global->Get(String::New("Card"));
        if (chat->IsObject()) {
            auto func = chat->ToObject()->Get(String::New("onClose"));
            if (func->IsFunction()) {
                Handle<Function>::Cast(func)->Call(global, 0, nullptr);
            }
        }
    });

}

void Card::LoadStorage()
{
    std::string storage_path = std::string(STORAGE_DIR) + "/"
            + unicode::utf82sjis(name_) + ".json";
    if (boost::filesystem3::exists(storage_path)) {

        std::ifstream ifs(storage_path);
        std::string script((std::istreambuf_iterator<char>(ifs)),
                std::istreambuf_iterator<char>());

        script_.ParseJSON(script,
                [this](const Handle<Value>& value, const std::string& error) {
                    if (value->IsObject()) {
                        auto storage = value->ToObject();
                        local_storage_ = Persistent<Object>::New(storage);
                    }
                });
    }
}

void Card::SaveStorage()
{

    script_.With(
            [&](const Handle<Context>& context) {
                if (!local_storage_.IsEmpty() && local_storage_->IsObject()) {
                    auto object = local_storage_->ToObject();
                    auto names = object->GetOwnPropertyNames();
                    uint32_t length = names->Length();
                    if (length > 0) {

                        std::string storage_path = std::string(STORAGE_DIR) + "/" + unicode::utf82sjis(name_) + ".json";
                        std::ofstream ofs(storage_path);

                        // データ最大サイズ
                        int max_length = max_local_storage_size;

                        ofs << "{";

                        for (uint32_t i = 0; (i < length && ofs.tellp() < max_length); i++) {
                            auto key = names->Get(i)->ToString();
                            auto value = object->Get(key);
                            if (i > 0) {
                                ofs << ",";
                            }
                            if (value->IsString()) {
                                ofs << "\"" << *v8::String::Utf8Value(key) << "\": ";
                                ofs << "\"" << EscapeString(*v8::String::Utf8Value(value->ToString())) << "\"";
                            } else if (value->IsBoolean()) {
                                ofs << "\"" << *v8::String::Utf8Value(key) << "\": ";
                                ofs << (value->ToBoolean()->BooleanValue() ? "true" : "false");
                            } else if (value->IsNumber()) {
                                ofs << "\"" << *v8::String::Utf8Value(key) << "\": ";
                                ofs << value->ToNumber()->Value();
                            }
                        }

                        ofs << "}";
                    }
                }
            });

}

std::string Card::EscapeString(const std::string& str)
{
    std::string result(str);

    uint32_t pos = 0;
    while ((pos = result.find('\\', pos)) != std::string::npos) {
        result.replace(pos, 1, "\\\\");
        pos += 2;
    }
    while ((pos = result.find('\n', pos)) != std::string::npos) {
        result.replace(pos, 1, "\\n");
        pos += 2;
    }
    while ((pos = result.find('\r', pos)) != std::string::npos) {
        result.replace(pos, 1, "\\r");
        pos += 2;
    }
    while ((pos = result.find('"', pos)) != std::string::npos) {
        result.replace(pos, 1, "\\\"");
        pos += 2;
    }

    return result;
}

std::string Card::author() const {
    return author_;
}

bool Card::autorun() const {
    return autorun_;
}

std::string Card::caption() const {
    return caption_;
}

bool Card::group() const {
    return group_;
}

std::string Card::icon() const {
    return icon_;
}

std::string Card::name() const {
    return name_;
}

std::vector<std::string> Card::scripts() const {
    return scripts_;
}

std::string Card::source_folder() const {
    return source_folder_;
}

bool Card::running() const {
    return running_;
}

int Card::icon_handle() {
    return icon_handle_;
}

bool Card::close_flag() const {
    return close_flag_;
}

std::string Card::input_message() const
{
    return inputbox.message_;
}

int Card::focus_index() const
{
    if (!ui_board_obj_.IsEmpty() && ui_board_obj_->IsObject()) {
        auto ptr = *static_cast<UIBasePtr*>(ui_board_obj_->GetPointerFromInternalField(0));
        return ptr->focus_index();
    }
    return -1;
}

void Card::set_max_local_storage_size(int size)
{
    max_local_storage_size = size;
}

