#pragma once

#include <vector>
#include <array>
#include <string>
#include <DxLib.h>
#include "../unicode.hpp"
#include "../InputManager.hpp"
#include "Character.hpp"
#include "CharacterManager.hpp"
#include "CharacterDataProvider.hpp"
#include "MotionPlayer.hpp"
#include "Timer.hpp"

class Stage;
typedef std::shared_ptr<Stage> StagePtr;

struct CameraStatus
{
    //float height;           // カメラY座標のプレイヤー身長に対する比率
    float radius;          // プレイヤーを中心とするカメラ回転軌道の半径
    float target_height;    // 注視点Y座標のプレイヤー身長に対する比率
    float theta, phi;    // カメラの横方向の回転thetaと、縦方向の回転phi、単位はラジアン
    bool manual_control; // true:カメラの方向決定は手動モード
    std::pair<int, int> manual_control_startpos; // 手動モードになった瞬間のマウス座標
};

struct PlayerStatus
{
    VECTOR pos, vel, acc; // プレイヤーのマップ上での位置、速度、加速度
    float roty, roty_speed; // プレイヤーの鉛直軸周りの回転角
    int motion, prev_motion; // モーションの種類(BasicMotion::hoge)
    float blend_ratio; // モーションのブレンド比
    bool is_walking; // true:歩き, false:走り
};

class KeyChecker
{
public:
    int Check();
    int GetKeyCount(size_t key_code) const;

private:
    std::array<int, 256> key_count_;
};

// フィールド上のキャラクタ
class FieldPlayer : public Character
{
public:
    FieldPlayer(CharacterDataProvider& data_provider, const StagePtr& stage, const TimerPtr& timer);

    void Draw() const;
    void Update();
    void Init(tstring model_path);

public:
    int model_handle() const;
    void set_model_handle(int);
    const PlayerStatus& current_stat() const;
    float model_height() const;
    bool any_move() const;

    void LinkToCamera(float* roty);
    void UnlinkToCamera();
    void UpdateInput(InputManager* input);

private:
    void Move();
    void InputFromUser();

private:
    PlayerStatus prev_stat_, current_stat_;
    float model_height_;
    std::unique_ptr<MotionPlayer> motion_player_;
    TimerPtr timer_;

private:
    int model_handle_;
    StagePtr stage_;
    bool any_move_;
    InputManager input_;

    CharacterDataProvider& data_provider_;

    float* camera_roty_;
};

class GameLoop
{
public:
    GameLoop(const StagePtr& stage);
    int Init(const std::string& model_dir_path, std::shared_ptr<CharacterManager> character_manager);
    int Logic(InputManager* input);
    int Draw();

private:
    // 自分自身
    std::shared_ptr<CharacterManager> charmgr_;
    std::shared_ptr<FieldPlayer> myself_;

    StagePtr stage_;
    CameraStatus camera_default_stat, camera;

    void FixCameraPosition();
    void MoveCamera(InputManager* input);
};
