#include "model.hpp"

#include <iostream>
#include <fstream>
#include <vector>
#include <array>
#include <stdexcept>
#include <cstring>
#include <cmath>
#include <boost/format.hpp>
#include <DxLib.h>
#include "../unicode.hpp"
#include "dx_vector.hpp"
#include "BasicMotion.hpp"
#include "PMDLoader.hpp"
#include "CharacterModelManager.hpp"
#include "Stage.hpp"

namespace
{

inline std::ostream& operator <<(std::ostream& os, const std::wstring& s)
{
    os << unicode::ToString(s);
    return os;
}

template <typename T, typename U>
inline bool NearlyEqualRelative(const T& lhs, const T& rhs, const U& ratio)
{
    return abs(lhs - rhs) <= ratio * abs(rhs);
}

} // namespace

int KeyChecker::Check()
{
    std::array<char, 256> key_state;
    int result = GetHitKeyStateAll(key_state.data());

    for (size_t i = 0; i < key_state.size(); ++i)
    {
        if (key_state[i] == 1)
        {
            ++key_count_[i];
        }
        else
        {
            key_count_[i] = 0;
        }
    }

    return result;
}

int KeyChecker::GetKeyCount(size_t key_code) const
{
    return key_count_[key_code];
}

FieldPlayer::FieldPlayer(CharacterDataProvider& data_provider, const StagePtr& stage, const TimerPtr& timer)
:       prev_stat_(),
        current_stat_({ VGet(0, 0, 0), VGet(0, 0, 0), VGet(0, 0, 0),
                0.0f, 0.0f, BasicMotion::STAND, BasicMotion::STAND, 1.0f, false }),
        model_height_(0),
        motion_player_(),
        timer_(timer),
        model_handle_(),
        stage_(stage),
        any_move_(),
        data_provider_(data_provider),
        camera_roty_(nullptr)
{
}

void FieldPlayer::Draw() const
{
    //std::cout << "roty = " << current_stat_.roty << std::endl;

    MV1SetPosition(model_handle_, current_stat_.pos);
    MV1SetRotationXYZ(model_handle_, VGet(0, current_stat_.roty, 0));
    MV1PhysicsCalculation(model_handle_, timer_->DeltaSec());


    MV1DrawModel(model_handle_);

    // DrawLine3D(current_stat_.pos, current_stat_.pos + VGet(2 * stage_->map_scale(), 0, 0), GetColor(255, 0, 0));
    // DrawLine3D(current_stat_.pos, current_stat_.pos + VGet(0, 2 * stage_->map_scale(), 0), GetColor(0, 255, 0));
    // DrawLine3D(current_stat_.pos, current_stat_.pos + VGet(0, 0, 2 * stage_->map_scale()), GetColor(0, 0, 255));
}

void FieldPlayer::Init(tstring model_path)
{
    std::vector<tstring> basic_motion_paths = {
        _T("basic_stand.vmd"),
        _T("basic_walk.vmd"),
        _T("basic_run.vmd")
    };

    // TODO: PMDLoaderをResourceManagerへ移転させる
    PMDLoader loader;
    int handle;
    if ((handle = loader.Load(model_path, _T(""), basic_motion_paths)) == -1 ) {
        //return -1;
    }

    model_handle_ = handle;
    model_height_ = 1.58f;

    std::cout << "loaded" << std::endl;

    motion_player_.reset(new MotionPlayer(model_handle_));
    motion_player_->Play(BasicMotion::STAND, false, 0, -1, FALSE);

    // プレイヤー初期位置
    current_stat_.pos.y = stage_->GetFloorY(current_stat_.pos + VGet(0, 1000, 0), current_stat_.pos - VGet(0, 1000, 0));

    SetupCamera_Perspective(PHI_F * 60.0f / 180.0f); // 視野角60度
    SetCameraNearFar(0.1f * stage_->map_scale(), 200.0f * stage_->map_scale());
}

void FieldPlayer::Update()
{
    /*
    if (key_checker_.Check() == -1)
    {
        throw std::runtime_error("can't ckeck keyboard");
    }
    */

    prev_stat_ = current_stat_;
    Move();

    if (prev_stat_.motion != current_stat_.motion)
    {
        bool connect_prev = false;
        if ((current_stat_.motion == BasicMotion::WALK ||
                    current_stat_.motion == BasicMotion::RUN) &&
                (prev_stat_.motion == BasicMotion::WALK ||
                    prev_stat_.motion == BasicMotion::RUN))
        {
            connect_prev = true;
        }

        motion_player_->Play(current_stat_.motion, connect_prev, 200, -1, false);
    }
    // モーション再生時刻更新
    motion_player_->Next(timer_->Delta());

    data_provider_.set_position(current_stat_.pos);
    data_provider_.set_theta(current_stat_.roty);
    data_provider_.set_motion(current_stat_.motion);
}

void FieldPlayer::Move()
{
    // std::cout << "MovePlayer: " << timer.current_time() << std::endl;

    // myself_.prev_statを元にしてmyself_.current_statを計算する
    current_stat_.pos = prev_stat_.pos + prev_stat_.vel * timer_->DeltaSec();
    current_stat_.vel = prev_stat_.vel + prev_stat_.acc * timer_->DeltaSec();
    current_stat_.acc = prev_stat_.acc;
    current_stat_.roty = prev_stat_.roty + prev_stat_.roty_speed * timer_->DeltaSec();

    if (camera_roty_ != nullptr)
    {
        if (prev_stat_.roty_speed != 0)
        {
            *camera_roty_ = current_stat_.roty;
        }
        else
        {
            current_stat_.roty = *camera_roty_;
        }
    }

    // 50mの深さまで床検出
    auto floor_exists = stage_->FloorExists(current_stat_.pos, model_height_, 50);

    // 移動方向に障害物があるか、または床がない場合は移動不可能
    bool front_collides = stage_->FrontCollides(
            0.2, current_stat_.pos, prev_stat_.pos, 0.4 * stage_->map_scale(), (model_height_ - 0.1) * stage_->map_scale(), 128) ||
        !floor_exists.first;

    if (front_collides)
    {
        current_stat_.pos.x = prev_stat_.pos.x;
        current_stat_.pos.z = prev_stat_.pos.z;
        current_stat_.vel.x = current_stat_.vel.z = 0;
    }

    // 足が地面にめり込んでいるか
    auto foot_floor_exists = stage_->FloorExists(current_stat_.pos, model_height_, 0);

    const auto pos_diff = current_stat_.pos - prev_stat_.pos;
    const auto pos_diff_length = VSize(pos_diff);

    if (pos_diff_length > 0)
    {

        // 前回キャラが接地していたなら、今回もキャラを地面に接地させる
        if (prev_stat_.acc.y == 0)
        {
            // 前回接地していた
            // std::cout << "  previous on the ground" << std::endl;

            if (!front_collides)
            {

                // 登ったり下ったりできる段差の大きさの制限を求める
                static const float y_max_limit_factor = sin(45 * PHI_F / 180);
                static const float y_min_limit_factor = sin(-45 * PHI_F / 180);
                const float y_max_limit = y_max_limit_factor * pos_diff_length;
                const float y_min_limit = y_min_limit_factor * pos_diff_length;

                // 接地点計算
                //std::cout << "  ground collision check: current pos = " << current_stat_.pos << std::endl;

                auto coll_info = MV1CollCheck_Line(stage_->map_handle(), -1,
                        current_stat_.pos + VGet(0, y_max_limit, 0),
                        current_stat_.pos + VGet(0, y_min_limit, 0));
                if (coll_info.HitFlag && NearlyEqualRelative(coll_info.HitPosition.y, floor_exists.second.y, 0.001))
                {
                    // 今回も接地できる
                    //std::cout << "    current on the ground" << std::endl;
                    auto diff = coll_info.HitPosition - prev_stat_.pos;

                    // 角度が急になるほどdiffの長さが大きくなるから、補正する
                    if (VSize(diff) > 0)
                    {
                        current_stat_.pos = prev_stat_.pos + pos_diff_length * VNorm(diff);
                    }
                }
                else if (floor_exists.first)
                {
                    if (floor_exists.second.y < current_stat_.pos.y)
                    {
                        // 床はあるし、自分より低い位置なので落ちる
                        current_stat_.acc.y = -9.8 * stage_->map_scale();
                    }
                    else if (floor_exists.second.y < current_stat_.pos.y + 0.3 * stage_->map_scale())
                    {
                        // 床があり、平らなので登る
                        if (stage_->IsFlatFloor(current_stat_.pos, current_stat_.pos - prev_stat_.pos))
                        {
                            current_stat_.pos.y = floor_exists.second.y;
                        }
                        else
                        {
                            current_stat_.pos = prev_stat_.pos;
                        }
                    }
                    else
                    {
                        // 床があるが、高すぎるので移動不可能
                        current_stat_.pos = prev_stat_.pos;
                    }
                }
                else
                {
                    // 接地できない（移動可能範囲に地面が見つからない）
                    current_stat_.pos = prev_stat_.pos;
                }

            }
        }
        else if (prev_stat_.acc.y < 0)
        {
            //std::cout << "  falling now: current pos = " << current_stat_.pos << std::endl;
            // 空中にいる
            if (current_stat_.pos.y <= prev_stat_.pos.y)
            {
                // 落下している
                // std::cout << "  previous falling" << std::endl;

                // 地面に食い込むのを防止する

                if (foot_floor_exists.first)
                {
                    // 地面に到達した
                    // std::cout << "    current on the ground" << std::endl;

                    current_stat_.pos = foot_floor_exists.second;
                    current_stat_.acc.y = 0;
                    current_stat_.vel.y = 0;
                }
            }
            else
            {
                // 上昇している
                // std::cout << "  previous rising" << std::endl;

                const auto player_top = VGet(0, model_height_ * stage_->map_scale(), 0);
                auto coll_info = MV1CollCheck_Line(stage_->map_handle(), -1,
                        prev_stat_.pos + player_top,
                        current_stat_.pos + player_top);
                if (coll_info.HitFlag)
                {
                    // 天井に到達した
                    // std::cout << "    current collided to ceiling" << std::endl;

                    current_stat_.pos = coll_info.HitPosition - player_top;
                    current_stat_.vel.y = -prev_stat_.vel.y * 1.0; // 反射
                }
            }
        }
        else
        {
            std::cerr << "Player's Y-acceleration is a positive number. Is this an error?" << std::endl;
        }

    }

    any_move_ = false; // キャラを動かすための入力があったかどうか

    if (current_stat_.acc.y == 0)
    {
        current_stat_.vel = VGet(0, 0, 0);
    }

    InputFromUser();
}

void FieldPlayer::InputFromUser()
{
    InputManager& input = input_;
    // myself_.current_statを更新する
    const auto roty = prev_stat_.roty;
    const auto move_speed = prev_stat_.motion == BasicMotion::WALK ? 2.0f
        : prev_stat_.motion == BasicMotion::RUN ? 8.0f
        : 0;
    const auto rot_speed = (prev_stat_.motion == BasicMotion::WALK ? 90.0f
        : prev_stat_.motion == BasicMotion::RUN ? 180.0f
        : 90.0f) * PHI_F / 180;

    int move_dir = 0;
    if (input.GetKeyCount(InputManager::KEYBIND_FORWARD) > 0)
    {
        ++move_dir;
    }
    if (input.GetKeyCount(InputManager::KEYBIND_BACK) > 0)
    {
        --move_dir;
    }

    // Shiftで歩きと走りの切り替え
//    if (input.GetKeyCount(InputManager::KEYBIND_CHANGE_SPEED) == 1 ||
//        input.GetKeyCount(InputManager::KEYBIND_CHANGE_SPEED2) == 1)
//    {
//        current_stat_.is_walking = !prev_stat_.is_walking;
//    }

    if (current_stat_.acc.y == 0 && move_dir != 0)
    {
        // 接地しており、かつ移動する
        any_move_ = true;
        current_stat_.vel = VGet(sin(roty), 0, cos(roty)) * (-move_dir * move_speed * stage_->map_scale());
        current_stat_.motion =
            current_stat_.is_walking ? BasicMotion::WALK : BasicMotion::RUN;
    }
    else if (current_stat_.acc.y != 0)
    {
        // 空中にいる
        any_move_ = true;
        auto acc = 10.0f;
        auto vel = current_stat_.vel + VGet(sin(roty), 0, cos(roty)) * (-move_dir * acc * stage_->map_scale() * timer_->DeltaSec());
        vel.y = 0;

        if (VSize(vel) > std::max(move_speed, 1.0f) * stage_->map_scale())
        {
            vel = vel * (std::max(move_speed, 1.0f) * stage_->map_scale() / VSize(vel));
        }
        vel.y = current_stat_.vel.y;
        current_stat_.vel = vel;
    }
    else
    {
        // 接地しており、移動しない
        current_stat_.motion = BasicMotion::STAND;
    }

    int rot_dir = 0;
    if (input.GetKeyCount(InputManager::KEYBIND_RIGHT_TRUN) > 0)
    {
        ++rot_dir;
    }
    if (input.GetKeyCount(InputManager::KEYBIND_LEFT_TURN) > 0)
    {
        --rot_dir;
    }

    if (rot_dir != 0)
    {
        any_move_ = true;
        current_stat_.roty_speed = rot_dir * rot_speed;
    }
    else
    {
        current_stat_.roty_speed = 0;
    }

    if (current_stat_.acc.y == 0 && input.GetKeyCount(InputManager::KEYBIND_JUMP) > 0)
    {
        any_move_ = true;
        current_stat_.acc.y = -9.8 * stage_->map_scale();
        current_stat_.vel += VGet(0, 5.0 * stage_->map_scale(), 0);
    }
}

int FieldPlayer::model_handle() const
{
    return model_handle_;
}

void FieldPlayer::set_model_handle(int handle)
{
    model_handle_ = handle;
}

const PlayerStatus& FieldPlayer::current_stat() const
{
    return current_stat_;
}

float FieldPlayer::model_height() const
{
    return model_height_;
}

bool FieldPlayer::any_move() const
{
    return any_move_;
}

void FieldPlayer::LinkToCamera(float* roty)
{
    camera_roty_ = roty;
}

void FieldPlayer::UnlinkToCamera()
{
    camera_roty_ = nullptr;
}

void FieldPlayer::UpdateInput(InputManager* input)
{
    input_ = *input;
}

GameLoop::GameLoop(const StagePtr& stage)
    : stage_(stage),
      camera_default_stat({7.0f, 0.8f, 0.0f, 20 * PHI_F / 180, false, {0, 0}}),
      camera(camera_default_stat)
{
}

int GameLoop::Init(const std::string& model_dir_path, std::shared_ptr<CharacterManager> character_manager)
{
    charmgr_ = character_manager;

    auto model_dir_path_ = unicode::ToTString(model_dir_path);
    auto model_path = model_dir_path_ + _T("miku.pmd");

    //MV1SetLoadModelUsePhysicsMode(DX_LOADMODEL_PHYSICS_DISABLE);
    MV1SetLoadModelUsePhysicsMode(DX_LOADMODEL_PHYSICS_LOADCALC);

    myself_ = std::dynamic_pointer_cast<FieldPlayer>(charmgr_->Get(charmgr_->my_character_id()));
    myself_->Init(model_path);

    return 0;
}

int GameLoop::Logic(InputManager* input)
{
    MoveCamera(input);
    myself_->UpdateInput(input);
    for (auto character : charmgr_->GetAll())
    {
        character.second->Update();
    }
    return 0;
}

int GameLoop::Draw()
{
    //std::cout << "\nDraw" << std::endl;

    FixCameraPosition();

    stage_->Draw();
    //myself_->Draw();
    for (auto character : charmgr_->GetAll())
    {
        character.second->Draw();
    }

    return 0;
}

void GameLoop::FixCameraPosition()
{
    if (!camera.manual_control/* && myself_->current_stat().motion != BasicMotion::STAND*/)
    {
        camera = camera_default_stat;
        camera.theta = myself_->current_stat().roty;
    }

    //std::cout << "camera.theta = " << camera.theta << ", roty = " << myself_->current_stat().roty << std::endl;

    const auto target_pos = myself_->current_stat().pos +
        VGet(0, myself_->model_height() * camera.target_height + 0.2f, 0) * stage_->map_scale();
    auto camera_pos = target_pos +
        VGet(cos(camera.phi) * sin(camera.theta),
             sin(camera.phi),
             cos(camera.phi) * cos(camera.theta)) * (camera.radius * stage_->map_scale());

//    const auto coll_info = MV1CollCheck_Line(map_handle, -1, target_pos, camera_pos);
//    if (coll_info.HitFlag)
//    {
//        camera_pos = coll_info.HitPosition;
//    }

    SetCameraPositionAndTarget_UpVecY(camera_pos, target_pos);
}

void GameLoop::MoveCamera(InputManager* input)
{
    const bool prev_right = input->GetPrevMouseRight();
    const bool prev_left = input->GetPrevMouseLeft();
    const bool right = input->GetMouseRight();
    const bool left = input->GetMouseLeft();

    if ((right && !prev_right && !left) || (left && !prev_left && !right))
    {
        // クリックした瞬間
        if (!camera.manual_control)
        {
            camera.manual_control = true;
        }
        camera.manual_control_startpos = input->GetMousePos();
        SetMouseDispFlag(FALSE); // カーソル消去

        if (left)
        {
            myself_->LinkToCamera(&camera.theta);
        }
    }
    else if ((right && prev_right) || (left && prev_left))
    {
        // ドラッグしている
        // assert(camera.manual_control);

        int diff_x, diff_y;
        auto current_pos = input->GetMousePos();
        SetMousePoint(camera.manual_control_startpos.first, camera.manual_control_startpos.second);

        diff_x = current_pos.first - camera.manual_control_startpos.first;
        diff_y = current_pos.second - camera.manual_control_startpos.second;

        camera.theta += diff_x * 0.005f;
        camera.phi += diff_y * 0.005f;
    }
    else
    {
        // 左右ボタンを離した瞬間以降
        myself_->UnlinkToCamera();
        if (!GetMouseDispFlag())
        {
            SetMouseDispFlag(TRUE);
        }
        if (camera.manual_control && myself_->any_move())
        {
            camera.manual_control = false;
        }
    }
}

