#include "CharacterModelManager.hpp"
#include <map>
#include <boost/filesystem.hpp>
#include <boost/format.hpp>
#include <DxLib.h>
#include "../unicode.hpp"
#include "PMDLoader.hpp"

class CharacterModelManager::Impl
{
public:
    void AddSearchDirectoryPath(const std::string& search_dir_path)
    {
        search_dir_paths_.push_back(search_dir_path);
    }

    int GetModelHandle(int model_number)
    {
        namespace fs = boost::filesystem;

        auto cache_it = model_handle_cache_.find(model_number);
        if (cache_it != model_handle_cache_.end())
        {
            return MV1DuplicateModel(cache_it->second);
        }
        else
        {
            auto model_package_path =
                fs::path(GetModelPackageName(model_number) + ".pmd"); // TODO: 現在仮実装なので直す

            for (const auto& search_dir_path : search_dir_paths_)
            {
                auto model_path = fs::path(search_dir_path) / model_package_path;
                if (fs::exists(model_path))
                {
                    //int model_handle = MV1LoadModel(unicode::ToTString(model_path.native()).c_str());

                    std::vector<tstring> basic_motion_paths = {
                        _T("basic_stand.vmd"),
                        _T("basic_walk.vmd"),
                        _T("basic_run.vmd")
                    };
                    PMDLoader loader;
                    int model_handle =
                        loader.Load(unicode::ToTString(model_path.native()), _T(""), basic_motion_paths);
                    if (model_handle != -1)
                    {
                        model_handle_cache_[model_number] = model_handle;
                        return model_handle;
                    }
                    else
                    {
                        throw std::runtime_error("can't load the model package");
                    }
                }
            }

            throw std::runtime_error("can't find the model package");
        }
    }

    std::string GetModelPackageName(int model_number)
    {
        // 仮実装
        // TODO: モデルパッケージとモデル番号の対応はどうやって管理するか考える
        if (model_number == 0)
        {
            return "miku";
        }
        else if (model_number == 1)
        {
            return "yowane_haku";
        }
        throw std::runtime_error(
                (boost::format("unknown model number: %d") % model_number).str());
    }

private:
    std::vector<std::string> search_dir_paths_;
    std::map<int, int> model_handle_cache_;
};

CharacterModelManager::CharacterModelManager()
    : impl_(new Impl())
{}

CharacterModelManager::~CharacterModelManager()
{}

void CharacterModelManager::AddSearchDirectoryPath(const std::string& search_dir_path)
{
    impl_->AddSearchDirectoryPath(search_dir_path);
}

int CharacterModelManager::GetModelHandle(int model_number)
{
    return impl_->GetModelHandle(model_number);
}

std::string CharacterModelManager::GetModelPackageName(int model_number)
{
    return impl_->GetModelPackageName(model_number);
}

CharacterModelManager& CharacterModelManager::GetInstance()
{
    static CharacterModelManager instance;
    return instance;
}
