# -*- coding: utf-8 -*-

$:.unshift File.dirname(__FILE__) + '/../lib'
require 'mint'

Spec::Matchers.define :have_certain_param do |name, value|
  match do |object|
    object.instance_variable_get(name) == value
  end
end

shared_examples_for 'generator' do
  it do
    subject.should_receive(:setup).
      exactly(1).times.with(no_args)
    subject.should_receive(:generate_problem).
      exactly(1).times.and_return('problem')
    subject.should_receive(:teardown).
      exactly(1).times.with('problem')
    subject.stub(:validation_loop) {|r| r }
    subject.generate.should have(1).problems
  end

  context 'options' do

    before do
      @options = { :range => (5..10) }
      @defaults = [
        [:test1, 'test1'],
        [:test2, 'test2'],
        [:test3, 'test3'],
      ]

      klass = subject.class

      @original_options = klass.instance_variable_get(:@default_options)
      klass.instance_variable_set(:@default_options, {})

      klass.__send__(:option, *@defaults[0])
      klass.__send__(:option, *@defaults[1])
      klass.__send__(:option, *@defaults[2])

      subject.__send__(:options=, @options)
    end

    after { subject.class.instance_variable_set(:@default_options, @original_options) }

    context 'target class' do

      context 'set defaults' do
        before { @default_options = subject.class.__send__(:default_options) }
        it { @default_options.should include(*@defaults.map(&:first)) }
      end

      context 'refer' do
        before { @opts = subject.__send__(:options) }
        it { @opts.should include(:range) }
        it { @opts.should include(:test1) }
        it { @opts.should include(:test2) }
        it { @opts.should include(:test3) }
        it { @opts[:range].should == @options[:range] }
        it { @opts[:test1].should == @defaults.assoc(:test1)[1] }
        it { @opts[:test2].should == @defaults.assoc(:test2)[1] }
        it { @opts[:test3].should == @defaults.assoc(:test3)[1] }
      end
    end

    context 'subclass' do

      before(:all) do
        klass = Class.new(subject.class)
        klass_name = "OptionsChild_#{subject.class.to_s.split(/::/).last}"
        unless Mint::Generator.const_defined?(klass_name)
          Mint::Generator.const_set(klass_name, klass)
        end
        @subclass = Mint::Generator.const_get(klass_name)
      end
      after { @subclass.instance_variable_set(:@default_options, {}) }

      it { @subclass.new.__send__(:options).should have(@defaults.size).options }

      context 'set defaults' do
        before do
          @subclass.__send__(:option, :child, 'child')
          @subclass.__send__(:option, :test1, 'child')
          @child_options = @subclass.__send__(:default_options)
          @base_options = subject.class.__send__(:default_options)
        end
        it { @child_options.should_not == @base_options }
        it { @child_options.should include(:child) }
        it { @base_options.should_not include(:child) }
        it { @child_options[:test1].should == 'child' }
      end
    end
  end
end

shared_examples_for 'Arithmetic' do

  def set_options(options = {})
    defaults = {:term_number => 1, :operators => ['@']}
    subject.__send__(:options=, defaults.merge(options))
  end

  before { set_options }

  it_should_behave_like 'generator'

  [
    [' -25', ' (-25)'],
    ['-25', '-25'],
    [' 25', ' 25'],
    ['       -25', '       (-25)'],

  ].each do |from, to|
    it do
      subject.__send__(:teardown, from).should == to
    end
  end

  [
    [:factor, 5],
    [:spec, 10],
    [:hoge, 12],
    [:zero, 0],

  ].each do |position, term_number|
    context position do
      before do
        options = {
          :"#{position}_term_min" => term_number,
          :"#{position}_term_max" => term_number,
        }
        subject.stub(:options => options)
      end
      it { subject.__send__(:term_number, position).should == term_number }
    end
  end

  context 'operand' do

    before(:all) do
      @msg ='This operand need these options:'
      @int = '1'
      subject.stub(:create_integer => @int)
    end

    context 'integer' do
      before { set_options(:min => nil, :max => nil, :minus => nil) }
      it { subject.__send__(:integer).should == @int }

      context 'zero division' do
        before do
          set_options(:min => 0, :max => 0, :minus => false)
          subject.unstub(:create_integer)
          subject.stub(:last_operator => 'div')
        end
        it { subject.__send__(:integer).should_not == '0' }
      end
    end

    context 'decimal' do
      before do
        set_options(
          :min    => nil,
          :max    => nil,
          :minus  => nil,
          :digits => 2
        )
      end
      it { subject.__send__(:decimal).should match(/(?:#{@int}|0)\.\d\d?/) }
    end

    context 'fraction' do
      before do
        set_options(
          :numerator_min   => nil,
          :numerator_max   => nil,
          :denominator_min => nil,
          :denominator_max => nil,
          :minus           => nil
        )
        subject.should_receive(:numerator_part).and_return('5')
        subject.should_receive(:denominator_part).and_return('4')
      end
      it { subject.__send__(:fraction).should == "5/4" }
    end
  end
end

shared_examples_for 'HighOrderExpression' do

  it_should_behave_like 'generator'

  before(:all) { subject.instance_variable_set(:@options, {}) }

  context 'discriminant' do
    [
      [1, 2, 1, 0],
      [1, 4, 1, 12],
      [2, 2, 2, -12],

    ].each do |a, b, c, d|
      it { subject.__send__(:discriminant, a, b, c).should == d }
    end
  end

  context 'factorization' do

    (2..3).each do |i|
      it do
        subject.stub(:create_integer => i)
        subject.should_receive(:"easy_factorization#{i}")
        subject.__send__(:factorization)
      end
    end

    it do
      subject.stub(:create_integer => 0)
      expect { 
        subject.__send__(:factorization)
      }.to raise_error(NotImplementedError)
    end

    context 'easy' do

      before do
        subject.instance_variable_set(:@options, :x => ['x'])
        subject.stub(:create_integer).and_return(2, 3, 4)
      end

      it do
        subject.should_receive(:factorization2).with(1, 5, 6, 'x')
        subject.__send__(:easy_factorization2)
      end

      it do
        subject.should_receive(:factorization3).with(1, 9, 26, 24, 'x')
        subject.__send__(:easy_factorization3)
      end

      context 'factorization2' do

        values = [[2, 3, 4], [1, 3, 4], [-1, 3, 4], [2, 1, 4], [2, -1, 4]]

        values.zip([
          '2x^2 + 3x + 4',
          'x^2 + 3x + 4',
          '-x^2 + 3x + 4',
          '2x^2 + x + 4',
          '2x^2 - x + 4',
        ]).each do |(a, b, c), ptn|
          it ptn do
            expression = subject.__send__(:factorization2, a, b, c, 'x')
            expression.should == ptn
          end
        end

        values.zip([
          '2x^2 + 3xy + 4y^2',
          'x^2 + 3xy + 4y^2',
          '-x^2 + 3xy + 4y^2',
          '2x^2 + xy + 4y^2',
          '2x^2 - xy + 4y^2',
        ]).each do |(a, b, c), ptn|
          it ptn do
            expression = subject.__send__(:factorization2, a, b, c, ['x', 'y'])
            expression.should == ptn
          end
        end
      end

      context 'factorization3' do

        values = [[2, 3, 4, 5], [1, 3, 4, 5], [-1, 3, 4, 5], [2, 1, 4, 5], [2, -1, 4, 5]]

        values.zip([
          '2x^3 + 3x^2y + 4xy^2 + 5y^3',
          'x^3 + 3x^2y + 4xy^2 + 5y^3',
          '-x^3 + 3x^2y + 4xy^2 + 5y^3',
          '2x^3 + x^2y + 4xy^2 + 5y^3',
          '2x^3 - x^2y + 4xy^2 + 5y^3',
        ]).each do |(a, b, c, d), ptn|

          it ptn do
            expression = subject.__send__(:factorization3, a, b, c, d, ['x', 'y'])
            expression.should == ptn
          end
        end
      end
    end

    context 'expansion' do

      context 'one variable' do
        before do
          @single = '(2x + 2)'
          subject.stub(:create_integer => 2)
          subject.stub(:single => @single)
          subject.instance_variable_set(:@options, :x => ['x'])
        end
        it do
          expression = subject.__send__(:expansion)
          expression.should == "#{@single}#{@single}"
        end
        it do
          subject.should_receive(:single).and_return('x', @single)
          expression = subject.__send__(:expansion)
          expression.should == "#{@single}x"
        end
        it do
          subject.should_receive(:single).and_return(@single, 'x')
          expression = subject.__send__(:expansion)
          expression.should == "#{@single}x"
        end
      end
    end
  end
end

module ProblemExamples

  def self.get(*types)
    types.inject([]) {|memo, type|
      memo += __send__(type) }
  end

  private
  def self.existing
    [
      ['7 / 12 + 17 / 12', '7 / 12 + 17 / 12'],
      ['3 / 4 + 1 / 7', '3 / 4 + 1 / 7'],
      ['2 / 9 - 3 / 5', '2 / 9 - 3 / 5'],
      ['2', '2'],
      ['4/11', '4 / 11'],
      ['25/28', '25 / 28'],
      ['-7/12', '-7 / 12'],
      ['-17/45','-17 / 45' ],
      ['(2a + 1)(3a - 2)', '(2 * a + 1) * (3 * a - 2)'],
      ['(x + 2)^2', '(x + 2)^2'],
      ['(3P - 2Q)^3', '(3 * P - 2 * Q)^3'],
      ['1 / (x + 1)(x + 2)', '1 / ((x + 1) * (x + 2))'],
      ['2x^3 + x - 1', '2 * x^3 + x - 1'],
      ['(X^2 - Y^2)', '(X^2 - Y^2)'],
      ['root(2)root(8)', 'sqrt(2) * sqrt(8)'],
      ['sqrt(2)sqrt(8)', 'sqrt(2) * sqrt(8)'],
      ['(1 + root(3))(2 - root(3))', '(1 + sqrt(3)) * (2 - sqrt(3))'],
      ['(1 + sqrt(3))(2 - sqrt(3))', '(1 + sqrt(3)) * (2 - sqrt(3))'],
      ['1 / root(2)', '1 / sqrt(2)'],
      ['1 / sqrt(2)', '1 / sqrt(2)'],
      ['2 / (root(3) - 1)', '2 / (sqrt(3) - 1)'],
      ['2 / (sqrt(3) - 1)', '2 / (sqrt(3) - 1)'],
    ]
  end

  def self.original
    [
      ['(1) - 1', '1 - 1'],
      ['1/(2x^2 + 4) -1 / (2x -1)', '(1/(2 * x^2 + 4)) - 1 / (2 * x - 1)'],
      ['1 + 1', '1 + 1'],
      ['2 - 2', '2 - 2'],
      ['3 * 3', '3 * 3'],
      ['4 / 4', '4 / 4'],
      ['5 div 5', '5 div 5'],
      ['-233/23', '-233 / 23'],
      ['(x^2 + 4)(y^3 - 2)', '(x^2 + 4) * (y^3 - 2)'],
      ['(x^3 + 2)(x^2- 3)', '(x^3 + 2) * (x^2 - 3)'],
      ['2(x^2 + xy + y^2)', '2 * (x^2 + x * y + y^2)'],
      ['sqrt(2(x^(2 + 3) + xyz^5 + y^2))', 'sqrt(2 * (x^(2 + 3) + x * y * z^5 + y^2))'],
      ['sqrt(2(x^(2 + 3) + root(xyz^5) + y^2))', 'sqrt(2 * (x^(2 + 3) + sqrt(x * y * z^5) + y^2))'],
      ['5xc*1div5 + 1 (34 - 1 /8 + 1)', '5 * x * c * 1 div 5 + 1 * (34 - 1 / 8 + 1)'],
      ['2PI', '2 * %pi'],
      ['5.8*5.8*PI', '5.8 * 5.8 * %pi'],
      ['3e * 5x^2', '3 * %e * 5 * x^2'],
      ['3E * 5x^2', '3 * %e * 5 * x^2'],
      ['3 + 2i', '3 + 2 * %i'],
      ['(-24x + 37)(71x - 33)', '(-24 * x + 37) * (71 * x - 33)'],
      ['x^((3y^2+5y)(y^2+2-x))', 'x^((3 * y^2 + 5 * y) * (y^2 + 2 - x))'],
      ['-x^(-(-3y^2+-5y)(y^-2+2-x))', '-x^(-(-3 * y^2 + -5 * y) * (y^-2 + 2 - x))'],
      ['(x^3 + 1)(x^2 + 2)(x + 1)', '(x^3 + 1) * (x^2 + 2) * (x + 1)'],
      ['(x^3 + 1)(x^2 + 2)(x + 1)(y^2 - 11)', '(x^3 + 1) * (x^2 + 2) * (x + 1) * (y^2 - 11)'],
      ['(x^3 + 1)(x^2 + 2)(x + 1)(y^2 - 11)(y^3 - 4)', '(x^3 + 1) * (x^2 + 2) * (x + 1) * (y^2 - 11) * (y^3 - 4)'],
      ['a - 1', 'a - 1'],
      ['6a^2 - a - 1', '6 * a^2 - a - 1'],
      ['5 div 4', '5 div 4'],
      ['5.0 div 4', '5.0 div 4'],
      ['5/4', '5 / 4'],
      ['5.0/4', '5.0 / 4'],
      ['.04324134213', '0.043'],
      ['- .04324134213', '-0.043'],
      ['-50/12 div 12*32', '-50 / 12 div (12 * 32)'],
      ['-sqrt(2)', '-sqrt(2)'],
      ['-(5 - 2x)', '(2x - 5)'],
      # TODO: need more pattern
    ]
  end

  def self.bad_patterns
    [
      '1 + ', ' * 2', ' * ', '4 / 4 5',
      '2(x^2 + xy + y^2',
      'sqrt(2(x^2 + 3) + xyz^5 + y^2))',
      'sqrt(2(x^(2 + ) + rot(xyz^5) + y^2))',
      '5xc*1/5 + 1 (34 - 1 /8 + 1)* / 1',
      # TODO: need more pattern
    ]
  end
end

