# -*- coding: utf-8 -*-

require File.dirname(__FILE__) + '/../spec_helper.rb'

module Mint::Generator

  describe QuadraticEquation do

    subject { QuadraticEquation.new }

    context 'problems' do

      before do
        options = {
          :r_min => 4,  :r_max => 4,
          :p_min => 10, :p_max => 10,
          :q_min => 20, :q_max => 20,
        }
        @problem = subject.generate(options)
      end

      it { @problem.should have(1).problem }
      it { @problem.first.scan('x').should have(2).variables }
      it { @problem.first.split(/[+-]/).should have(3).terms }
    end

    context 'generate expression' do

      before do
        type = :imaginary
        subject.__send__(:options=, {:answer_type => type})
        subject.should_receive(:generate_quadeqn_params).with(type).and_return([3, 2, 1])
        subject.should_receive(:generate_coefficients).with(3, 2, 1).and_return([1, 2, 3])
        subject.should_receive(:term1).with(1).and_return('x^2')
        subject.should_receive(:term2).with(2).and_return(' + 2 * x')
        subject.should_receive(:term3).with(3).and_return(' + 3')
      end

      it { subject.__send__(:expression).should == 'x^2 + 2 * x + 3' }
    end

    context 'generate params' do

      it { subject.__send__(:squares, 0, 10).should == [0, 1, 4, 9, 16, 25, 36, 49, 64, 81, 100] }

      [:rational, :irrational, :imaginary].each do |type|
        it do
          subject.should_receive(:generate_r).and_return(1)
          subject.should_receive(:generate_p).and_return(1)
          subject.should_receive(:"#{type}_q").and_return(1)
          subject.__send__(:generate_quadeqn_params, type).should == [1, 1, 1]
        end
      end

      context 'each params' do

        [
          [[1, 20], [  0, 12], [0, 40]],
          [[1, 10], [-10, 10], [0, 10]],
          [[1,  1], [-10, 10], [0, 10]],
          [[1, 10], [ -1,  1], [0, 10]],
          [[1, 10], [  1,  1], [0, 10]],
          [[1, 10], [  4,  4], [0, 10]],

        ].each do |(r_min, r_max), (p_min, p_max), (q_min, q_max)|

          before do
            @options = {
              :r_min => r_min, :r_max => r_max,
              :p_min => p_min, :p_max => p_max, :p_minus => false,
              :q_min => q_min, :q_max => q_max,
            }
            subject.__send__(:options=, @options)
            @squares = subject.__send__(:squares, @options[:q_min], @options[:q_max])
          end

          it { (@options[:r_min]..@options[:r_max]).should include(subject.__send__(:generate_r)) }
          it { (@options[:p_min]..@options[:p_max]).should include(subject.__send__(:generate_p)) }
          it do
            subject.__send__(:options=, subject.__send__(:options).merge(:p_minus => true))
            (-@options[:p_min]..@options[:p_max]).should include(subject.__send__(:generate_p))
            subject.__send__(:options=, subject.__send__(:options).merge(:p_minus => false))
          end

           it { @squares.should include(subject.__send__(:rational_q)) }
           it { @squares.should_not include(subject.__send__(:irrational_q)) }
           it { (@options[:q_min]..@options[:q_max]).should include(subject.__send__(:irrational_q)) }
           it { (@options[:q_min]..@options[:q_max]).should include(-subject.__send__(:imaginary_q)) }
        end
      end
    end

    context 'generate coefficient' do

      [
        [[1, 1, 1], [1, -2, 0]],
        [[2, 2, 2], [2, -4, 1]],
        [[10, 10, 10], [10, -20, 9]],

      ].each do |params, coefficients|
        it { subject.__send__(:generate_coefficients, *params).should == coefficients }
      end

      context 'each type' do

        def discriminant(a, b, c)
          b**2 - 4 * a * c
        end

        context 'irrational' do
          before  do
            r, p, q = subject.__send__(:generate_quadeqn_params, :irrational)
            @params = subject.__send__(:generate_coefficients, r, p, q)
          end
          it { discriminant(*@params).should > 0 }
          it { Math.sqrt(discriminant(*@params)).should_not be_an_instance_of(Fixnum) }
        end

        context 'rational' do
          before  do
            r, p, q = subject.__send__(:generate_quadeqn_params, :rational)
            @params = subject.__send__(:generate_coefficients, r, p, q)
          end
          it { discriminant(*@params).should > 0 }
          it { (Math.sqrt(discriminant(*@params)) % 1).should == 0 }
        end

        context 'imaginary' do
          before  do
            r, p, q = subject.__send__(:generate_quadeqn_params, :imaginary)
            @params = subject.__send__(:generate_coefficients, r, p, q)
          end
          it { discriminant(*@params).should < 0 }
        end
      end
    end

    context 'generate term' do

      [
        [0,  ''],
        [1,  'x^2'],
        [-1,  '-x^2'],
        [2,  '2 * x^2'],
        [-2,  '-2 * x^2'],
        [10, '10 * x^2'],

      ].each do |a, expr|
        it { subject.__send__(:term1, a).should == expr }
      end

      [
        [0,   ''],
        [1,   ' + x'],
        [-1,  ' - x'],
        [2,   ' + 2 * x'],
        [-2,  ' - 2 * x'],
        [10,  ' + 10 * x'],
        [-10, ' - 10 * x'],

      ].each do |b, expr|
        it { subject.__send__(:term2, b).should == expr }
      end

      [
        [0,   ''],
        [1,   ' + 1'],
        [-1,  ' - 1'],
        [2,   ' + 2'],
        [-2,  ' - 2'],
        [100, ' + 100'],

      ].each do |c, expr|
        it { subject.__send__(:term3, c).should == expr }
      end
    end

    context 'rational' do

      before do
        options = { :answer_type => :rational }
        @problem = subject.generate(options)
      end

      it 'correct answer' do
        problem = Mint::Builder.build(@problem.first)
        query = "solve(#{problem}, x);"
        ans = Mint::Solver::Maxima.exec_maxima(query)
        ans.should_not match(/sqrt/)
      end
    end

    context 'irrational' do

      before do
        options = { :answer_type => :irrational }
        @problem = subject.generate(options)
      end

      it 'correct answer' do
        problem = Mint::Builder.build(@problem.first)
        query = "solve(#{problem}, x);"
        ans = Mint::Solver::Maxima.exec_maxima(query)
        ans.should match(/sqrt/)
      end
    end

    context 'imaginary' do

      before do
        options = { :answer_type => :imaginary }
        @problem = subject.generate(options)
      end

      it 'correct answer' do
        problem = Mint::Builder.build(@problem.first)
        query = "solve(#{problem}, x);"
        ans = Mint::Solver::Maxima.exec_maxima(query)
        ans.should match(/%i/)
      end
    end

    context 'solvable' do

      [
        [[1,  20], [  0,   12], [ 0,  40]],
        [[1,  10], [-10,   10], [ 0,  10]],
        [[1,   1], [-10,   10], [ 0,  10]],
        [[1,  10], [ -1,    1], [ 0,  10]],
        [[1,  10], [  1,    1], [ 0,  10]],
        [[1, 100], [  0,  100], [ 0, 100]],
        [[5, 100], [ 20,  100], [49, 100]],

      ].each do |(r_min, r_max), (p_min, p_max), (q_min, q_max)|

        before do
          options = {
            :r_min => r_min, :r_max => r_max,
            :p_min => p_min, :p_max => p_max,
            :q_min => q_min, :q_max => q_max,
          }
          @problem = Mint::Builder.build(subject.generate(options).first)
          @solver = Mint::Solver::Maxima::Factory.create(:quadratic_equation)
        end

        it { expect { @solver.solve(@problem) }.to_not raise_error }
      end
    end
  end
end

