# -*- coding: utf-8 -*-

require File.dirname(__FILE__) + '/spec_helper.rb'

module Mint

  describe Expression do

    describe 'comparing' do
      context 'equality' do
        ProblemExamples.get(:original, :existing).each do |expression, other|
          it expression do
            expr  = Mint::Builder.build(expression)
            othr = Mint::Builder.build(other)
            expr.should == othr
          end
          it 'use maxima' do
            expr  = Mint::Builder.build(expression)
            othr = Mint::Builder.build(other)
            othr_maxima = othr.to_maxima
            othr.stub(:normalize => '')
            othr.stub(:to_maxima => othr_maxima)
            expr.should == othr
          end
        end
        context 'null expression' do
          ProblemExamples.get(:original, :existing).each do |expression, _|
            context expression do
              subject { Mint::Builder.build(expression) }
              before { @null = Mint::Builder.build(nil) }
              it { subject.should_not == @null }
            end
          end
        end
      end
    end

    describe 'formating' do

      [
        [
          '1 / (x + 4)(x - 2) div 5',
          '1 / ((x + 4) * (x - 2)) / 5',
          '1 / ((x + 4) * (x - 2)) div 5',
          '\frac{1}{(x + 4)(x - 2)} \div 5',
          '\frac{1}{(x + 4)(x - 2)} \divide 5',
        ],
        [
          '25.33 - (-0.99)',
          '25.33 - (-0.99)',
          '25.33 - (-0.99)',
          '25.33 - (-0.99)',
          '25.33 - (-0.99)',
        ],
        [
          '(-83.68) * 10.23',
          '(-83.68) * 10.23',
          '(-83.68) * 10.23',
          '(-83.68) \times 10.23',
          '(-83.68) \times 10.23',
        ],
        [
          'x^2 + 7x - 8',
          'x^2 + 7 * x - 8',
          'x^2 + 7 * x - 8',
          'x^2 + 7x - 8',
          'x^2 + 7x - 8',
        ],
        [
          '3.4 / 54 div 2 / 6',
          '3.4 / 54 / (2 / 6)',
          '3.4 / 54 div (2 / 6)',
          '\frac{3.4}{54} \div \frac{2}{6}',
          '\frac{3.4}{54} \divide \frac{2}{6}',
        ],
        [
          '5 * (-2)',
          '5 * (-2)',
          '5 * (-2)',
          '5 \times (-2)',
          '5 \times (-2)',
        ],
        [
          '5 * (-2x)',
          '5 * ((-2) * x)',
          '5 * ((-2) * x)',
          '5 \times ((-2)x)',
          '5 \times ((-2)x)',
        ],
        [
          '5x * 2y',
          '5 * x * 2 * y',
          '5 * x * 2 * y',
          '5x \times 2y',
          '5x \times 2y',
        ],
        [
          '5x^2 * 2y^3',
          '5 * x^2 * 2 * y^3',
          '5 * x^2 * 2 * y^3',
          '5x^2 \times 2y^3',
          '5x^2 \times 2y^3',
        ],
        [
          '2root(2) * 35',
          '2 * sqrt(2) * 35',
          '2 * sqrt(2) * 35',
          '2\sqrt{2} \times 35',
          '2\sqrt{2} \times 35',
        ],
        [ '2i', '2 * %i', '2 * %i', '2\i', '2\i', ],
        [ '2e', '2 * %e', '2 * %e', '2\e', '2\e', ],
        [ '2pi', '2 * %pi', '2 * %pi', '2\pi', '2\pi', ],
      ].each do |original, maxima, normalize, latex, ascii_math_ml|

        context original do

          before(:all) do
            @original  = original
            @maxima    = maxima
            @latex     = latex
            @normalize = normalize
            @ascii_math_ml = ascii_math_ml
          end

          subject { MintExpressionParser.new.parse(@original) }

          it 'original' do
            subject.to_original.should == @original
          end

          it 'maxima' do
            subject.to_maxima.should == @maxima
          end

          it 'latex' do
            subject.to_latex.should == @latex
          end

          it 'ascii_math_ml' do
            subject.to_ascii_math_ml.should == @ascii_math_ml
          end

          it 'normalize' do
            subject.normalize.should == @normalize
          end

          it 'to_s as normalize' do
            subject.to_s.should == @normalize
          end
        end
      end
    end
  end

  describe NullExpression do

    subject { NullExpression.new }

    before { @other = Expression.new('5', LiteralNode.new(5)) }

    [
      :to_original,
      :to_latex,
      :to_maxima,
      :to_ascii_math_ml,
      :normalize,
    ].each do |meth|
      it { subject.__send__(meth).should be_empty }
    end

    it { subject.==(@other).should be_false }
    it { subject.inspect.should match(/#<Mint::Expression:\d+ ''>/) }
  end
end

