# -*- coding: utf-8 -*-

require File.dirname(__FILE__) + '/spec_helper.rb'

module Mint
  describe Builder do

    subject { Builder }

    describe 'build expression' do

      before(:all) do
        @original = '(x^-2 + 4)(y^3 --2)'
        @normalized = '(x^-2 + 4) * (y^3 - (-2))'
        @latex = '(x^-2 + 4)(y^3 - (-2))'
      end

      subject { Builder.build(@original) }

      it 'original' do
        subject.to_original.should == @original
      end

      it 'normalized' do
        subject.normalize.should == @normalized
      end

      it 'to_s as normalized' do
        subject.to_s.should == @normalized
      end

      it 'latex' do
        subject.to_latex.should == @latex
      end
    end

    describe 'validates expression' do

      context 'valid expression' do
        [
          '1 + 2',
          '2 + 5',
          ' 6+ 5',
          ' 2 *3 ',
          '   2- 2 * 3',
          'x - 2 * 3 +      8',
          '2+3-1',
        ].each do |expr|
          it "without parenthesis #{expr}" do
            expect { subject.build(expr) }.to_not raise_error
          end
        end

        [
          '2 * ( 3 + 5 )',
          '2 * (( 3 + 5 ) - 2)',
          '2 * (3 + 5)',
          ' 2   * (3+5 )',
          '(2 * ( 3 + 5 ) )',
          '     ( 2   * 3   +5)',
          '2 * (3) + 5',
          '(x^3 + 2)(x^2- 3)(x + 5)',
        ].each do |expr|
          it "with parenthesis #{expr}" do
            expect { subject.build(expr) }.to_not raise_error
          end
        end
      end

      context 'invalid expression' do
        [
          '1 +',
          '+ 5',
          '+ 5 *',
          '2 * 3 +',
          '/ 2 * 3',
          '* 2 * 3 +',
          '2 * + 3',
        ].each do |expr|
          it "without parenthesis #{expr}" do
            expect { subject.build(expr) }.to raise_error
          end
        end

        [
          '2 * 3 + 5 )',
          '2 * ( 3 + 5',
          '2 * ( ( 3 + 5 )',
          '2 * ( 3 + 5 ) )',
          '( 2 * 3 + 5',
          '( 2 * 3 +) 5',
          '2 (* 3 + 5)',
          '2 ()* 3 + 5)',
          '2 () * 3 + 5',
        ].each do |expr|
          it "with parenthesis #{expr}" do
            expect { subject.build(expr) }.to raise_error
          end
        end
      end
    end

    context 'null expression' do
      [
        ['empty string', ''],
        ['nil', nil],
      ].each do |name, value|
        context name do
          subject { Builder.build(value) }
          it { subject.should be_an_instance_of(NullExpression) }
        end
      end
    end
  end
end

