# -*- coding: nil -*-

module Mint::Generator

  #
  # 複素数の問題を生成するジェネレータ
  #
  # == オプション
  # [_term_number_]
  #   生成する式の項数を１以上の整数で指定します。
  # [_operators_]
  #   使用可能な演算子を指定します。
  #   *, / の２種類から使用したいものを配列で指定します。
  # [_min_]
  #   生成する式の各項の最小値を０以上の整数で指定します。
  # [_max_]
  #   生成する式の各項の最大値を０以上の整数で指定します。
  #   _min_ よりも小さい値を指定することは出来ません。
  # [_fractional_mode_]
  #   真を指定すると分数形式の項を生成します。
  #
  class ComplexNumberArithmetic < Arithmetic

    private

    option :min,             1
    option :max,             9
    option :fractional_mode, false
    option :operators,       %w[ * / ]

    def setup
      if options[:fractional_mode]
        @options[:operators] = %w[ + - ]
      end
    end

    def operand
      return create_fractional_complex_number if options[:fractional_mode]
      return "#{create_complex_number}^2"     if options[:term_number] == 1
      return conjugate_complex(last_operand)  if last_operator == '/'
      create_complex_number
    end

    def create_complex_number
      real_part      = create_integer(options[:min], options[:max], false)
      imaginary_part = create_integer(options[:min], options[:max], false)
      return create_integer(2, 9, false).to_s if [real_part, imaginary_part].include? 0
      "(#{real_part} #{plus_or_minus} #{imaginary_part.to_s.sub(/\A1\z/, '')}%i)"
    end

    def plus_or_minus
      %w[ + - ].sample
    end

    def create_fractional_complex_number
      "#{create_integer(1, 9, false)} / #{create_complex_number}"
    end

    def conjugate_complex(expression)
      a, op, b = *expression.split
      op = op == '+' ? '-' : '+'
      "#{a} #{op} #{b}"
    end
  end
end

