# -*- coding: utf-8 -*-

module Mint::Generator

  #
  # 全てのジェネレータの基底クラス
  #
  # generate_problem をオーバライドする必要がある。
  #
  class Base

    def validation_add(pattern)
      @validations << pattern
    end

    def initialize(amount = 1) # :nodoc:
      @validations = @@validations[self.class.name] || []
      @amount = amount
      @options = {}
    end

    #
    # 問題を生成する
    #
    def generate(options = {})
      self.options = options
      Array(@amount.times.map { do_generate })
    end

    private

    #
    # バリデーションパターンを追加する
    #
    def self.validation(pattern)
      @@validations[self.name] << pattern
    end
    @@validations = Hash.new {|h, k| h[k] = [] }

    #
    # 問題生成時の前処理
    #
    def setup
    end

    #
    # 問題生成時の後処理
    #
    def teardown(problem)
      problem
    end

    #
    # 問題を実際に生成するメソッド
    # オーバライドして使用する
    #
    def generate_problem
      raise 'you must overwrite generate_problem.'
    end

    #
    # 前処理、問題生成、バリデーション、後処理をするメソッド
    #
    def do_generate
      setup
      result = generate_problem
      unless @validations.size.zero?
        result = validation_loop(result)
      end
      teardown result
    end

    #
    # バリデーションを行うメソッド
    #
    def validation_loop(problem)
      start = Time.now
      until @validations.any? {|ptn| ptn =~ problem }
        problem = generate_problem
        if (Time.now - start) >= 1 # MEMO: Wait 1 second
          raise 'failed to create a correct problem'
        end
      end
      problem
    end

    attr_writer :options

    class << self; attr_accessor :default_options end
    def self.get_default_options
      self.default_options ||= {}
      superclass_default_options = {}
      if self.superclass.respond_to?(:default_options, true)
        superclass_default_options = self.superclass.default_options
      end
      superclass_default_options.merge(self.default_options)
    end
    self.default_options ||= {}

    #
    # デフォルトとマージしたオプションを取り出す
    #
    def options
      self.class.get_default_options.merge(@options)
    end

    #
    # デフォルトオプションを設定する
    #
    def self.option(key, value)
      self.default_options ||= {}
      self.default_options[key] = value
    end
  end
end

