module Mint::Generator

  module Utilities

    #
    # 整数を生成する
    #
    # 引数 _min_, _max_ には生成する整数の最小値と最大値を、
    # _minus_ を真にすると2分の１の確立で負の整数を生成する
    # ブロック引数を指定すると、ブロックの戻り値が真になるまで
    # 生成を繰り返す
    #
    def create_integer(min, max, minus)
      min, max = [min, max].map {|i| i < 0 ? -i : i }
      max = min if max < min
      begin
        n = rand(max+1 - min) + min
      end while block_given? && !yield(n)
      (n * sign(minus))
    end

    def create_integers(amount, min, max, minus)
      amount.times.map { create_integer(min, max, minus) }
    end

    #
    # 真を与えると２分の１の確率で -1 を返す
    #
    def sign(minus)
      if minus
        rand(2) == 0 ? 1 : -1
      else
        1
      end
    end

    #
    # オプションを利用して整数を生成する
    #
    # 引数 _target_ に応じて、 min, max, minus
    # を _options_ から取り出して利用する
    # (ex. target='example' #=> options[:example_min]
    #
    def factor(options, target = 'factor')
      create_integer(options[:"#{target}_min"], options[:"#{target}_max"], options[:"#{target}_minus"])
    end

    #
    # (ax + by) の形の式を生成する
    #
    # 引数 _v_ に文字列を与えると (ax + b) の形、
    # 配列を与えると (ax + by) の形の式を生成する
    #
    def single(a, b, v = 'x')
      v = Array(v)
      return v[0]   if b == 0
      return b.to_s if a == 0
      x, y = v.values_at(0, 1)
      ("(%d%s + %d%s)" % [a, x, b, y]).
        gsub(%r!\+ -!, '- ').
        gsub(/([^\d])1(#{v.join('|')})/, '\1\2')
    end

    #
    # 引数 _targets_ の各値に応じて、
    # _options_ から min, max を取り出し
    # max が min より小さければ min の
    # 値に揃える
    #
    def min_max_order(options, targets)
      targets.each do |type|
        min = options[:"#{type}_min"]
        if min > options[:"#{type}_max"]
          options[:"#{type}_max"] = min
        end
      end
      options
    end
  end
end

class Array
  unless public_method_defined?(:sample)
    def sample(n = nil)
      if n
        shuffle.slice(0..(n-1))
      else
        choice
      end
    end
  end
end

class Integer

  # MEMO: only run correctly on positive number
  def factorize
    raise ZeroDivisionError if self.zero?
    return [] if self < 2

    ans = []
    n = self

    [2, 3].each do |i|
      c = 0
      while n % i == 0
        c += 1
        n /= i
      end
      ans << [i, c] if c > 0
    end
 
    i   = 5
    add = 2
    loop do
      c = 0
      while n % i == 0
        c += 1
        n /= i
      end
      ans << [i, c] if c > 0
      i += add
      break if n < i ** 2
      add = 6 - add
    end
    ans << [n, 1] if n > 1
    ans
  end
end

