# -*- coding: nil -*-

module Mint::Generator

  #
  # 無理数の四則演算を生成するジェネレータ
  #
  # == オプション
  # [_term_number_]
  #   生成する式の項数を１以上の整数で指定します。
  # [_operators_]
  #   使用可能な演算子を指定します。
  #   +, - の２種類から使用したいものを配列で指定します。
  # [_min_]
  #   生成する式の各項の最小値を０以上の整数で指定します。
  # [_max_]
  #   生成する式の各項の最大値を０以上の整数で指定します。
  #   _min_ よりも小さい値を指定することは出来ません。
  # [_upper_limit_]
  #    すべてが数字だった場合は強制的に１つ根号を追加します。
  #    そのときの根号内の数字の最大値を2以上の整数で指定します。
  #    なお、最小値は自動的に２に設定されます。
  # [_single_term_min_]
  #   生成する式の項数の最小値を１以上の整数で指定します。
  # [_single_term_max_]
  #   生成する式の項数の最大値を１以上の整数で指定します。
  #   _single_term_min_ より小さい値は指定出来ません。　
  # [_use_coefficient_]
  #   真を指定すると根号を含む項の係数を生成します。
  # [_use_power_]
  #   真を指定すると２乗項を生成します。
  #
  class SquareRootArithmetic < Arithmetic

#     root_pattern = /[2-9]?sqrt\((?:[2-9]|\d\d)\)/
#     term_pattern = /#{root_pattern} [+\-] #{root_pattern}/
#     validation /\A\(#{root_pattern}\)\^2\z/
#     validation /\A#{term_pattern}\z/
#     validation /\A\(#{term_pattern}\)(?:\^2)? [+\-] \(#{term_pattern}(?:\^2)?\)\z/
#     validation /\A\(#{term_pattern}\) [\*\/] \(#{term_pattern}\)\z/
#     validation /\A#{root_pattern} \* #{root_pattern}\z/

    private

    option :min,             0
    option :max,             20
    option :upper_limit,     20
    option :operators,       %w[ * ]
    option :use_power,       false
    option :use_coefficient, false
    option :single_term_min, 2
    option :single_term_max, 2

    def operand
      results = square_root_expression_parts
      result = results[0..-2].join(' ')
      result = "(#{result})" if results.size > 2
      result = "#{result}^2" if options[:use_power] && rand(2) == 0
      result
    end

    def swap_options_with(temp)
      temp_options = options.dup
      self.options = temp_options.merge(temp)
      yield
      self.options = temp_options
    end

    def have_common_prime_divisor?(numbers)
      return true unless numbers.size == 2
      return true if numbers.include?(0)
      return true if numbers.any? {|n| n < 10 }
      prime_divisors = numbers.map {|num| num.factorize }
      prime_divisors[0].each do |hoge|
        prime_divisors[1].each do |piyo|
          commons = hoge & piyo
          return true unless commons.empty? || commons.first == 1
        end
      end
      false
    end

    def square_root_expression_parts
      term_number = term_number(:single)
      results = []
      if options[:use_power]
        begin
          numbers = Array.new(term_number) { create_integer(options[:min], options[:max], false) }
        end until numbers.size == 1 || have_common_prime_divisor?(numbers)
        results = numbers.map {|number| create_square_root_number(number) }
      else
        results = Array.new(term_number) { create_square_root_number }
      end
      results.sort!
      if results.all? {|r| /sqrt/ !~ r }
        results.delete_at rand(results.length)
        swap_options_with(:min => 2, :max => options[:upper_limit]) do
          results << create_square_root_number
        end
      end
      operators = term_number.times.map { %w[ + - ].sample }
      results.zip(operators).flatten
    end

    def create_square_root_number(number = nil)
      if number.nil?
        number = create_integer(options[:min], options[:max], false)
      end
      "#{coefficient(number)}#{root(number)}"
    end

    def root(number)
      zero_or_one?(number) ? '' : "sqrt(#{number})"
    end

    def coefficient(number)
      if options[:use_coefficient] || zero_or_one?(number)
        return create_integer(2, 9, false)
      end
      nil
    end

    def zero_or_one?(number)
      number == 0 || number == 1
    end
  end
end

