package info.projectkyoto.mms.assetmanager;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteDatabase.CursorFactory;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;
import android.net.Uri;
import android.os.Environment;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;

/**
 *
 * @author kobayasi
 */
public class AssetDB extends SQLiteOpenHelper{
    public static final int DB_VERSION = 3;
    public final File contentDir;
    public final File archiveDir;
    final Context context;
    SQLiteStatement insertArchiveStatement;
    final byte[] buf = new byte[1024 * 16];
    private static final Logger logger = Logger.getLogger(AssetDB.class.getName());
    public AssetDB(Context context) {
        super(context, context.getExternalFilesDir(null).getAbsolutePath() + "/assets.db", null, DB_VERSION);
        contentDir = context.getExternalFilesDir("content");
        archiveDir = context.getExternalFilesDir("archive");
        this.context = context;
    }
    @Override
    public void onCreate(SQLiteDatabase db) {
        db.execSQL("CREATE TABLE archive ("
                    + "_id  INTEGER PRIMARY KEY AUTOINCREMENT,"
                    + "fileName TEXT,"
                    + "size INTEGER);");
        db.execSQL("CREATE UNIQUE INDEX idx_fileName on archive(fileName);");
        db.execSQL("CREATE TABLE zipEntry ("
                    + "_id  INTEGER PRIMARY KEY AUTOINCREMENT,"
                    + "archiveId INTEGER,"
                    + "entryName,"
                    + "size INTEGER"
                    + ");");
        db.execSQL("CREATE INDEX idx_entryName on zipEntry(archiveId, entryName);");
    }

    @Override
    public void onOpen(SQLiteDatabase db) {
        if (!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
            throw new IllegalStateException("sdcard is not mounted.");
        }
        try {
            contentDir.mkdirs();
            archiveDir.mkdirs();
            File nomedia = new File(context.getExternalFilesDir(null),".nomedia");
            nomedia.createNewFile();
        } catch(IOException ex) {
            throw new IllegalStateException(ex);
        }
        super.onOpen(db);
    }

    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        logger.info("Upgrade "+oldVersion+" to "+newVersion);
        db.execSQL("DROP TABLE zipEntry;");
        db.execSQL("DROP TABLE archive;");
        onCreate(db);
    }
    public int deleteArchive(long archiveId) {
        return deleteArchive(getWritableDatabase(), archiveId);
    }
    private int deleteArchive(SQLiteDatabase db, long archiveId) {
        logger.info("deleteArchive");
        db.execSQL("DELETE FROM archive WHERE _id = ?", new Object[]{archiveId});
        Cursor cursor = null;
        try {
            cursor = db.query("zipEntry", new String[]{"_id"}, "archiveId = ?", new String[]{Long.toString(archiveId)}, null, null, null);
            if (!cursor.moveToFirst()) {
                return 0;
            }
            do {
                long id = cursor.getLong(0);
                String hexId = Long.toHexString(cursor.getLong(0));
                File file = new File(contentDir, hexId);
                file.delete();
                db.delete("zipEntry", "_id = ?", new String[]{Long.toString(id)});
            }while(cursor.moveToNext());
            return 1;
        }finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }
    private File copyToFile(Uri uri) throws IOException  {
        logger.info("copyToFile");
        File tempDir = context.getExternalCacheDir();
        File tempFile = File.createTempFile("archive", "tmp", tempDir);
        BufferedInputStream is = null;
        BufferedOutputStream os = null;
        try {
            is = new BufferedInputStream(context.getContentResolver().openInputStream(uri));
            os = new BufferedOutputStream(new FileOutputStream(tempFile));
            for(;;) {
                int len = is.read(buf);
                if (len == -1) {
                    break;
                }
                os.write(buf,0,len);
            }
        }finally {
            if (is != null) {
                is.close();
            }
            if (os != null) {
                os.close();
            }
        }
        return tempFile;
    }
    public long addZipFile(Uri uri) {
        logger.info("addZipFile");
        BufferedOutputStream os = null;
        SQLiteDatabase db = getWritableDatabase();
        db.beginTransaction();
        Cursor cursor = null;
        List<File> fileList = new ArrayList<File>();
        File tempFile = null;
        ZipFile zipFile = null;
        try {
            tempFile = copyToFile(uri);
            String fileName = uri.getLastPathSegment();
            String[] columns = {"_id","fileName","size"};
            String where = "fileName = ?";
            String params[] = {fileName};
            cursor = db.query("archive", columns, where, params,null,null,null);
            if (cursor.moveToFirst()) {
                deleteArchive(db, cursor.getLong(0));
            }
            ContentValues cv = new ContentValues();
            cv.put("fileName", fileName);
            cv.put("size", tempFile.length());
            long archiveId = db.insert("archive",null,cv);
            logger.info("open archive");
            logger.info("tempFile = "+tempFile.getAbsolutePath());
            zipFile = new ZipFile(tempFile, "MS932");
            Enumeration<ZipArchiveEntry> entries = zipFile.getEntries();
            while(entries.hasMoreElements()) {
                ZipArchiveEntry ze = entries.nextElement();
                logger.info("entries.nextElement");
                if (ze.isDirectory()) {
                    logger.info("isDirectory");
                    continue;
                }
                cv.clear();
                cv.put("entryName", ze.getName());
                cv.put("archiveId", archiveId);
                cv.put("size", ze.getSize());
                long id = db.insert("zipEntry", null, cv);
                File contentFile = new File(contentDir, Long.toHexString(id));
                fileList.add(contentDir);
                os = new BufferedOutputStream(new FileOutputStream(contentFile));
                InputStream is = zipFile.getInputStream(ze);
                for(;;) {
                    int length = is.read(buf);
                    if (length == -1) {
                        break;
                    }
                    os.write(buf,0, length);
                }
                os.close();
                os = null;
            }
            db.setTransactionSuccessful();
            fileList.clear();
            return archiveId;
        } catch (IOException ex) {
            Logger.getLogger(AssetDB.class.getName()).log(Level.SEVERE, null, ex);
            throw new IllegalArgumentException(ex);
        } finally {
            if (zipFile != null) {
                try {
                    zipFile.close();
                } catch(Exception ex) {
                    logger.log(Level.WARNING,"error",ex);
                }
            }
            if (os != null) {
                try {
                    os.close();
                } catch(Exception ex) {
                    logger.log(Level.WARNING,"error",ex);
                }
            }
            if (cursor != null) {
                cursor.close();
            }
            db.endTransaction();
            for(File file :fileList) {
                file.delete();
            }
            if (tempFile != null) {
                tempFile.delete();
            }
//            db.close();
        }
    }
    public boolean checkCondition() {
        try {
            SQLiteDatabase db = getReadableDatabase();
            db.execSQL("SELECT _id FROM archive WHERE _id = -1");
            return true;
        } catch(Exception ex) {
            logger.log(Level.WARNING,"checkCondition:false",ex);
        }
        return false;
    }
}
