/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx40layer;

import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidiUI;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX40Layer {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX40Layer.class);

    public static final int MOD_NONE = 0;
    public static final int MOD_ASFROM = 1;
    public static final int MOD_FIXED = 2;
    
    public static String modName(int mod) {
        switch(mod) {
            case MOD_NONE: return "mod-none";
            case MOD_ASFROM: return "mod-asfrom";
            case MOD_FIXED: return "mod-fixed";
        }
        return "mod-unknown";
    }
    
    public static MXWrapList createSendOption(boolean haveNoneOption) {
        MXWrapList<Integer> list = new MXWrapList();
        list.addNameAndValue("Fixed", MOD_FIXED);
        list.addNameAndValue("AsFrom", MOD_ASFROM);
        if (haveNoneOption) {
            list.addNameAndValue("none", MOD_NONE);
        }
        return list;
    }
    
    public String _title = "New Layer";

    public int _modPort = MOD_ASFROM;
    public int _fixedPort;

    public int _modChannel = MOD_ASFROM;
    public int _fixedChannel;

    public int _modBank = MOD_ASFROM;
    public int _fixedBankMSB;
    public int _fixedBankLSB;

    public int _modProgram = MOD_ASFROM;
    public int _fixedProgram;

    public int _adjustPan = 64;
    
    public int _adjustTranspose;
    public int _adjustVelocity;
    public int _adjustExpression;

    public int _acceptKeyLowest = 0;
    public int _acceptKeyHighest = 127;
    
    public boolean _disabled = false;
    
    public MX40Group _parentGroup;
    public MX40Process _process;
   
    private MXNoteOffWatcher _noteOff = new MXNoteOffWatcher();
    
    public MX40Layer(MX40Process process, MX40Group parentGroup) {
        _process = process;
        _parentGroup = parentGroup;
    }
    
    public String toString() {
        StringBuffer str = new StringBuffer();
        if (_modChannel == MOD_FIXED) {
            str.append("[Channel=").append(MXMidiUI.nameOfPort(_fixedPort)).append("/").append(_fixedChannel+1).append("]");
        }
        if (_modBank == MOD_FIXED) {
            str.append("[Bank=").append(MXUtil.toHexFF(_fixedBankMSB));
            str.append(":").append(MXUtil.toHexFF(_fixedBankLSB)).append("]");
        }else if (_modBank == MOD_ASFROM) {
            str.append("[Bank=AsFrom]");
        }
        if (_modProgram == MOD_FIXED) {
            str.append("[Program=").append(_fixedProgram).append("]");
        }else if (_modProgram == MOD_ASFROM) {
            str.append("[Program=AsFrom]");
        }
        if (_acceptKeyLowest != 0 || _acceptKeyHighest != 127) {
            str.append("[Note=");
            str.append(MXMidiUI.nameOfNote(_acceptKeyLowest));
            str.append("-");
            str.append(MXMidiUI.nameOfNote(_acceptKeyHighest));
            str.append("]");
        }
        if (_adjustTranspose != 0) {
            str.append("[Transpose=");
            str.append(_adjustTranspose);
            str.append("]");
        }
        
        return str.toString();
    }

    public boolean processByLayer(MXMessage message) {
        if (message.isMessageTypeChannel() == false) {
            _process.sendToNext(message);
            return true;
        }

        boolean changed = false;
        
        int port = message.getPort();
        int command = message.getCommand();
        int channel = message.getChannel();
        int data1 = message.getData1FromBytes();
        int data2 = message.getData2FromBytes();
        
        if (command == MXMidi.COMMAND_PROGRAMCHANGE) {
            processProgramChange(message);
            return true;
        }
        
        int port_trans = port;
        int command_trans = command;
        int channel_trans = channel;
        int data1_trans = data1;
        int data2_trans = data2;

        if (command == MXMidi.COMMAND_NOTEOFF) {
            _noteOff.notifyNoteOffEvent(port, channel, data1, "@2");
            return true;
        }

        if (_modPort == MOD_FIXED) {
            changed = true;
            port_trans = _fixedPort;
        }
        
        if (_modChannel == MOD_FIXED) {
            changed = true;
            channel_trans = _fixedChannel;
        }

        if (command == MXMidi.COMMAND_NOTEON || command == MXMidi.COMMAND_NOTEOFF || command == MXMidi.COMMAND_POLYPRESSURE) {
            if (_adjustTranspose != 0) {
                data1_trans = data1 + _adjustTranspose;
                changed = true;
            }
            if (command == MXMidi.COMMAND_NOTEON && _adjustVelocity != 0) {
                data2_trans = data2 + _adjustVelocity;
                if(data2_trans < 1) data2_trans  = 1;
                if(data2_trans > 127) data2_trans = 127;
                changed = true;
            }
        }

        if (command == MXMidi.COMMAND_NOTEON) {
            MXMessage target = MXMessageFactory.fromShortMessage(port_trans, MXMidi.COMMAND_NOTEOFF + channel_trans, data1_trans, 0);
            _noteOff.addListener(message, target, new MXNoteOffWatcher.Handler() {
                public void onNoteOffEvent(MXMessage target) {
                    _process.sendToNext(target);
                }
            }, "@2");
        }

        if(data1_trans < 0) return true;
        if(data1_trans > 128) return true;

        if(data1_trans < _acceptKeyLowest) return true;
        if(data1_trans > _acceptKeyHighest) return true;

        /* TODO
        if (command == MXMidi.COMMAND_CONTROLCHANGE && message.getAsCCType() == MXMidi.DATA1_CC_EXPRESSION) {
            if (adjustExpression != 100) {
                double exp = message.getAsValue();
                exp = exp * adjustExpression;
                exp = exp / 100;
                int iexp = (int)exp;
                if (iexp < 0) iexp = 0;
                if (iexp > 127) iexp = 127;
                data2 = iexp;
                changed = true;
            }
        }
        if (command == MXMidi.COMMAND_CONTROLCHANGE && message.getAsCCType() == MXMidi.DATA1_CCPANPOT) {
            int x = message.getAsValue();
            int y = x + adjustPan;
            if (y < 0) {
                y = 0;
            }
            if (y > 127) {
                y = 127;
            }
            data2 = y;
            changed = true;
        }*/

        if (changed) {
            MXMessage message2 = MXMessageFactory.fromShortMessage(port_trans, channel_trans + command, data1_trans, data2_trans);
            message = message2;
        }

        _process.sendToNext(message);

        return true;
    }
    
    public void processProgramChange(MXMessage prevMsg) {
        int port = prevMsg.getPort();
        int channel = prevMsg.getChannel();

        MXChannelInfo e = _process._inputInfo.getChannelInfo(port, channel);
        int bankMSB = e._havingBank ? e._bankMSB : -1;
        int bankLSB = e._havingBank ? e._bankLSB : -1;
        int program =  e._havingProgram ? e._program : -1;
        
        if (_modPort == MOD_FIXED) {
            port = _fixedPort;
        }

        if (_modChannel == MOD_FIXED) {
            channel = _fixedChannel;
        }

        if (_modBank == MOD_ASFROM) {
            if (bankMSB >= 0 && bankLSB >= 0) {
                MXMessage msb = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CCBANKSELECT, bankMSB);
                MXMessage lsb = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CCBANKSELECT + 32 , bankLSB);

                _process.sendToNext(msb);
                _process.sendToNext(lsb);
            }
        }else if (_modBank == MOD_FIXED) {
            MXMessage msb  = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CCBANKSELECT, _fixedBankMSB);
            MXMessage lsb  = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + channel, MXMidi.DATA1_CCBANKSELECT + 32, _fixedBankLSB);

            _process.sendToNext(msb);
            _process.sendToNext(lsb);
        }

        if (_modProgram == MOD_ASFROM  && program >= 0) {
            MXMessage message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_PROGRAMCHANGE + channel, program, 0);

            _process.sendToNext(message);
        }else if (_modProgram == MOD_FIXED) {
            MXMessage message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_PROGRAMCHANGE + channel, _fixedProgram, 0);

            _process.sendToNext(message);
        }

        // TODO places ok ?
        /*

        if (adjustExpression != 100) {
            double exp = e._infoExpression;
            exp = exp * adjustExpression * 0.01;
            int data2_exp = (int)exp;
            int command = MXMidi.COMMAND_CONTROLCHANGE;
            int data1_cc = MXMidi.DATA1_CC_EXPRESSION;
            if (data2_exp < 0) data2_exp = 0;
            if (data2_exp > 127) data2_exp = 127;
            MXMessage message = MXMessageFactory.fromShortMessage(port, command + channel, data1_cc, data2_exp);
            _process.sendToNext(message);
        }

        //TODO places ok ?
        if (adjustPan != 0) {
            int command = MXMidi.COMMAND_CONTROLCHANGE;
            int data1_cc = MXMidi.DATA1_CCPANPOT;
            int data2_value = e._infoPan + adjustPan;
            if (data2_value < 0) data2_value = 0;
            if (data2_value > 127) data2_value = 127;
            MXMessage message = MXMessageFactory.fromShortMessage(port, command + channel, data1_cc, data2_value);
            _process.sendToNext(message);
        }*/
    }
    
    public boolean equals(Object o) {
        MX40Layer target = (MX40Layer)o;
        if (!_title.equals(target._title)) {
            return false;
        }
        if (_modPort != target._modPort || _fixedPort != target._fixedPort) {
            return false;
        }
        if (_modChannel != target._modChannel && _fixedChannel != target._fixedChannel) {
            return false;
        }
        if (_modBank != target._modBank || _fixedBankMSB != target._fixedBankMSB || _fixedBankLSB != target._fixedBankLSB) {
            return false;
        }
        if (_modProgram != target._modProgram && _fixedProgram != target._fixedProgram) {
            return false;
        }
        if (_adjustPan == target._adjustPan
         && _adjustTranspose == target._adjustTranspose
         && _adjustVelocity == target._adjustVelocity
         && _adjustExpression == target._adjustExpression
         && _acceptKeyLowest == target._acceptKeyLowest
         && _acceptKeyHighest == target._acceptKeyHighest) {
            return true;
        }
        return false;
    }
}
