/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller.ui;

import java.awt.Color;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import javax.swing.JLabel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.swing.MXSliderUIForTablet;
import jp.synthtarou.midimixer.libs.swing.MXSpinnerMouseExt;
import jp.synthtarou.midimixer.libs.swing.MXFocusAble;
import jp.synthtarou.midimixer.libs.swing.MXFocusGroupElement;
import jp.synthtarou.midimixer.mx30controller.MX32PageProcess;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MGSlider extends javax.swing.JPanel implements MXFocusAble {
    private static final MXDebugConsole _debug = new MXDebugConsole(MGSlider.class);

    MX32PageProcess _process;

    final int _row;
    final int _column;
    
    boolean _disconnectMoment = false;
    
    public MGSlider(MX32PageProcess process, int row, int col) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            _debug.println("SwingUtilities.isEventDispatchThread() == false");
        }
        _process = process;
        _row = row;
        _column = col;
        initComponents();
        new MXSliderUIForTablet(jSliderValue);
        jSpinnerValue.setVisible(false);

        new MXSpinnerMouseExt(MGSlider.this, jSpinnerValue);

        updateUI();
    }

    public void updateUI() {
        super.updateUI();
        if (_process != null && _process._data != null) {
            MXUIValue status = _process._data.getSliderStatus(_row, _column);
            jSliderValue.setMinimum(status._uiValueMinimum);
            jSliderValue.setMaximum(status._uiValueMaximum);
            jSliderValue.setInverted(status._uiValueInvert);
            jSliderValue.setPaintLabels(true);
            jSliderValue.setValue(status.getValue());
            jSpinnerValue.setModel(new SpinnerNumberModel(status.getValue(), status._uiValueMinimum, status._uiValueMaximum, 1));
            jLabel1.setText(status._name);
            focusStatusChanged(false);
        }
    }

    ArrayList<MouseListener> backupedListener = new ArrayList();
    
    public void setValueChangeable(boolean usable) {
        if (usable == false) {            
            for (MouseListener l : jSliderValue.getMouseListeners()) {
                if (l instanceof MXFocusGroupElement.Listen) {
                    continue;
                }else {
                    backupedListener.add(l);
                }
            }
            for (MouseListener l : backupedListener) {
                jSliderValue.removeMouseListener(l);
            }
        }else {
            for (MouseListener l : backupedListener) {
                jSliderValue.addMouseListener(l);
            }
            backupedListener.clear();
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jSliderValue = new javax.swing.JSlider();
        jSpinnerValue = new javax.swing.JSpinner();
        jLabel1 = new javax.swing.JLabel();

        setLayout(new java.awt.GridBagLayout());

        jSliderValue.setOrientation(javax.swing.JSlider.VERTICAL);
        jSliderValue.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderValueStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weighty = 1.0;
        add(jSliderValue, gridBagConstraints);

        jSpinnerValue.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSpinnerValueStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        add(jSpinnerValue, gridBagConstraints);

        jLabel1.setText("-");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        add(jLabel1, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void jSliderValueStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderValueStateChanged
        int newValue = jSliderValue.getValue();
        jSpinnerValue.setValue(newValue);
        if (_updating == newValue) {
            return;
        }
        _process._data.doUpdateSlider(_row, _column, newValue, false);
    }//GEN-LAST:event_jSliderValueStateChanged

    private void jSpinnerValueStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSpinnerValueStateChanged
        Object value = jSpinnerValue.getValue();
        if (value instanceof Number) {
            int newValue = ((Number)value).intValue();
            jSliderValue.setValue(newValue);
            if (_updating == newValue) {
                return;
            }
            _process._data.doUpdateSlider(_row, _column, newValue, false);
        }
    }//GEN-LAST:event_jSpinnerValueStateChanged

    int _updating = -1;

    public void updateUIOnly(final int newValue) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    updateUIOnly(newValue);
                }
            });
            return;
        }
        
        _updating = newValue;

        MXUIValue status = _process._data.getSliderStatus(_row, _column);
        status.setValue(newValue);
        if (jSliderValue.getValue() != newValue) {
            jSliderValue.setValue(newValue);
        }
        try {
            if ((int)jSpinnerValue.getValue() != newValue) {
                jSpinnerValue.setValue(newValue);
            }
        }catch(Exception e) {
            e.printStackTrace();
        }

        _updating = -1;
    }

    public JLabel labelFor(int num, int max) {
        String name = "";
        if (max >= 256) {
            int msb = num / 128;
            int cut = msb * 128;
            int lsb = num - cut;
            name = MXUtil.toHexFF(msb) + ":" + MXUtil.toHexFF(lsb);
        }else {
            name = MXUtil.toHexFF(num);
        }
        JLabel label = new JLabel(name);
        //label.setFont(new Font(Font.SANS_SERIF, Font.ITALIC, 10));
        return label;
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel jLabel1;
    private javax.swing.JSlider jSliderValue;
    private javax.swing.JSpinner jSpinnerValue;
    // End of variables declaration//GEN-END:variables

    public void setBackground(Color col) {
        super.setBackground(col);
        if (jLabel1 != null) {
            jLabel1.setBackground(col);
        }
        if (jSliderValue != null) {
            jSliderValue.setBackground(col);
        }
    }
    
    public void setDefaultBackground() {
        int block = _column / 4;
        boolean toggle = (block % 2 == 0) ? true : false;
        if (toggle) {
            setBackground(MXStatic.colorSlider1);
        }else {
            setBackground(MXStatic.colorSlider2);
        }
    }

    @Override
    public void focusStatusChanged(boolean flag) {
        if (flag) {
            _process._parent.updateFocusInformation(MXUIValue.TYPE_SLIDER,  _process._port, _row, _column);
        }else {
            setDefaultBackground();
        }
    }

    @Override
    public void focusedMousePressed() {
        if (_process._parent._editingControl) {
            _process._parent.enterEditMode(false);
            focusStatusChanged(false);
            editContoller();
        }
    }

    @Override
    public void focusChangingValue() {
        _process._parent.updateFocusInformation(MXUIValue.TYPE_SLIDER, _process._port, _row, _column);
    }

    public void increment() {
        MXUIValue status = _process._data.getSliderStatus(_row, _column);
        int newValue = status.getValue() + 1;
        if (newValue > status._uiValueMaximum) {
            newValue = status._uiValueMaximum;
        }
        if (newValue != status.getValue()) {
            _process._data.doUpdateSlider(_row, _column, newValue, false);
        }
    }
    
    public void decriment() {
        MXUIValue status = _process._data.getSliderStatus(_row, _column);
        int newValue = status.getValue() - 1;
        if (newValue < status._uiValueMinimum) {
            newValue = status._uiValueMinimum;
        }
        if (newValue != status.getValue()) {
            _process._data.doUpdateSlider(_row, _column, newValue, false);
        }
    }
    
    public void beHomePosition() {
        final MXUIValue status = _process._data.getSliderStatus(_row, _column);
        final int current = status.getValue();
        final int value = status._uiValueDefault;
        Thread t = new Thread(new Runnable() {
            public void run() {
                try {
                    for (int i = 0; i < 5; ++ i) {
                        int x = current * (5 - i) + value * i;
                        x /= 5;
                        Thread.sleep(70);
                        status.setValue(x);
                        updateUIOnly(x);
                        if (x == value) { 
                            break;
                        }
                    }
                }catch(Exception e) {
                    e.printStackTrace();
                }finally {
                    status.setValue(value);
                    updateUIOnly(value);
                }
            }
        });
        t.start();
    }

    public void editContoller() {
        _process._parent.enterEditMode(false);
        MXUIValue status = _process._data.getSliderStatus(_row, _column);
        MXUIValueConfig config = new MXUIValueConfig(status);
        MXUtil.showAsDialog(this, config, "Edit " + ui.toString());
        if (config._okOption) {
            MXMessage message = MXMessageFactory.fromDtext(config._textAfter);
            message.setPort(_process._port);
            if (message.isMessageTypeChannel()) {
                message.setChannel(config._channelAfter);
            }
            if (message.hasGateField()) {
                message.setGate(config._gateAfter);
            }
            status.setMonitoring(message);
            status._name = config._nameAfter;
            jLabel1.setText(status._name);
        }
    }
}
