/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.util.ArrayList;
import javax.swing.JComponent;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidiUI;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingNode;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;
import jp.synthtarou.midimixer.mx30controller.ui.MXUIValue;
import jp.synthtarou.midimixer.mx40layer.MXChannelInfoReceiver;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX32PageProcess extends MXReceiver implements MXSettingTarget {
    public final int _port;
    public final MX30Process _parent;
    public final MX32PageData _data;
    public final MX32PageView _view;
    MXChannelInfoReceiver _outputInfo;
    MXNoteOffWatcher _noteOff;
    MXSetting _setting;
    
    public MX32PageProcess(MX30Process parent, int port) {
        _parent = parent;
        _port = port;
        _data = new MX32PageData(this);
        _view = new MX32PageView(this);
        _setting = new MXSetting("Mixing" + (port+1));
        _setting.setTarget(this);
    }

    public void readSettings() {
        _setting.readFile();
    }

    @Override
    public String getReceiverName() {
        return "IN " + MXMidiUI.nameOfPort(_port);
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (usingThis() == false) { sendToNext(message); return; }
        if (_data.catchMessage(message, false) == false) {
            sendToNext(message);
        }
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("Circle[].name");
        setting.register("Circle[].note");
        setting.register("Circle[].type");
        setting.register("Circle[].row");
        setting.register("Circle[].column");
        setting.register("Circle[].message");
        setting.register("Circle[].channel");
        setting.register("Circle[].gate");
        setting.register("Circle[].value");
        setting.register("Circle[].valuemin");
        setting.register("Circle[].valuemax");
        setting.register("Circle[].valueinvert");
        setting.register("Circle[].attributes");
        setting.register("Slider[].name");
        setting.register("Slider[].note");
        setting.register("Slider[].type");
        setting.register("Slider[].row");
        setting.register("Slider[].column");
        setting.register("Slider[].message");
        setting.register("Slider[].channel");
        setting.register("Slider[].gate");
        setting.register("Slider[].value");
        setting.register("Slider[].valuemin");
        setting.register("Slider[].valuemax");
        setting.register("Slider[].valueinvert");
        setting.register("Slider[].attributes");
        setting.register("Pad[].name");
        setting.register("Pad[].note");
        setting.register("Pad[].type");
        setting.register("Pad[].row");
        setting.register("Pad[].column");
        setting.register("Pad[].message");
        setting.register("Pad[].channel");
        setting.register("Pad[].gate");
        setting.register("Pad[].value");
        setting.register("Pad[].valuemin");
        setting.register("Pad[].valuemax");
        setting.register("Pad[].valueinvert");
        setting.register("Pad[].toggletype");
        setting.register("Pad[].attributes");
   }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        ArrayList<MXSettingNode> children;
        children = setting.findByPath("Circle[]");
        for (MXSettingNode node : children) {
            int type = node.getSettingAsInt("type", -1);
            int row = node.getSettingAsInt("row", -1);
            int column = node.getSettingAsInt("column", -1);
            if (type < 0 || row < 0 || column < 0) {
                break;
            }
            MXUIValue status = new MXUIValue(type, row, column);
            try {
                status._name =  node.getSetting("name");
                status.setMonitoring(MXMessageFactory.fromDtext(node.getSetting("message")));
                status.setPortChannelGate(_port, node.getSettingAsInt("channel",0), node.getSettingAsInt("gate", 0));
                status.setValue(node.getSettingAsInt("value", 0));
                status._uiValueMinimum = node.getSettingAsInt("valuemin", 0);
                status._uiValueMaximum = node.getSettingAsInt("valuemax", 127);
                status._uiValueInvert = node.getSettingAsBoolean("valueinvert", false);
                _data.setCircleStatus(row, column, status);
                if (_data.getCircle(row, column) != null) {
                    _data.getCircle(row, column).updateUI();
                }
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
        children = setting.findByPath("Slider[]");
        for (MXSettingNode node : children) {
            int type = node.getSettingAsInt("type", -1);
            int row = node.getSettingAsInt("row", -1);
            int column = node.getSettingAsInt("column", -1);
            if (type < 0 || row < 0 || column < 0) {
                break;
            }
            MXUIValue status = new MXUIValue(type, row, column);
            try {
                status._name =  node.getSetting("name");
                status.setMonitoring(MXMessageFactory.fromDtext(node.getSetting("message")));
                status.setPortChannelGate(_port, node.getSettingAsInt("channel",0), node.getSettingAsInt("gate", 0));
                status.setValue(node.getSettingAsInt("value", 0));
                status._uiValueMinimum = node.getSettingAsInt("valuemin", 0);
                status._uiValueMaximum = node.getSettingAsInt("valuemax", 127);
                status._uiValueInvert = node.getSettingAsBoolean("valueinvert", false);
                _data.setSliderStatus(row, column, status);
                if (_data.getSlider(row, column) != null) {
                    _data.getSlider(row, column).updateUI();
                }
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
        children = setting.findByPath("Pad[]");
        for (MXSettingNode node : children) {
            int type = node.getSettingAsInt("type", -1);
            int row = node.getSettingAsInt("row", -1);
            int column = node.getSettingAsInt("column", -1);
            if (type < 0 || row < 0 || column < 0) {
                break;
            }
            MXUIValue status = new MXUIValue(type, row, column);
            try {
                status._name =  node.getSetting("name");
                status.setMonitoring(MXMessageFactory.fromDtext(node.getSetting("message")));
                status.setPortChannelGate(_port, node.getSettingAsInt("channel",0), node.getSettingAsInt("gate", 0));
                status.setValue(node.getSettingAsInt("value", 0));
                status._uiValueMinimum = node.getSettingAsInt("valuemin", 0);
                status._uiValueMaximum = node.getSettingAsInt("valuemax", 127);
                status._uiValueInvert = node.getSettingAsBoolean("valueinvert", false);
                status._drumIsToggleSwitch = node.getSettingAsBoolean("toggletype", false);
                _data.setDrumStatus(row, column, status);
                if (_data.getDrum(row, column) != null) {
                    _data.getDrum(row, column).updateUI();
                }
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        int counter;
        counter = 1;
        for (int column = 0; column < MXStatic.TOTAL_PORT_COUNT; ++ column) {
            for (int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
                String prefix = "Circle[" + counter + "]";
                MXUIValue status = this._data.getCircleStatus(row, column);
                MXMessage message = status.toMXMessage();
                setting.setSetting(prefix + ".name", status._name);
                setting.setSetting(prefix + ".type", status._controllerType);
                setting.setSetting(prefix + ".row", status._controllerRow);
                setting.setSetting(prefix + ".column", status._controllerColumn);
                setting.setSetting(prefix + ".message", MXMessageFactory.toDText(message));
                setting.setSetting(prefix + ".channel", message.getChannel());
                setting.setSetting(prefix + ".gate", message.getGate());
                setting.setSetting(prefix + ".value", status.getValue());
                setting.setSetting(prefix + ".valuemin", status._uiValueMinimum);
                setting.setSetting(prefix + ".valuemax", status._uiValueMaximum);
                setting.setSetting(prefix + ".valueinvert", status._uiValueInvert);
                counter ++;
            }
        }
        counter = 1;
        for (int column = 0; column < MXStatic.TOTAL_PORT_COUNT; ++ column) {
            for (int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
                String prefix = "Slider[" + counter + "]";
                MXUIValue status = this._data.getSliderStatus(row, column);
                MXMessage message = status.toMXMessage();
                setting.setSetting(prefix + ".name", status._name);
                setting.setSetting(prefix + ".type", status._controllerType);
                setting.setSetting(prefix + ".row", status._controllerRow);
                setting.setSetting(prefix + ".column", status._controllerColumn);
                setting.setSetting(prefix + ".message", MXMessageFactory.toDText(message));
                setting.setSetting(prefix + ".channel", message.getChannel());
                setting.setSetting(prefix + ".gate", message.getGate());
                setting.setSetting(prefix + ".value", status.getValue());
                setting.setSetting(prefix + ".valuemin", status._uiValueMinimum);
                setting.setSetting(prefix + ".valuemax", status._uiValueMaximum);
                setting.setSetting(prefix + ".valueinvert", status._uiValueInvert);
                counter ++;
            }
        }
        counter = 1;
        for (int column = 0; column < MXStatic.TOTAL_PORT_COUNT; ++ column) {
            for (int row = 0; row < MXStatic.DRUM_ROW_COUNT; ++ row) {
                String prefix = "Pad[" + counter + "]";
                MXUIValue status = this._data.getDrumStatus(row, column);
                MXMessage message = status.toMXMessage();
                setting.setSetting(prefix + ".name", status._name);
                setting.setSetting(prefix + ".type", status._controllerType);
                setting.setSetting(prefix + ".row", status._controllerRow);
                setting.setSetting(prefix + ".column", status._controllerColumn);
                setting.setSetting(prefix + ".message", MXMessageFactory.toDText(message));
                setting.setSetting(prefix + ".channel", message.getChannel());
                setting.setSetting(prefix + ".gate", message.getGate());
                setting.setSetting(prefix + ".value", status.getValue());
                setting.setSetting(prefix + ".valuemin", status._uiValueMinimum);
                setting.setSetting(prefix + ".valuemax", status._uiValueMaximum);
                setting.setSetting(prefix + ".valueinvert", status._uiValueInvert);
                setting.setSetting(prefix + ".toggletype", status._drumIsToggleSwitch);
                counter ++;
            }
        }

    }
}
