/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import jp.synthtarou.midimixer.mx30controller.ui.MGDrum;
import jp.synthtarou.midimixer.mx30controller.ui.MGCircle;
import jp.synthtarou.midimixer.mx30controller.ui.MGSlider;
import jp.synthtarou.midimixer.mx30controller.ui.MXUIValue;
import java.util.ArrayList;
import java.util.List;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXNumberExpansion;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;

/**
 *
 * @author YOSHIDA Shintarou
 */

public class MX32PageData {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX32PageData.class);

    String _mixerName;
    final MX32PageProcess _process;

    private ArrayList<MGSlider>[] _matrixSliderComponent;
    private ArrayList<MGCircle>[] _matrixCircleComponent;
    private ArrayList<MGDrum>[] _matrixDrumComponent;

    private ArrayList<MXUIValue>[] _matrixSliderStatus;
    private ArrayList<MXUIValue>[] _matrixCircleStatus;
    private ArrayList<MXUIValue>[] _matrixDrumStatus;

    private ArrayList<MXUIValue>[][] _cachedControlChange;
    private ArrayList<MXUIValue>[][] _cachedChannelMessage;
    private ArrayList<MXUIValue>_cachedSystemMessage;

    int _patchToMixer = -1;
    boolean _patchTogether = false;    
    
    public MX32PageData(MX32PageProcess process) {
        this._process = process;
        initMixer();
    }
 
    public void updatePadView(int row, int column, boolean value, int lastVelocity, boolean linkedControl) {
        if (_matrixDrumComponent == null) {
            return;
        }
        MGDrum rhythm = _matrixDrumComponent[row].get(column);
        MXUIValue status = _matrixDrumStatus[row].get(column);;
        
        status._drumHitVelocity = lastVelocity;
        status._drumCurrentPushing = value;

        if (_patchToMixer >= 0) {
            MX32PageProcess nextMixer = _process._parent.getPage(_patchToMixer);
            MGDrum next = nextMixer._data._matrixDrumComponent[row].get(column);
            MXUIValue nextStatus = nextMixer._data._matrixDrumStatus[row].get(column);
            boolean preview = nextStatus._drumCurrentPushing;
            nextStatus.catchDrumValue(value, linkedControl);
            nextMixer._data.updatePadView(row, column, nextStatus._drumCurrentPushing, lastVelocity, false);
            if (_patchTogether == false) {
                return;
            }
        }

        MXMessage message = status.toMXMessage();
        if (message != null) {
            if (linkedControl == false) {
                catchMessage(message, true);
                _process._parent.sendToNext(message);
            }
        }
    }

    public void doUpdateSlider(int row, int column, int newValue, boolean linkedControl) {
        if (_matrixSliderComponent == null) {
            return;
        }
        MGSlider slider = _matrixSliderComponent[row].get(column);
        MXUIValue status = _process._data._matrixSliderStatus[row].get(column);

        status.setValue(newValue);
        slider.updateUIOnly(newValue);
        
        if (_patchToMixer >= 0) {
            MX32PageData nextMixer = _process._parent.getPage(_patchToMixer)._data;

            MGSlider next = nextMixer._matrixSliderComponent[row].get(column);
            MXUIValue nextStatus  = nextMixer._matrixSliderStatus[row].get(column);

            if (nextStatus._uiValueMaximum != status._uiValueMaximum
             || nextStatus._uiValueMinimum != status._uiValueMinimum) {
                int fromRange = status._uiValueMaximum - status._uiValueMinimum;
                int toRange = nextStatus._uiValueMaximum - nextStatus._uiValueMinimum;
                int fromOffset = status.getValue() - status._uiValueMinimum;
                boolean invert = false;
                if (status._uiValueInvert) {
                    invert = !nextStatus._uiValueInvert;
                }else {
                    invert = nextStatus._uiValueInvert;
                }

                MXNumberExpansion trans = new MXNumberExpansion(fromRange, toRange);
                int toOffset = trans.getNewNumber(fromOffset, invert);
                newValue = nextStatus._uiValueMinimum + toOffset;
            }
                
            nextStatus.setValue(newValue);
            MGSlider nextSlider = nextMixer._matrixSliderComponent[row].get(column);
            nextSlider.updateUIOnly(newValue);

            if (_patchTogether == false) {
                return;
            }
        }

        if (status._uiValueLastSent != newValue) {
            status._uiValueLastSent = newValue;
            MXMessage message = status.toMXMessage();
            if (message != null) {
                if (linkedControl == false) {
                    catchMessage(message, true);
                    _process._parent.sendToNext(message);
                }
            }
        }
    }
    
    public void doUpdateCircle(int row, int column, int newValue, boolean linkedControll) {
        if (_matrixCircleComponent == null) {
            return;
        }
        MGCircle circle = _matrixCircleComponent[row].get(column);
        MXUIValue status = _matrixCircleStatus[row].get(column);;

        status.setValue(newValue);
        circle.updateUIOnly(newValue);
        
        if (_patchToMixer >= 0) {
            MX32PageData nextMixer = _process._parent.getPage(_patchToMixer)._data;
            MGCircle next = nextMixer._matrixCircleComponent[row].get(column);

            MXUIValue nextStatus = nextMixer._matrixCircleStatus[row].get(column);;
            if (nextStatus._uiValueMaximum != status._uiValueMaximum
             || nextStatus._uiValueMinimum != status._uiValueMinimum) {
                int fromRange = status._uiValueMaximum - status._uiValueMinimum;
                int toRange = nextStatus._uiValueMaximum - nextStatus._uiValueMinimum;
                int fromOffset = status.getValue() - status._uiValueMinimum;
                boolean invert = false;
                if (status._uiValueInvert) {
                    invert = !nextStatus._uiValueInvert;
                }else {
                    invert = nextStatus._uiValueInvert;
                }

                MXNumberExpansion trans = new MXNumberExpansion(fromRange, toRange);
                int toOffset = trans.getNewNumber(fromOffset, invert);
                newValue = nextStatus._uiValueMinimum + toOffset;
            }

            nextStatus.setValue(newValue);
            MGCircle nextCircle  = nextMixer._matrixCircleComponent[row].get(column);
            nextCircle.updateUIOnly(newValue);

            if (_patchTogether == false) {
                return;
            }
        }
        if (status._uiValueLastSent != newValue) {
            status._uiValueLastSent = newValue;
            MXMessage message = status.toMXMessage();
            if (message != null) {
                if (linkedControll == false) {
                    catchMessage(message, true);
                    _process._parent.sendToNext(message);
                }
            }
        }
    }

    public synchronized List<MXUIValue> getCachedList(MXMessage request) {
        if (_cachedControlChange == null) {
            _cachedControlChange = new ArrayList[16][256];
            _cachedChannelMessage = new ArrayList[16][256];
            _cachedSystemMessage = new ArrayList();

            for (int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
                for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++column) {
                    MXUIValue status = _matrixSliderStatus[row].get(column);
                    MXMessage message = status.toMXMessage();
                    if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                        int data1 = message.getCCCodeFromBytes();
                        if (_cachedControlChange[message.getChannel()][data1] == null) {
                            _cachedControlChange[message.getChannel()][data1] = new ArrayList();
                        }
                        _cachedControlChange[message.getChannel()][data1].add(status);
                    }else if (status.watchingChannelMessage()) {
                        if (_cachedChannelMessage[message.getChannel()][message.getCommand()] == null) {
                            _cachedChannelMessage[message.getChannel()][message.getCommand()] = new ArrayList();
                        }
                        _cachedChannelMessage[message.getChannel()][message.getCommand()].add(status);
                    }else {
                        _cachedSystemMessage.add(status);
                    }
                }
            }

            for (int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
                for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++column) {
                    MXUIValue status = _matrixCircleStatus[row].get(column);
                    MXMessage message = status.toMXMessage();
                    if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                        int data1 = message.getCCCodeFromBytes();
                        if (_cachedControlChange[message.getChannel()][data1] == null) {
                            _cachedControlChange[message.getChannel()][data1] = new ArrayList();
                        }
                        _cachedControlChange[message.getChannel()][data1].add(status);
                    }else if (status.watchingChannelMessage()) {
                        if (_cachedChannelMessage[message.getChannel()][message.getCommand()] == null) {
                            _cachedChannelMessage[message.getChannel()][message.getCommand()] = new ArrayList();
                        }
                        _cachedChannelMessage[message.getChannel()][message.getCommand()].add(status);
                    }else {
                        _cachedSystemMessage.add(status);
                    }
                }
            }

            for (int row = 0; row < MXStatic.DRUM_ROW_COUNT; ++ row) {
                for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++column) {
                    MXUIValue status = _matrixDrumStatus[row].get(column);
                    MXMessage message = status.toMXMessage();
                    if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                        int data1 = message.getCCCodeFromBytes();
                        if (_cachedControlChange[message.getChannel()][data1] == null) {
                            _cachedControlChange[message.getChannel()][data1] = new ArrayList();
                        }
                        _cachedControlChange[message.getChannel()][data1].add(status);
                    }else if (status.watchingChannelMessage()) {
                        if (_cachedChannelMessage[message.getChannel()][message.getCommand()] == null) {
                            _cachedChannelMessage[message.getChannel()][message.getCommand()] = new ArrayList();
                        }
                        _cachedChannelMessage[message.getChannel()][message.getCommand()].add(status);
                    }else {
                        _cachedSystemMessage.add(status);
                    }
                }
            }
        }
        if (request.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            return _cachedControlChange[request.getChannel()][request.getCCCodeFromBytes()];
        }else if (request.isMessageTypeChannel()) {
            return _cachedChannelMessage[request.getChannel()][request.getCommand()];
        }else {
            return _cachedSystemMessage;
        }
    }
    
    public boolean catchMessage(MXMessage message, boolean linkedControl) {
        List<MXUIValue> list = getCachedList(message);
        if (list != null) {
            boolean done = false;
            for (int i = 0; i < list.size(); ++ i) {
                MXUIValue status = list.get(i);
                if (status.catchMessageValue(message)) {
                    switch(status._controllerType) {
                        case MXUIValue.TYPE_SLIDER:
                            doUpdateSlider(status._controllerRow, status._controllerColumn, status.getValue(), linkedControl);
                            done = true;
                            break;
                        case MXUIValue.TYPE_CIRCLE:
                            doUpdateCircle(status._controllerRow, status._controllerColumn, status.getValue(), linkedControl);
                            done = true;
                            break;
                        case MXUIValue.TYPE_DRUM:
                            if (status.getValue() >= 1) {
                                MXMessage message2 = status.toMXMessage();
                                if (message2.getCommand() == MXMidi.COMMAND_NOTEON) {
                                    status._drumHitVelocity = message2.getValue();
                                }                    
                                status.catchDrumValue(true, linkedControl);
                            }else {
                                status.catchDrumValue(false, linkedControl);
                            }
                            //done = true;
                            break;
                    }
                }
            }
            if (done) {
                return true;
            }
        }
        return false;
    }

    public void initMixer() {
        ArrayList<MXUIValue>[] circleMatrix = new ArrayList[4];
        ArrayList<MXUIValue>[] sliderMatrix = new ArrayList[1];
        ArrayList<MXUIValue>[] padMatrix = new ArrayList[2];

        for(int row = 0; row < sliderMatrix.length; ++ row) {
            int column = 0;
            ArrayList<MXUIValue> slider = new ArrayList();

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                MXUIValue cs = new MXUIValue(MXUIValue.TYPE_SLIDER, row, column);
                MXMessage message;
                if (column >= 16) {
                    message = MXMessageFactory.fromDtext("F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h");
                    message.setPort(_process._port);
                    message.setValue(127 + 127 * 128);
                    cs.setMonitoring(message);
                }else {
                    message = MXMessageFactory.fromShortMessage(_process._port, MXMidi.COMMAND_CONTROLCHANGE + column, MXMidi.DATA1_CC_CHANNEL_VOLUME, 0);
                    cs.setValue(127);
                    cs.setMonitoring(message);
                }
                slider.add(cs);
                column ++;
            }
            sliderMatrix[row] = slider;
        }
        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MXUIValue> circle = new ArrayList();
            int column = 0;
            int[] ccCode = new int[] { 
                MXMidi.DATA1_EFFECT3_CHORUS,
                MXMidi.DATA1_EFFECT1_REVERVE, 
                MXMidi.DATA1_CC_EXPRESSION,
                MXMidi.DATA1_CCPANPOT
            };
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                MXUIValue cs2 = new MXUIValue(MXUIValue.TYPE_CIRCLE, row, column);
                MXMessage message;
                if (column >= 16) {
                    message = MXMessageFactory.fromDtext("F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h");
                    message.setPort(_process._port);
                    message.setValue(127 + 127 * 128);
                    cs2.setMonitoring(message);
                }else {
                    message = MXMessageFactory.fromShortMessage(_process._port, MXMidi.COMMAND_CONTROLCHANGE + column, ccCode[row], 0);
                }
                cs2.setMonitoring(message);
                circle.add(cs2);
                column ++;
            }
            circleMatrix[row] = circle;
        }
        for (int row = 0; row < padMatrix.length; ++ row) {
            int column = 0;
            ArrayList<MXUIValue> pad = new ArrayList();
            
            while (pad.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                MXUIValue cs3 = new MXUIValue(MXUIValue.TYPE_DRUM, row, column);
                MXMessage message = null;
                if (column >= 16) {
                    message = MXMessageFactory.fromDtext("F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h");
                    message.setPort(_process._port);
                    message.setValue(127 + 127 * 128);
                    cs3.setMonitoring(message);
                }else {
                    message = MXMessageFactory.fromShortMessage(_process._port, MXMidi.COMMAND_PROGRAMCHANGE + column, column, 0);
                }
                cs3.setMonitoring(message);
                pad.add(cs3);
                column ++;
            }
            padMatrix[row] = pad;
        }
 
        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _matrixDrumStatus = padMatrix;
    }

    public void sendMessageOfPad(int row, int column) {
        MGDrum rhythm =  _matrixDrumComponent[row].get(column);
        MXUIValue rhythmStatus =  _matrixDrumStatus[row].get(column);
        if (rhythmStatus._drumCurrentPushing) {
            MXMessage message = rhythmStatus.toMXMessage();
            if (message != null) {
                _process._parent.sendToNext(message);
            }
        }else {
            MXMessage message = rhythmStatus.toMXMessage();
            if (message != null
            && (message.getCommand() != MXMidi.COMMAND_PROGRAMCHANGE)) {
                _process._parent.sendToNext(message);
            }
        }
    }

    public MGSlider getSlider(int row, int column) {
        return _matrixSliderComponent[row].get(column);
    }
    
    public MGCircle getCircle(int row, int column) {
        return _matrixCircleComponent[row].get(column);
    }
    
    public MGDrum getDrum(int row, int column) {
        return _matrixDrumComponent[row].get(column);
    }

    public MXUIValue getSliderStatus(int row, int column) {
        return _matrixSliderStatus[row].get(column);
    }
    
    public MXUIValue getCircleStatus(int row, int column) {
        return _matrixCircleStatus[row].get(column);
    }
    
    public MXUIValue getDrumStatus(int row, int column) {
        return _matrixDrumStatus[row].get(column);
    }

    public void setSlider(int row, int column, MGSlider slider) {
        _matrixSliderComponent[row].set(column, slider);
    }
    
    public void setCircle(int row, int column, MGCircle circle) {
        _matrixCircleComponent[row].set(column, circle);
    }
    
    public void setDrum(int row, int column, MGDrum drum) {
        _matrixDrumComponent[row].set(column, drum);
    }

    public void setSliderStatus(int row, int column, MXUIValue status) {
        _matrixSliderStatus[row].set(column, status);
    }
    
    public void setCircleStatus(int row, int column, MXUIValue status) {
        _matrixCircleStatus[row].set(column, status);
    }
    
    public void setDrumStatus(int row, int column, MXUIValue status) {
        _matrixDrumStatus[row].set(column, status);
    }

    public void setEveryComponents(ArrayList<MGSlider>[] slider, ArrayList<MGCircle>[] circle, ArrayList<MGDrum>[] drum) {
        _matrixSliderComponent = slider; 
        _matrixCircleComponent = circle;
        _matrixDrumComponent = drum;
    }
}
