/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import java.util.TreeMap;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIInManager implements MXSettingTarget {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMIDIInManager.class);
    private static final MXMIDIInManager _instance = new MXMIDIInManager();
    
    public static MXMIDIInManager getManager() {
        return _instance;
    }

    public void reloadDeviceList() {
        //TODO Java not support?
    }

    protected MXMIDIInManager() {
    }
    
    private MXSetting _setting;
    public void initWithSetting() {
        if (_setting == null) {
            _setting = new MXSetting("MIDIInput");
            _setting.setTarget(this);
            listAllInput();
            _setting.readFile();

            if (MXMIDIIn.INTERNAL_SEQUENCER.assigned() < 0) {
                int found = -1;
                for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
                    boolean retry = false;
                    for (MXMIDIIn in : listAllInput().valueList()) {
                        if (in.assigned() == i) {
                            retry = true;
                            break;
                        }
                    }
                    if (!retry) {
                        found = i;
                        break;
                    }
                }
                if (found >= 0) {
                    reserveInput(MXMIDIIn.INTERNAL_SEQUENCER, found);
                }
            }
        }
    }

    protected MXWrapList<MXMIDIIn> _listAllInput;
    protected MXWrapList<MXMIDIIn> _listUsingInput;
    protected MXMIDIIn[] _cache;

    public synchronized MXWrapList<MXMIDIIn> listAllInput() {
        if (_listAllInput != null) {
            return _listAllInput;
        }

        try {
            MXWrapList<MXMIDIIn> temp = new MXWrapList<MXMIDIIn>();
            MidiDevice.Info[] infoList = MidiSystem.getMidiDeviceInfo();
           
            MXMIDIIn sequencer = MXMIDIIn.INTERNAL_SEQUENCER;
            temp.addNameAndValue(sequencer.getName(), sequencer);
            
            for (int i = 0; i < infoList.length; i++) {
                MidiDevice device = MidiSystem.getMidiDevice(infoList[i]);

                if (device.getMaxTransmitters() != 0) {
                    MXMIDIIn in = null;
                    try {
                        String name = device.getDeviceInfo().getName();
                        if (name.contains(MXStatic.LOOPMIDI_NAME)) {
                            continue;
                        }
                        in = new MXMIDIIn(name, device);
                        temp.addNameAndValue(in.getName(), in);
                    } catch (Exception e) {
                    }
                }
            }

            _listAllInput = temp;
            return _listAllInput;
        }catch(MidiUnavailableException e){
            throw new Error("something wrote", e);
        }
    }

    public MXMIDIIn findMIDIInput(String deviceName) {
        MXWrapList<MXMIDIIn> model = listAllInput();
        return model.valueOfName(deviceName);
    }

    public MXMIDIIn findMIDIInput(int assigned) {
        listSelectedInput();
        if (assigned >= 0) {
            return _cache[assigned];
        }else {
            return null;
        }
    }

    synchronized void onClose(MXMIDIIn input) {
        clearMIDIInCache();
    }
    
    protected synchronized void clearMIDIInCache() {
        _listUsingInput = null;
        _cache = null;        
    }

    public synchronized MXWrapList<MXMIDIIn>listSelectedInput() {
        if (_listUsingInput != null) {
            return _listUsingInput;
        }
        _cache = new MXMIDIIn[MXStatic.TOTAL_PORT_COUNT];
        MXWrapList<MXMIDIIn> newInput = new MXWrapList();
        for (MXMIDIIn midi : listAllInput().valueList()) {
            if (midi.assigned() >= 0) {
                newInput.addNameAndValue(midi.toString(), midi);
                _cache[midi.assigned()] = midi;
            }
        }
        _listUsingInput = newInput;
        return newInput;
    }

    public synchronized void closeAll() {
        for( MXMIDIIn input : listSelectedInput().valueList()) {
            input.close();
        }
    }
    
    public synchronized boolean reserveInput(MXMIDIIn input, int assignnew) {
        MXWrapList<MXMIDIIn> list = listAllInput();
        if (input.assigned() >= 0) {
            if (input.assigned() == assignnew) {
                return true;
            }
            for (int i = 0; i < list.size(); ++ i) {
                MXMIDIIn x = list.valueOfIndex(i);
                if (x.assigned() == assignnew) {
                    x.close();
                }
            }
            clearMIDIInCache();
            input.assign(assignnew);
            return true;
        }else {
            for (int i = 0; i < list.size(); ++ i) {
                MXMIDIIn x = list.valueOfIndex(i);
                if (x.assigned() == assignnew) {
                    x.close();
                    x.assign(-1);
                }
            }
            clearMIDIInCache();
            int inAssigned = input.assigned();
            if (inAssigned >= 0) {
                if (assignnew < 0) {
                    input.close();
                }
                return true;
            }
            input.assign(assignnew);
            if (input.open()) {
                return true;
            }
            return false;
        }
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("device[].name");
        setting.register("device[].open");
        setting.register("device[].toMaster");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        TreeMap<String, MXMIDIIn> dummy = new TreeMap();
        
        for (int x = 0; x < MXStatic.TOTAL_PORT_COUNT; ++ x) {
            String deviceName = setting.getSetting("device[" + x + "].name");
            String deviceOpen = setting.getSetting("device[" + x + "].open");
            String deviceMaster = setting.getSetting("device[" + x + "].toMaster");
            
            if (deviceName == null) {
                continue;
            }
            if (deviceOpen == null) {
                deviceOpen = "0";
            }
            
            MXWrapList<MXMIDIIn> detected = listAllInput();
            MXMIDIIn in = detected.valueOfName(deviceName);
            if (in != null) {
                in.assign(x);
                if (deviceOpen.equals("1")) {
                    in.open();
                }
            }else {
                in  = new MXMIDIIn(deviceName, null);
                in.assign(x);
                dummy.put(deviceName, in);
            }
            in.setMasterList(deviceMaster);
        }
        for (MXMIDIIn in : dummy.values()) {
            _listAllInput.addNameAndValue(in.getName(),in);
        }
        clearMIDIInCache();
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        MXWrapList<MXMIDIIn> all = listAllInput();
        for (MXMIDIIn e : all.valueList()) {
            int x = e.assigned();
            if (x >= 0) {
                setting.setSetting("device[" + x + "].name", e.getName());
                setting.setSetting("device[" + x + "].open", e.isOpen() ? "1" : "0");
                setting.setSetting("device[" + x + "].toMaster", e.getMasterList());
            }
        }
    }
    }
