/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx60output;

import javax.swing.JComponent;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.port.FinalMIDIOut;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;
import jp.synthtarou.midimixer.mx10input.MX10Data;
import jp.synthtarou.midimixer.mx20patchbay.MX20Process;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX60Process extends MXReceiver implements MXSettingTarget {
    public MX60Data _data;
    private MX60View _view;

    private MX20Process _patchBay;

    MXSetting _setting;

    public MX60Process() {
        _data = new MX60Data();
        _patchBay = new MX20Process(false);
        _view = new MX60View(this);
        _view.setLogger(FinalMIDIOut.getInstance().getLogger());
        _setting = new MXSetting("OutputSkip");
        _setting.setTarget(this);
        _patchBay.setNextReceiver(new MXReceiver()  {
            @Override
            public String getReceiverName() {
                return "dummy";
            }

            @Override
            public JComponent getReceiverView() {
                return null;
            }

            @Override
            public void processMXMessage(MXMessage message) {
                if (MX60Process.this.isUsingThisRecipe()) {
                    if (_data.isMessageToSkip(message)) {
                        return;
                    }
                }
                MX60Process.this.sendToNext(message);
            }
            
        });
    }
    
    public void readSettings() {
        _setting.readFile();
        _patchBay.readSettings();
    }

    public void setUseLogging(boolean log) {
        FinalMIDIOut.getInstance().getLogger().setUsingThisRecipe(log);
    }
    
    public boolean isUseLogging() {
        if (FinalMIDIOut.getInstance().getLogger() != null) {
            return FinalMIDIOut.getInstance().getLogger().isUsingThisRecipe();
        }
        return true;
    }
    
    @Override
    public void processMXMessage(MXMessage message) {
        _patchBay.processMXMessage(message);
    }

    @Override
    public String getReceiverName() {
        return "Output Dispatcher";
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }
    
    public MX20Process getPatchBay() {
        return _patchBay;
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String prefix = "Setting[" + port + "].";
            StringBuffer str = new StringBuffer();
            for (int j = 0; j <_data.countOfTypes(); ++ j) {
                String name = _data.typeNames[j];
                //System.out.println(name + " = " + setting.getSetting(prefix + name));
                boolean set = setting.getSettingAsBoolean(prefix + name, false);
                _data.setSkip(port, j, set);
            }
        }
        _view.resetTableModel();
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String prefix = "Setting[" + port + "].";
            StringBuffer str = new StringBuffer();
            for (int j = 0; j <_data.countOfTypes(); ++ j) {
                boolean set = _data.isSkip(port, j);
                String name = _data.typeNames[j];
                setting.setSetting(prefix + name, set);
            }
        }
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        String prefix = "Setting[].";
        for (String text : MX10Data.typeNames) {
            setting.register(prefix + text);
        }
    }
}
