/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.util.ArrayList;
import java.util.List;
import javax.swing.JComponent;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXNumberExpansion;
import jp.synthtarou.midimixer.libs.MXTimer;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXUtilMidi;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingNode;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;
import jp.synthtarou.midimixer.mx40layer.MXChannelInfo;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX32MixerProcess extends MXReceiver implements MXSettingTarget {
    final int _port;
    final MX30Process _parent;
    final MX32MixerView _view;
    MXNoteOffWatcher _noteOff;
    MXSetting _setting;
    
    String _mixerName;

    MX32MixerData _data;

    int _patchToMixer = -1;
    boolean  _patchTogether = false;    

    public MX32MixerProcess(MX30Process parent, int port) {
        _parent = parent;
        _port = port;
        _data = new MX32MixerData(this);
        _view = new MX32MixerView(this);
        _setting = new MXSetting("Mixing" + (port+1));
        _setting.setTarget(this);
    }

    public void readSettings() {
        _setting.readFile();
        _view.updateUI();
    }

    @Override
    public String getReceiverName() {
        return "IN " + MXUtilMidi.nameOfPort(_port);
    }

    @Override
    public JComponent getReceiverView() {
        return _view;
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (isUsingThisRecipe() == false) { 
            sendToNext(message); return; 
        }
        letsTryMessage(message, new DoubleCheck(_parent.getNextReceiver()));
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("PatchToMixer");
        
        setting.register("Circle[].name");
        setting.register("Circle[].note");
        setting.register("Circle[].type");
        setting.register("Circle[].row");
        setting.register("Circle[].column");
        setting.register("Circle[].message");
        setting.register("Circle[].channel");
        setting.register("Circle[].gate");
        setting.register("Circle[].value");
        setting.register("Circle[].valuemin");
        setting.register("Circle[].valuemax");
        setting.register("Circle[].valueinvert");
        setting.register("Circle[].attributes");

        setting.register("Circle[].dataentryType");
        setting.register("Circle[].dataentryMSB");
        setting.register("Circle[].dataentryLSB");
        setting.register("Circle[].isValue14bit");

        setting.register("Slider[].name");
        setting.register("Slider[].note");
        setting.register("Slider[].type");
        setting.register("Slider[].row");
        setting.register("Slider[].column");
        setting.register("Slider[].message");
        setting.register("Slider[].channel");
        setting.register("Slider[].gate");
        setting.register("Slider[].value");
        setting.register("Slider[].valuemin");
        setting.register("Slider[].valuemax");
        setting.register("Slider[].valueinvert");
        setting.register("Slider[].attributes");

        setting.register("Slider[].dataentryType");
        setting.register("Slider[].dataentryMSB");
        setting.register("Slider[].dataentryLSB");
        setting.register("Slider[].isValue14bit");

        setting.register("Pad[].name");
        setting.register("Pad[].note");
        setting.register("Pad[].type");
        setting.register("Pad[].row");
        setting.register("Pad[].column");
        setting.register("Pad[].message");
        setting.register("Pad[].channel");
        setting.register("Pad[].gate");
        setting.register("Pad[].value");
        setting.register("Pad[].valuemin");
        setting.register("Pad[].valuemax");
        setting.register("Pad[].valueinvert");
        setting.register("Pad[].attributes");
        setting.register("Pad[].isValue14bit");

        setting.register("Pad[].dataentryType");
        setting.register("Pad[].dataentryMSB");
        setting.register("Pad[].dataentryLSB");

        setting.register("Pad[].switchType");
        setting.register("Pad[].switchWithToggle");
        setting.register("Pad[].switchInputType");

        setting.register("Pad[].switchOutPort");
        setting.register("Pad[].switchOutChannel");

        setting.register("Pad[].switchOutOnType");
        setting.register("Pad[].switchOutOnValue");
        setting.register("Pad[].switchOutOnValueFixed");
        setting.register("Pad[].switchOutOnText");
        setting.register("Pad[].switchOutOnTextGate");
        setting.register("Pad[].switchOutOff");
        setting.register("Pad[].switchOutOffValue");
        setting.register("Pad[].switchOutOffText");
        setting.register("Pad[].switchOutOffTextGate");
        setting.register("Pad[].switchHarmonyVelocityType");
        setting.register("Pad[].switchHarmonyVelocityFixed");
        setting.register("Pad[].switchHarmonyNotes");
        setting.register("Pad[].switchSequencerFile");
        setting.register("Pad[].switchSequencerSingltTrack");
        setting.register("Pad[].switchSequencerOneChannel");
        setting.register("Pad[].switchSequencerSeekStart");
        setting.register("Pad[].switchSequencerFilterNote");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        ArrayList<MXSettingNode> children;
        children = setting.findByPath("Circle[]");
        _patchToMixer = setting.getSettingAsInt("PatchToMixer", -1);
        for (MXSettingNode node : children) {
            int type = node.getSettingAsInt("type", -1);
            int row = node.getSettingAsInt("row", -1);
            int column = node.getSettingAsInt("column", -1);
            if (type < 0 || row < 0 || column < 0) {
                break;
            }
            MGStatus status = new MGStatus(_port, type, row, column);
            try {
                status.setName(node.getSetting("name"));
                String msgText = node.getSetting("message");
                int channel = node.getSettingAsInt("channel",0);
                int gate = node.getSettingAsInt("gate", 0);
                int value = node.getSettingAsInt("value", 0);
                status.setRangeMin(node.getSettingAsInt("valuemin", 0));
                status.setRangeMax(node.getSettingAsInt("valuemax", 127));
                status.setUiValueInvert(node.getSettingAsBoolean("valueinvert", false));
                status.setMonitoringTarget(msgText, channel, gate, value);

                status.setDataentryType(node.getSettingAsInt("dataentryType", MXMidi.DATAENTRY_TYPE_NONE));
                status.setDataentryMSB(node.getSettingAsInt("dataentryMSB", 0));
                status.setDataentryLSB(node.getSettingAsInt("dataentryLSB", 0));
                status.setValue14bit(node.getSettingAsBoolean("isValue14bit", false));

                _data.setCircleStatus(row, column, status);
                /*
                if (getCircle(row, column) != null) {
                    getCircle(row, column).updateUI();
                }*/
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
        children = setting.findByPath("Slider[]");
        for (MXSettingNode node : children) {
            int type = node.getSettingAsInt("type", -1);
            int row = node.getSettingAsInt("row", -1);
            int column = node.getSettingAsInt("column", -1);
            if (type < 0 || row < 0 || column < 0) {
                break;
            }
            MGStatus status = new MGStatus(_port, type, row, column);
            try {
                status.setName(node.getSetting("name"));
                String msgText = node.getSetting("message");

                int channel = node.getSettingAsInt("channel",0);
                int gate = node.getSettingAsInt("gate", 0);
                int value = node.getSettingAsInt("value", 0);
                status.setMonitoringTarget(msgText, channel, gate, value);

                status.setRangeMin(node.getSettingAsInt("valuemin", 0));
                status.setRangeMax(node.getSettingAsInt("valuemax", 127));
                status.setUiValueInvert(node.getSettingAsBoolean("valueinvert", false));

                status.setDataentryType(node.getSettingAsInt("dataentryType", MXMidi.DATAENTRY_TYPE_NONE));
                status.setDataentryMSB(node.getSettingAsInt("dataentryMSB", 0));
                status.setDataentryLSB(node.getSettingAsInt("dataentryLSB", 0));
                status.setValue14bit(node.getSettingAsBoolean("isValue14bit", false));

                _data.setSliderStatus(row, column, status);
                /*
                if (getSlider(row, column) != null) {
                    getSlider(row, column).updateUI();
                }*/
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
        children = setting.findByPath("Pad[]");
        int count = 0;
        for (MXSettingNode node : children) {
            int type = node.getSettingAsInt("type", -1);
            int row = node.getSettingAsInt("row", -1);
            int column = node.getSettingAsInt("column", -1);
            String msgText;
            if (type < 0 || row < 0 || column < 0) {
                break;
            }
            if (row == 0 && column == 0) {
                count ++;
            }
            MGStatus status = new MGStatus(_port, type, row, column);

            try {
                status.setName(node.getSetting("name"));
                msgText = node.getSetting("message");

                int channel = node.getSettingAsInt("channel",0);
                int gate = node.getSettingAsInt("gate", 0);
                int value = node.getSettingAsInt("value", 0);
                status.setMonitoringTarget(msgText, channel, gate, value);
                
                status.setRangeMin(node.getSettingAsInt("valuemin", 0));
                status.setRangeMax(node.getSettingAsInt("valuemax", 127));
                status.setUiValueInvert(node.getSettingAsBoolean("valueinvert", false));

                status.setDataentryType(node.getSettingAsInt("dataentryType", MXMidi.DATAENTRY_TYPE_NONE));
                status.setDataentryMSB(node.getSettingAsInt("dataentryMSB", 0));
                status.setDataentryLSB(node.getSettingAsInt("dataentryLSB", 0));
                status.setValue14bit(node.getSettingAsBoolean("isValue14bit", false));

                status.setSwitchType(node.getSettingAsInt("switchType", MGStatus.SWITCH_TYPE_ONOFF));
                status.setSwitchWithToggle(node.getSettingAsBoolean("switchWithToggle", false));
                status.setSwitchInputType(node.getSettingAsInt("switchInputType", MGStatus.SWITCH_ON_IF_PLUS1));
                status.setSwitchOutOnType(node.getSettingAsInt("switchOutOnType", MGStatus.SWITCH_OUT_ON_SAME_AS_INPUT));
                status.setSwitchOutOnTypeOfValue(node.getSettingAsInt("switchOutOnValue", MGStatus.SWITCH_OUT_ON_VALUE_AS_INPUT));
                status.setSwitchOutOnText(node.getSetting("switchOutOnText"));
                status.setSwitchOutOnTextGate(node.getSettingAsInt("switchOutOnTextGate", 127));
                status.setSwitchOutOnValueFixed(node.getSettingAsInt("switchOutOnValueFixed", 127));
                status.setSwitchOutOffType(node.getSettingAsInt("switchOutOff", MGStatus.SWITCH_OUT_OFF_SAME_AS_INPUT));
                status.setSwitchOutOffTypeOfValue(node.getSettingAsInt("switchOutOffValue", MGStatus.SWITCH_OUT_OFF_VALUE_0));
                status.setSwitchOutOffText(node.getSetting("switchOutOffText"));
                status.setSwitchOutOffTextGate(node.getSettingAsInt("switchOutOffTextGate", 0));
                status.setSwitchOutChannel(node.getSettingAsInt("switchOutChannel", 0));
                status.setSwitchHarmonyVelocityType(node.getSettingAsInt("switchHarmonyVelocityType", MGStatus.SWITCH_HARMONY_VELOCITY_FIXED));
                status.setSwitchHarmonyVelocityFixed(node.getSettingAsInt("switchHarmonyVelocityFixed", 100));
                status.setSwitchHarmonyNotes(node.getSetting("switchHarmonyNotes"));
                status.setSwitchSequencerFile(node.getSetting("switchSequencerFile"));
                status.setSwitchSequencerToSingltTrack(node.getSettingAsBoolean("switchSequencerSingltTrack", true));
                status.setSwitchSequenceSeekStart(node.getSettingAsBoolean("switchSequencerSeekStart", true));
                status.setSwitchSequencerFilterNote(node.getSettingAsBoolean("switchSequencerFilterNote", true));

                status.setSwitchOutPort(node.getSettingAsInt("switchOutPort", 0));
                status.setSwitchOutChannel(node.getSettingAsInt("switchOutChannel", 0));

                _data.setDrumPadStatus(row, column, status);

                /*
                if (getDrumPad(row, column) != null) {
                    getDrumPad(row, column).updateUI();
                }*/
            }catch(Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        int counter;
        counter = 1;
        setting.setSetting("PatchToMixer", _patchToMixer);
        for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++ column) {
            for (int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
                String prefix = "Circle[" + counter + "].";
                MGStatus status = _data.getCircleStatus(row, column);
                MXMessage message = status.toMXMessage();
                setting.setSetting(prefix + "name", status.getName());
                setting.setSetting(prefix + "type", status.getUiType());
                setting.setSetting(prefix + "row", row);
                setting.setSetting(prefix + "column", column);
                setting.setSetting(prefix + "message", MXMessageFactory.toDText(message));
                setting.setSetting(prefix + "channel", message.getChannel());
                setting.setSetting(prefix + "gate", message.getGate());
                setting.setSetting(prefix + "value", status.getValue());
                setting.setSetting(prefix + "valuemin", status.getRangeMin());
                setting.setSetting(prefix + "valuemax", status.getRangeMax());
                setting.setSetting(prefix + "valueinvert", status.isUiValueInvert());
                setting.setSetting(prefix + "dataentryType", status.getDataentryType());
                setting.setSetting(prefix + "dataentryMSB", status.getDataentryMSB());
                setting.setSetting(prefix + "dataentryLSB", status.getDataentryLSB());
                setting.setSetting(prefix + "isValue14bit", status.isValue14bit());
                counter ++;
            }
        }
        counter = 1;
        for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++ column) {
            for (int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
                String prefix = "Slider[" + counter + "].";
                MGStatus status = _data.getSliderStatus(row, column);
                MXMessage message = status.toMXMessage();
                setting.setSetting(prefix + "name", status.getName());
                setting.setSetting(prefix + "type", status.getUiType());
                setting.setSetting(prefix + "row", row);
                setting.setSetting(prefix + "column", column);
                setting.setSetting(prefix + "message", MXMessageFactory.toDText(message));
                setting.setSetting(prefix + "channel", message.getChannel());
                setting.setSetting(prefix + "gate", message.getGate());
                setting.setSetting(prefix + "value", status.getValue());
                setting.setSetting(prefix + "valuemin", status.getRangeMin());
                setting.setSetting(prefix + "valuemax", status.getRangeMax());
                setting.setSetting(prefix + "valueinvert", status.isUiValueInvert());
                setting.setSetting(prefix + "dataentryType", status.getDataentryType());
                setting.setSetting(prefix + "dataentryMSB", status.getDataentryMSB());
                setting.setSetting(prefix + "dataentryLSB", status.getDataentryLSB());
                setting.setSetting(prefix + "isValue14bit", status.isValue14bit());
                counter ++;
            }
        }
        counter = 1;
        for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++ column) {
            for (int row = 0; row < MXStatic.DRUM_ROW_COUNT; ++ row) {
                String prefix = "Pad[" + counter + "].";
                MGStatus status = _data.getDrumPadStatus(row, column);
                MXMessage message = status.toMXMessage();
                setting.setSetting(prefix + "name", status.getName());
                setting.setSetting(prefix + "type", status.getUiType());
                setting.setSetting(prefix + "row", row);
                setting.setSetting(prefix + "column", column);
                setting.setSetting(prefix + "message", MXMessageFactory.toDText(message));
                setting.setSetting(prefix + "channel", message.getChannel());
                setting.setSetting(prefix + "gate", message.getGate());
                setting.setSetting(prefix + "value", status.getValue());
                setting.setSetting(prefix + "valuemin", status.getRangeMin());
                setting.setSetting(prefix + "valuemax", status.getRangeMax());
                setting.setSetting(prefix + "valueinvert", status.isUiValueInvert());
                setting.setSetting(prefix + "dataentryType", status.getDataentryType());
                setting.setSetting(prefix + "dataentryMSB", status.getDataentryMSB());
                setting.setSetting(prefix + "dataentryLSB", status.getDataentryLSB());
                setting.setSetting(prefix + "isValue14bit", status.isValue14bit());

                setting.setSetting(prefix + "switchType", status.getSwitchType());
                setting.setSetting(prefix + "switchWithToggle", status.isSwitchWithToggle());
                setting.setSetting(prefix + "switchInputType", status.getSwitchInputType());
                setting.setSetting(prefix + "switchOutOnType", status.getSwitchOutOnType());
                setting.setSetting(prefix + "switchOutOnValue", status.getSwitchOutOnTypeOfValue());
                setting.setSetting(prefix + "switchOutOnValueFixed", status.getSwitchOutOnValueFixed());
                setting.setSetting(prefix + "switchOutOnText", status.getSwitchOutOnText());
                setting.setSetting(prefix + "switchOutOnTextGate", status.getSwitchOutOnTextGate());
                setting.setSetting(prefix + "switchOutOff", status.getSwitchOutOffType());
                setting.setSetting(prefix + "switchOutOffValue", status.getSwitchOutOffTypeOfValue());
                setting.setSetting(prefix + "switchOutOffText", status.getSwitchOutOffText());
                setting.setSetting(prefix + "switchOutOffTextGate", status.getSwitchOutOffTextGate());
                setting.setSetting(prefix + "switchOutPort", status.getSwitchOutPort());
                setting.setSetting(prefix + "switchOutChannel", status.getSwitchOutChannel());
                setting.setSetting(prefix + "switchHarmonyVelocityType", status.getSwitchHarmonyVelocityType());
                setting.setSetting(prefix + "switchHarmonyVelocityFixed", status.getSwitchHarmonyVelocityFixed());
                setting.setSetting(prefix + "switchHarmonyNotes", status.getSwitchHarmonyNotes());
                setting.setSetting(prefix + "switchSequencerFile", status.getSwitchSequencerFile());
                setting.setSetting(prefix + "switchSequencerSingltTrack", status.isSwitchSequencerToSingltTrack());
                setting.setSetting(prefix + "switchSequencerSeekStart", status.isSwitchSequenceSeekStart());
                setting.setSetting(prefix + "switchSequencerFilterNote", status.isSwitchSequencerFilterNote());

                counter ++;
            }
        }
    }
    
    public void catchedValue(MGStatus status, int newValue, DoubleCheck already) {
        if (_data.ready() == false) {
            //constructor
            return;
        }
        if (already == null) {
            already = new DoubleCheck(getNextReceiver());
        }
        if (already.checkAlready(status)) {
            return;
        }
        
        int row = status.getRow(), column = status.getColumn();

        MGSlider slider = null; 
        MGCircle circle  = null;
        MGPad drumpad = null;

        if (status.getUiType() == MGStatus.TYPE_SLIDER) {
            slider = _data.getSlider(row, column);
            already.push(status);
            status.setValue(newValue);
            slider.updateUIOnly(newValue);
        }
        if (status.getUiType() == MGStatus.TYPE_CIRCLE) {
            circle = _data.getCircle(row, column);
            already.push(status);
            status.setValue(newValue);
            circle.updateUIOnly(newValue);
        }
        if (status.getUiType() == MGStatus.TYPE_DRUMPAD) {
            status.setValue(newValue);
            if (status.isDrumOn(newValue)) {
                catchedValueDrum(status, true, newValue, already);
            }else {
                catchedValueDrum(status, false, 0, already);
            }
            return;
        }

        if (_patchToMixer >= 0) {
            MX32MixerProcess nextMixer = _parent.getPage(_patchToMixer);

            MGStatus nextStatus = null;
            switch(status.getUiType()) {
                case MGStatus.TYPE_SLIDER:
                    nextStatus  = nextMixer._data.getSliderStatus(row, column);
                    break;
                case MGStatus.TYPE_CIRCLE:
                    nextStatus  = nextMixer._data.getCircleStatus(row, column);
                    break;
                case MGStatus.TYPE_DRUMPAD:
                    nextStatus  = nextMixer._data.getDrumPadStatus(row, column);
                    break;
            }

            if (nextStatus.getRangeMax() != status.getRangeMax()
             || nextStatus.getRangeMin() != status.getRangeMin()) {
                int fromRange = status.getRangeMax() - status.getRangeMin();
                int toRange = nextStatus.getRangeMax() - nextStatus.getRangeMin();
                int fromOffset = status.getValue() - status.getRangeMin();
                boolean invert = false;
                if (status.isUiValueInvert()) {
                    invert = !nextStatus.isUiValueInvert();
                }else {
                    invert = nextStatus.isUiValueInvert();
                }

                MXNumberExpansion trans = new MXNumberExpansion(fromRange, toRange);
                int toOffset = trans.getNewNumber(fromOffset, invert);
                newValue = nextStatus.getRangeMin() + toOffset;
            }

            nextMixer.catchedValue(nextStatus, newValue, already);

            if (_patchTogether == false) {
                return;
            }
        }

        MXMessage message = status.toMXMessage();
        if (message != null) {
            message = MXMessageFactory.fromClone(message);
            if (message.getValue() == 0 && message.getCommand() == MXMidi.COMMAND_NOTEON) {
                message.setStatus(MXMidi.COMMAND_NOTEOFF + message.getChannel());
            }
        }
        already.sendOnlyNeed(message);
        letsTryMessage(message, already);
    }
    
    public void catchedValueDrum(MGStatus status, boolean newValue, int velocity, DoubleCheck already) {
        if (already == null) {
            already = new DoubleCheck(getNextReceiver());
        }
        if (already.checkAlready(status)) {
            return;
        }
        already.push(status);

        int row = status.getRow(), column = status.getColumn();

        MGSlider slider = null; 
        MGCircle circle  = null;
        MGPad drumpad = null;

        if (status.getUiType() == MGStatus.TYPE_DRUMPAD) {
            drumpad = _data.getDrumPad(row, column);
        }else {
            throw new IllegalStateException();
        }

        if (_patchToMixer >= 0) {
            MX32MixerProcess nextMixer = _parent.getPage(_patchToMixer);

            MGStatus nextStatus = null;
            switch(status.getUiType()) {
                case MGStatus.TYPE_SLIDER:
                    nextStatus  = nextMixer._data.getSliderStatus(row, column);
                    break;
                case MGStatus.TYPE_CIRCLE:
                    nextStatus  = nextMixer._data.getCircleStatus(row, column);
                    break;
                case MGStatus.TYPE_DRUMPAD:
                    nextStatus  = nextMixer._data.getDrumPadStatus(row, column);
                    break;
            }

            nextMixer.catchedValueDrum(nextStatus, newValue, velocity, already);

            if (_patchTogether == false) {
                return;
            }
        }

        boolean now = newValue;
        boolean prev = status.isValueLastDetect();
        
        int value = status.getValue();
        if (status.toMXMessage().getCommand() == MXMidi.COMMAND_PROGRAMCHANGE) {
            value = 0;
        }

        status.setSwitchLastDetected(velocity);

        boolean flag = (prev != now);

        if (!flag && status.toMXMessage().hasValueLowField() == false) {
            flag = true;
        }

        if (!flag && status.getSwitchInputType() == MGStatus.SWITCH_ON_WHEN_ANY) {
            flag = true;
        }

        if (flag) { // ワンショットまたは、画面上の数値が切り替わった (nowへ)
            status.setValueLastDetect(now);
            if (now) { // オンにきりかわった
                if (status.isSwitchWithToggle()) {
                    boolean lastSent = status.isValueLastSent();
                    now = !lastSent;
                }
            }else { // オフにきりかわたｔ
                if (status.isSwitchWithToggle()) {
                    //　トグルなら終了
                    return;    
                }
            }
            MXMessage message = null;
            status.setValueLastSent(now);
            drumpad.updateUIOnly(now);
            if (now) {
                if (status.getSwitchType() == MGStatus.SWITCH_TYPE_SEQUENCE) {
                    status.startSequence();
                    return;
                }
                if (status.getSwitchType() == MGStatus.SWITCH_TYPE_HARMONY) {
                    String notes = status.getSwitchHarmonyNotes();
                    int veltype = status.getSwitchHarmonyVelocityType();
                    if (veltype == MGStatus.SWITCH_HARMONY_VELOCITY_SAME_AS_INPUT) {
                        if (value != 0) {
                            velocity = value;
                        }else {
                            velocity = status.getSwitchHarmonyVelocityFixed();
                        }
                    }else if (veltype == MGStatus.SWITCH_HARMONY_VELOCITY_FIXED) {
                        velocity = status.getSwitchHarmonyVelocityFixed();
                    }else {
                        throw  new IllegalStateException("velocity unknown");
                    }
                    int[] noteList = MXUtilMidi.textToNoteList(notes);
                    for (int note : noteList) {
                        message = MXMessageFactory.fromShortMessage(_port, MXMidi.COMMAND_NOTEON + status.getSwitchOutChannel(), note, velocity);
                        already.sendOnlyNeed(message);
                    }
                    return;
                }
                message = status.toMXMessageCaseDrumOn();
                if (message == null) {
                    return;
                }
            }else {
                if (status.getSwitchType() == MGStatus.SWITCH_TYPE_ON) {
                    return;
                }
                if (status.getSwitchType() == MGStatus.SWITCH_TYPE_SEQUENCE) {
                    status.stopSequence();
                    return;
                }
                if (status.getSwitchType() == MGStatus.SWITCH_TYPE_HARMONY) {
                    String notes = status.getSwitchHarmonyNotes();
                    int[] noteList = MXUtilMidi.textToNoteList(notes);
                    for (int note : noteList) {
                        message = MXMessageFactory.fromShortMessage(_port, MXMidi.COMMAND_NOTEOFF + status.getSwitchOutChannel(), note, 0);
                        already.sendOnlyNeed(message);
                    }
                    return;
                }
                message = status.toMXMessageCaseDrumOff();
                if (message == null) {
                    return;
                }
            }
            message = MXMessageFactory.fromClone(message);
            if (message.getValue() == 0 && message.getCommand() == MXMidi.COMMAND_NOTEON) {
                message.setStatus(MXMidi.COMMAND_NOTEOFF + message.getChannel());
            }
            letsTryMessage(message, already);
        }
    }
    

    public void makeCacheInternal1(MGStatus status) {
        if (status == null) {
            return;
        }
        
        MXMessage message = status.toMXMessage();
        if (message.getDataentryType() != MXMidi.DATAENTRY_TYPE_NONE) {
            _data._cachedDataentry.add(status);
        }else if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            int data1 = message.getGate();
            if (_data._cachedControlChange[message.getChannel()][data1] == null) {
                _data._cachedControlChange[message.getChannel()][data1] = new ArrayList();
            }
            _data._cachedControlChange[message.getChannel()][data1].add(status);
            int data2 = -1;
            if (data1 >= 0 && data1 <= 31 && status.isValue14bit()) {
                data2 = data1 + 32;
                if (_data._cachedControlChange[message.getChannel()][data2] == null) {
                    _data._cachedControlChange[message.getChannel()][data2] = new ArrayList();
                }
                _data._cachedControlChange[message.getChannel()][data2].add(status);
            }
        }else if (message.getCommand() == MXMidi.COMMAND_NOTEON || message.getCommand() == MXMidi.COMMAND_NOTEOFF
                ||message.getCommand() == MXMidi.COMMAND_POLYPRESSURE) {
            int note = message.getGate();
            if (_data._cachedNoteMessage[message.getChannel()][note] == null) {
                _data._cachedNoteMessage[message.getChannel()][note] = new ArrayList();
            }
            _data._cachedNoteMessage[message.getChannel()][note].add(status);    
        }else if (message.isMessageTypeChannel()) {
            if (_data._cachedChannelMessage[message.getChannel()][message.getCommand()] == null) {
                _data._cachedChannelMessage[message.getChannel()][message.getCommand()] = new ArrayList();
            }
            _data._cachedChannelMessage[message.getChannel()][message.getCommand()].add(status);
        }else {
            _data._cachedSystemMessage.add(status);
        }
    }

    public synchronized List<MGStatus> getCachedList(MXMessage request) {
        if (_data._cachedControlChange == null) {
            _data._cachedControlChange = new ArrayList[16][256];
            _data._cachedChannelMessage = new ArrayList[16][256];
            _data._cachedNoteMessage = new ArrayList[16][256];
            _data._cachedSystemMessage = new ArrayList();
            _data._cachedDataentry = new ArrayList();
            
            for (int row = 0; row < MXStatic.SLIDER_ROW_COUNT; ++ row) {
                for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++column) {
                    makeCacheInternal1(_data.getSliderStatus(row, column));
                }
            }

            for (int row = 0; row < MXStatic.CIRCLE_ROW_COUNT; ++ row) {
                for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++column) {
                    makeCacheInternal1(_data.getCircleStatus(row, column));
                }
            }

            for (int row = 0; row < MXStatic.DRUM_ROW_COUNT; ++ row) {
                for (int column = 0; column < MXStatic.SLIDER_COLUMN_COUNT; ++column) {
                    MGStatus status = _data.getDrumPadStatus(row, column);
                    makeCacheInternal1(status);
                }
            }
        }
        if (request.getDataentryType() != MXMidi.DATAENTRY_TYPE_NONE) {
            return _data._cachedDataentry;
        }else if (request.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            return _data._cachedControlChange[request.getChannel()][request.getGate()];
        }else if (request.getCommand() == MXMidi.COMMAND_NOTEON || request.getCommand() == MXMidi.COMMAND_NOTEOFF
                ||request.getCommand() == MXMidi.COMMAND_POLYPRESSURE) {
            return _data._cachedNoteMessage[request.getChannel()][request.getNoteNumberFromBytes()];
        }else if (request.isMessageTypeChannel()) {
            return _data._cachedChannelMessage[request.getChannel()][request.getCommand()];
        }else {
            return _data._cachedSystemMessage;
        }
    }
    
    MXMessage _poolFor14bit = null;
    MGStatus _poolFor14bitStatus = null;
    int _gotValue14bit = 0;
    
    public boolean isPairToPooled14bit(MXMessage message) {
        if (_poolFor14bit != null) {
            if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                if (message.getGate() >= 0 && message.getGate() < 32) {
                    if (message.getGate() +32 == _poolFor14bit.getGate()) {
                        return true;
                    }
                }else {
                    if (message.getGate() == _poolFor14bit.getGate() +32) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
    
    public int valueForPair(MXMessage message) {
        if (isPairToPooled14bit(message)) {
            if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                if (message.getGate() >= 0 && message.getGate() < 32) {
                    if (message.getGate() +32 == _poolFor14bit.getGate()) {
                        return message.getValue() * 128 + _poolFor14bit.getValue();
                    }
                }else if (message.getGate() >= 32 && message.getChannel() <= 64) {
                    if (message.getGate() == _poolFor14bit.getGate() +32) {
                        return message.getValue() + _poolFor14bit.getValue()  * 128;
                    }
                }
            }
        }
        throw new IllegalStateException("valueForPair not work at the moment");
    }

    public boolean isSameToPooled14bit(MXMessage message) {
        if (_poolFor14bit != null) {
            if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                if (message.getGate() == _poolFor14bit.getGate()) {
                    return true;
                }
            }
        }
        return false;
    }

    long lastTick = 0;
    
    public synchronized  void clearPoolImpl() {
        if (lastTick + 400 < System.currentTimeMillis()) {
            if (_poolFor14bit != null) {
                MXMessage prev = _poolFor14bit;
                _poolFor14bit = null;
                prev.setValue14bitAndZoom(true);
                sendToNext(prev);
                return;
            }
        }
    }

    public synchronized  void letsTryMessage(MXMessage message, DoubleCheck already) {
        if (already == null) {
            already = new DoubleCheck(getNextReceiver());
        }
        
        
        if (message.getDataentryType() == MXMidi.DATAENTRY_TYPE_NONE) {
            if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                if (message.getGate() == MXMidi.DATA1_CC_DATAENTRY) {
                    int value = message.getValue();
                    MXChannelInfo info = MXMain.getMain().get40InputInfo().getChannelInfo(message.getPort(), message.getChannel());
                    if (info._dataentryType != MXMidi.DATAENTRY_TYPE_NONE) {
                        if (message.isValue14bit()) {
                            MXMessage mes2 = MXMessageFactory.fromDataentry(info._dataentryType, message.getPort(), message.getChannel(),
                                        info._dataentryMSB, info._dataentryLSB, value >> 7, value & 0x7f);
                            message = mes2;
                        }else {
                            MXMessage mes2 = MXMessageFactory.fromDataentry(info._dataentryType, message.getPort(), message.getChannel(),
                                        info._dataentryMSB, info._dataentryLSB, value, -1);
                            message = mes2;
                        }
                    }else {
                    }
                }
            }
        }
        int fail = 0;
        int hit = 0;
        if (_poolFor14bit != null) {
            if (isPairToPooled14bit(message)) {
                int value = valueForPair(message);
                _poolFor14bit = null;
                message = MXMessageFactory.fromClone(message);
                message.setValue14bit(true);
                message.setValue(value);
                already.sendOnlyNeed(message);
                catchedValue(_poolFor14bitStatus, value, already);
                return;
            }else {
                clearPoolImpl();
            }
        }
        List<MGStatus> list = getCachedList(message);
        if (list != null) {
            boolean foundSome = false;
            for (int i = 0; i < list.size(); ++ i) {
                MGStatus status = list.get(i);
                if (status.statusTryCatch(message)) {
                    int command = message.getCommand();
                    int gate = message.getGate();
                    if (command == MXMidi.COMMAND_CONTROLCHANGE && gate >= 0 && gate < 32 && status.isValue14bit() && message.isValue14bit() == false) {
                        if (gate == message.getGate()) {
                            _poolFor14bit = message;
                            _poolFor14bitStatus = status;
                        }else if (gate + 32 == message.getGate()) {
                            _poolFor14bit = message;
                            _poolFor14bitStatus = status;
                        }else {
                            continue;
                        }
                        lastTick = System.currentTimeMillis();
                        foundSome = true;
                        hit ++;
                        MXTimer.letsCountdown(500, new Runnable() {
                            @Override
                            public void run() {
                                clearPoolImpl();
                            }
                        });
                    }
                    catchedValue(status, status.getValue(), already);
                    foundSome = true;
                    hit ++;
                }else {
                    fail ++;
                }
            }
            if (foundSome) {
                return;
            }
        }
        already.sendOnlyNeed(message);
    }

    public void notifyCacheBroken() {
        _data._cachedControlChange = null;
    }
}
