/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.util.ArrayList;
import javax.swing.JOptionPane;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX32MixerData {
    MX32MixerProcess _process;

    private ArrayList<MGSlider>[] _matrixSliderComponent;
    private ArrayList<MGCircle>[] _matrixCircleComponent;
    private ArrayList<MGPad>[] _matrixDrumComponent;

    private ArrayList<MGStatus>[] _matrixSliderStatus;
    private ArrayList<MGStatus>[] _matrixCircleStatus;
    private ArrayList<MGStatus>[] _matrixDrumStatus;

    public boolean ready() {
        return _matrixDrumComponent != null;
    }
    
    ArrayList<MGStatus>[][] _cachedControlChange;
    ArrayList<MGStatus>[][] _cachedChannelMessage;
    ArrayList<MGStatus>[][] _cachedNoteMessage;
    ArrayList<MGStatus>_cachedSystemMessage;
    ArrayList<MGStatus>_cachedDataentry;
    
    protected MX32MixerData(MX32MixerProcess process) {
        _process = process;
        initVolumeMixer();
    }
    
    public static final int INIT_TYPE_ZERO = 0;
    public static final int INIT_TYPE_MIXER = 1;
    public static final int INIT_TYPE_GMTOME = 2;
    public static final int INIT_TYPE_DAW = 3;
    public static final int INIT_TYPE_SOUDMODULE = 4;
    
    public boolean initializeData(int initType) {
        switch(initType) {
            case INIT_TYPE_MIXER:
                initVolumeMixer();
                break;
            case INIT_TYPE_DAW:
                initDaw();
                break;
            case INIT_TYPE_ZERO:
                initZero();
                break;
            case INIT_TYPE_GMTOME:
                initGMTone();
                break;
            case INIT_TYPE_SOUDMODULE:
                initSoundModule();
                break;
            default:
                JOptionPane.showMessageDialog(_process._view, "Not ready", "Sorry", JOptionPane.ERROR_MESSAGE);
                return false;
        }
        _cachedControlChange = null;
        _cachedChannelMessage = null;
        _cachedNoteMessage = null;
        _cachedSystemMessage = null;
        _cachedDataentry = null;
        return true;
    }
    
    private void initVolumeMixer() {
        MX32MixerProcess process = _process;
        ArrayList<MGStatus>[] circleMatrix = new ArrayList[4];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[1];

        int port = process._port;
        int column;
        
        circleMatrix[0] = new ArrayList();
        circleMatrix[1] = new ArrayList();
        circleMatrix[2] = new ArrayList();
        circleMatrix[3] = new ArrayList();

        sliderMatrix[0] = new ArrayList();
        
        MGStatus status;
        MXMessage message;

        for(int row = 0; row < sliderMatrix.length; ++ row) {
            column = 0;
            ArrayList<MGStatus> slider = sliderMatrix[row];

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                String text;
                if (column >= 16) {
                    text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";
                    status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                    message = MXMessageFactory.fromDtext(text, _process._port, 0, 0, 128 * 128-1);
                    status.setMonitoringTarget(text, 0, 0, 128 * 128 - 1);
                }else {
                    status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + column, MXMidi.DATA1_CC_CHANNEL_VOLUME, 128 -1);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), message.getChannel(), message.getGate(), message.getValue());
                }
                slider.add(status);
                    
                column ++;
            }
            sliderMatrix[row] = slider;
        }

        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            column = 0;
            int[] ccCode = new int[] { 
                MXMidi.DATA1_EFFECT3_CHORUS,
                MXMidi.DATA1_EFFECT1_REVERVE, 
                MXMidi.DATA1_CC_EXPRESSION,
                MXMidi.DATA1_CCPANPOT
            };
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                if (column >= 16) {
                    status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                    String text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";
                    message = MXMessageFactory.fromDtext(text, _process._port, 0, 0, 128 * 128 -1);
                    status.setMonitoringTarget(text, 0, 0, 128 * 128 - 1);
                    circle.add(status);

                    column ++;
                }else {
                    status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + column, ccCode[row], 64);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), column, message.getGate(), message.getValue());
                    circle.add(status);
                    
                    column ++;
                }
            }
            circleMatrix[row] = circle;
        }
 
        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _cachedControlChange = null;
        initDrumMinMax();
    }
    
    private void initDaw() {
        MX32MixerProcess process = _process;
        ArrayList<MGStatus>[] circleMatrix = new ArrayList[4];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[1];

        circleMatrix[0] = new ArrayList();
        circleMatrix[1] = new ArrayList();
        circleMatrix[2] = new ArrayList();
        circleMatrix[3] = new ArrayList();

        sliderMatrix[0] = new ArrayList();

        MGStatus status;
        MXMessage message;
        int port = process._port;
        
        int[] cclist = {
            73, 75, 79, 72, 80, 81, 82, 83, 85
        };
        int[] cclist2 = {
            74, 71, 76, 77, 93, 18, 19, 16, 17
        };
        
        int column = 0;
        for(int row = 0; row < sliderMatrix.length; ++ row) {
            ArrayList<MGStatus> slider = new ArrayList();

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                if (column < cclist.length) {
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + row, cclist[column], 128 -1);
                }else {
                    message = MXMessageFactory.createDummy();
                }
                status.setMonitoringTarget(MXMessageFactory.toDText(message), 0, message.getGate(), message.getValue());
                slider.add(status);
                column ++;
            }
            sliderMatrix[row] = slider;
        }

        column = 0;
        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                if (column < cclist2.length) {
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + row, cclist2[column], 128 -1);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), row, message.getGate(), message.getValue());
                }
                circle.add(status);
                column ++;
            }
            circleMatrix[row] = circle;
        }

        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _cachedControlChange = null;
        initDrumMinMax();
    }

    private void initZero() {
        MX32MixerProcess process = _process;
        ArrayList<MGStatus>[] circleMatrix = new ArrayList[4];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[1];

        circleMatrix[0] = new ArrayList();
        circleMatrix[1] = new ArrayList();
        circleMatrix[2] = new ArrayList();
        circleMatrix[3] = new ArrayList();

        sliderMatrix[0] = new ArrayList();
        
        int port = process._port;
        int column = 0;
        MGStatus status;
        MXMessage message;
        
        for(int row = 0; row < sliderMatrix.length; ++ row) {
            ArrayList<MGStatus> slider = new ArrayList();

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                slider.add(status);
                column ++;
            }
            sliderMatrix[row] = slider;
        }
        column = 0;
        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                circle.add(status);
                column ++;
            }
            circleMatrix[row] = circle;
        }

        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _cachedControlChange = null;
        initDrumMinMax();
    }

    private void initSoundModule() {
        MX32MixerProcess process = _process;

        ArrayList<MGStatus>[] circleMatrix = new ArrayList[4];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[1];

        int port = process._port;
        MGStatus status;
        MXMessage message;
        
        int[] cclist = {
            114, 18, 19, 16, 17, 91, 79, 72
        };
        int[] cclist2 = {
            112, 74, 71, 76, 77, 93, 73, 75
        };
        
        int column = 0;
        for(int row = 0; row < sliderMatrix.length; ++ row) {
            ArrayList<MGStatus> slider = new ArrayList();

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                if (column < cclist.length) {
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + row, cclist[column], 128 -1);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), 0, message.getGate(), message.getValue());
                }else {
                    message = MXMessageFactory.createDummy();
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), 0, message.getGate(), message.getValue());
                }
                slider.add(status);
                column ++;
            }
            sliderMatrix[row] = slider;
        }

        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            column = 0;
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                if (column < cclist2.length) {
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + row, cclist2[column], 128 -1);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), 0, message.getGate(), message.getValue());
                }else {
                    message = MXMessageFactory.createDummy();
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), 0, message.getGate(), message.getValue());
                }
                circle.add(status);
                column ++;
            }
            circleMatrix[row] = circle;
        }
 
        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _cachedControlChange = null;
        initDrumMinMax();
    }

    private void initGMTone() {
        MX32MixerProcess process = _process;

        ArrayList<MGStatus>[] circleMatrix = new ArrayList[4];
        ArrayList<MGStatus>[] sliderMatrix = new ArrayList[1];
        ArrayList<MGStatus>[] padMatrix = new ArrayList[2];
        
        int port = process._port;
        MXMessage message = null;
        MGStatus status = null;

        for(int row = 0; row < sliderMatrix.length; ++ row) {
            int column = 0;
            ArrayList<MGStatus> slider = new ArrayList();

            while (slider.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_SLIDER, row, column);
                String text;
                if (column >= 16) {
                    text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";
                    message = MXMessageFactory.fromDtext(text, _process._port, 0, 0, 128*128-1);
                    status.setMonitoringTarget(text, 0, 0, 0);
                }else {
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + column, MXMidi.DATA1_CC_EXPRESSION, 128 -1);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), column, message.getGate(), message.getValue());
                }
                slider.add(status);
                column ++;
            }
            sliderMatrix[row] = slider;
        }
        
        for (int row = 0; row < circleMatrix.length; ++ row) {
            ArrayList<MGStatus> circle = new ArrayList();
            int column = 0;
            int[] ccCode = new int[] { 
                MXMidi.DATA1_SOUND_ATTACKTIME,
                MXMidi.DATA1_SOUND_DECAYTIME, 
                MXMidi.DATA1_SOUND_RELEASETIME, 
                MXMidi.DATA1_SOUND_BLIGHTNESS,
            };
            while (circle.size() < MXStatic.SLIDER_COLUMN_COUNT) {
                status = new MGStatus(process._port, MGStatus.TYPE_CIRCLE, row, column);
                if (column >= 16) {
                    String text = "F0h, 7Fh, 0gh, 04h, 01h, #VL, #VH, F7h";
                    message = MXMessageFactory.fromDtext(text, _process._port, 0, 0, 128 * 128 -1);
                    status.setMonitoringTarget(text, 0, 0, 128 * 128 + 1);
                }else {
                    message = MXMessageFactory.fromShortMessage(port, MXMidi.COMMAND_CONTROLCHANGE + column, ccCode[row], 64);
                    status.setMonitoringTarget(MXMessageFactory.toDText(message), column, message.getGate(), message.getValue());
                }
                circle.add(status);
                column ++;
            }
            circleMatrix[row] = circle;
        }
        
        int[] proglist = {
            -1, 0, 2, 5, 8, 10, 13, 27, 36, 40, 56, 65, 72, 82, 96, 106, 120
        };

        ArrayList<MGStatus> pad = new ArrayList();
        for (int i = 0; i < 17; ++ i) {
            int prog = proglist[i];
            if (prog < 0) {
                message = MXMessageFactory.fromShortMessage(process._port, MXMidi.COMMAND_CONTROLCHANGE + 0, MXMidi.DATA1_CCMODULATION, 127);
            }else {
                message = MXMessageFactory.fromShortMessage(process._port, MXMidi.COMMAND_PROGRAMCHANGE + 0, prog, 0);
            }
            status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 1, i);
            status.setMonitoringTarget(MXMessageFactory.toDText(message), message.getChannel(), message.getGate(), message.getValue());
            pad.add(status);
        }
        padMatrix[0] = pad;

        pad = new ArrayList();
        for (int i = 0; i < 17; ++ i) {
            int prog = proglist[i];
            if (prog < 0) {
                message = MXMessageFactory.fromShortMessage(process._port, MXMidi.COMMAND_CONTROLCHANGE + 1, MXMidi.DATA1_CCMODULATION, 127);
            }else {
                message = MXMessageFactory.fromShortMessage(process._port, MXMidi.COMMAND_PROGRAMCHANGE + 1, prog, 0);
            }
            status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 1, i);
            status.setMonitoringTarget(MXMessageFactory.toDText(message), message.getChannel(), message.getGate(), message.getValue());
            pad.add(status);
        }
        padMatrix[1] = pad;

        _matrixSliderStatus = sliderMatrix;
        _matrixCircleStatus = circleMatrix;
        _matrixDrumStatus = padMatrix;
    }
    
    public void fillMaxOfSlider(MGStatus status, int column) {
        MXMessage message = MXMessageFactory.fromClone(_matrixSliderStatus[0].get(column).toMXMessage());
        int max = 127;
        if (message.hasValueHiField()) {
            max = 16383;
        }

        message.setValue(max);
        status.setMonitoringTarget(MXMessageFactory.toDText(message), message.getChannel(), message.getGate(), max);
        status.setSwitchType(MGStatus.SWITCH_TYPE_ON); // 1回のみで
        status.setRangeMin(max);
        status.setRangeMax(max);
        status.setSwitchOutOnTypeOfValue(MGStatus.SWITCH_OUT_ON_VALUE_FIXED);
        status.setSwitchOutOnValueFixed(max);
    }
    
    public void fillMinOfSlider(MGStatus status, int column) {
        MXMessage message = MXMessageFactory.fromClone(_matrixSliderStatus[0].get(column).toMXMessage());

        message.setValue(0);
        status.setMonitoringTarget(MXMessageFactory.toDText(message), message.getChannel(), message.getGate(), 0);
        status.setSwitchType(MGStatus.SWITCH_TYPE_ON); // 1回のみで
        status.setRangeMin(0);
        status.setRangeMax(0);
        status.setSwitchOutOnTypeOfValue(MGStatus.SWITCH_OUT_ON_VALUE_FIXED);
        status.setSwitchOutOnValueFixed(0);
    }
    
    public void initDrumMinMax() {
        MX32MixerProcess process = _process;

        ArrayList<MGStatus>[] sliderMatrix = _matrixSliderStatus;
        ArrayList<MGStatus>[] padMatrix = new ArrayList[2];

        padMatrix[0] = new ArrayList();
        padMatrix[1] = new ArrayList();

        int column = 0;
        while (padMatrix[0].size() < MXStatic.SLIDER_COLUMN_COUNT) {
            MGStatus status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 0, column);
            fillMaxOfSlider(status, column);
            padMatrix[0].add(status);

            status = new MGStatus(process._port, MGStatus.TYPE_DRUMPAD, 1, column);
            fillMinOfSlider(status, column);
            padMatrix[1].add(status);
            
            column ++;
        }
        
        _matrixDrumStatus = padMatrix;
    }

    public  MGSlider getSlider(int row, int column) {
        if (_matrixSliderComponent == null) {
            return null;
        }
        return _matrixSliderComponent[row].get(column);
    }

    public  MGCircle getCircle(int row, int column) {
        if (_matrixCircleComponent == null) {
            return null;
        }
        return _matrixCircleComponent[row].get(column);
    }

    public  MGPad getDrumPad(int row, int column) {
        if (_matrixDrumComponent == null) {
            return null;
        }
        return _matrixDrumComponent[row].get(column);
    }

    public  MGStatus getSliderStatus(int row, int column) {
        return _matrixSliderStatus[row].get(column);
    }

    public  MGStatus getCircleStatus(int row, int column) {
        return _matrixCircleStatus[row].get(column);
    }

    public  MGStatus getDrumPadStatus(int row, int column) {
        return _matrixDrumStatus[row].get(column);
    }

    public  void  setSliderStatus(int row, int column, MGStatus status) {
        _matrixSliderStatus[row].set(column, status);
    }

    public  void  setCircleStatus(int row, int column, MGStatus status) {
        _matrixCircleStatus[row].set(column, status);
    }

    public  void  setDrumPadStatus(int row, int column, MGStatus status) {
        _matrixDrumStatus[row].set(column, status);
    }

    public void setEveryComponents(ArrayList<MGSlider>[] slider, ArrayList<MGCircle>[] circle, ArrayList<MGPad>[] drum) {
        _matrixSliderComponent = slider; 
        _matrixCircleComponent = circle;
        _matrixDrumComponent = drum;
        _cachedControlChange = null;
    }
}
