/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.awt.Color;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.mx40layer.MXChannelInfo;
import jp.synthtarou.midimixer.mx40layer.MXChannelInfoReceiver;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX30Process extends MXReceiver {
    private MX30View _rootView;
    private MX32MixerProcess[] _pageProcess;
    boolean _editingControl;

    public MX30Process() {
        _rootView = new MX30View(this);
        _pageProcess = new MX32MixerProcess[MXStatic.TOTAL_PORT_COUNT];
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _pageProcess[i] = new MX32MixerProcess(this, i);
        }
    }

    public void readSettings() {
        for (int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            _pageProcess[port].readSettings();
            MX32MixerProcess proc = _pageProcess[port];
            _rootView.addPage(port, proc);
            proc.setNextReceiver(new MXReceiver() {
                public String getReceiverName() {
                    return null;
                }
                public JComponent getReceiverView() {
                    return null;
                }
                public void processMXMessage(MXMessage message) {
                    MX30Process.this.sendToNext(message);
                }
           });
        }
    }
    
   
    public MX32MixerProcess getPage(int i) {
        return _pageProcess[i];
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
            int gate = message.getGate();
            if (gate == MXMidi.DATA1_CC_DATAENTRY
              || gate == MXMidi.DATA1_CC_DATAENTRY + 0x20
              || gate == MXMidi.DATA1_CC_DATAINC
              || gate == MXMidi.DATA1_CC_DATADEC) {
                if (message.getDataentryType() == MXMidi.DATAENTRY_TYPE_NONE) {
                    MXChannelInfoReceiver record = MXMain.getMain().get40InputInfo();
                    MXChannelInfo info = record.getChannelInfo(message.getPort(), message.getChannel());
                    if (info._dataentryType == MXMidi.DATAENTRY_TYPE_RPN) {
                        //System.out.println("fixed RPN dataentry " + Integer.toHexString(info._dataentryMSB) + " / " + Integer.toHexString(info._dataentryLSB) + " for " + message.getValue());
                        message.setDataentryType(MXMidi.DATAENTRY_TYPE_RPN);
                        message.setDataentryMSB(info._dataentryMSB);
                        message.setDataentryLSB(info._dataentryLSB);
                    }else if (info._dataentryType == MXMidi.DATAENTRY_TYPE_NRPN) {
                        //System.out.println("fixed NRPN dataentry " + Integer.toHexString(info._dataentryMSB) + " / " + Integer.toHexString(info._dataentryLSB) + " for " + message.getValue());
                        message.setDataentryType(MXMidi.DATAENTRY_TYPE_NRPN);
                        message.setDataentryMSB(info._dataentryMSB);
                        message.setDataentryLSB(info._dataentryLSB);
                    }
                }
            }
        }
        
        if (isUsingThisRecipe() == false) { 
            sendToNext(message); return; 
        }
        int port = message.getPort();
        _pageProcess[port].processMXMessage(message);
    }

    @Override
    public String getReceiverName() {
        return "Kontrolls";
    }

    @Override
    public JComponent getReceiverView() {
        return _rootView;
    }
    
    int focus_port, focus_type = -1, focus_row, focus_column;
    
    Color colorBack = null;
    Color colorEdit =  null;
    Color colorFocus = null;
    
    public boolean isFocusControl(MGStatus status) {
        if (status.getPort() == focus_port) {
            if (status.getUiType() == focus_type) {
                if (status.getRow() == focus_row && status.getColumn() == focus_column) {
                    return true;
                }
            }
        }
        return false;
    }
    
    public boolean isFocusEditing(MGStatus status) {
        if (isFocusControl(status) == false) {
            return false;
        }
        return _editingControl;
    }
    
    public void showTextForFocus(int type, int port, int row,int column)
    {
        String value = "(NONE)";
        
        try {
            if (colorBack == null) {
                colorBack = new JLabel().getBackground();
                colorEdit = Color.orange;
                colorFocus = Color.cyan;
            }

            if (focus_type > 0) {
                Color c = MXStatic.sliderColor(focus_column);

                MX32MixerData focus_data = getPage(focus_port)._data;
                switch(focus_type) {
                    case MGStatus.TYPE_SLIDER:
                        MGSlider slider = focus_data.getSlider(focus_row, focus_column);
                        MXUtil.backgroundRecursive(slider, c);
                        break;
                    case MGStatus.TYPE_CIRCLE:
                        MGCircle circle  = focus_data.getCircle(focus_row, focus_column);
                        MXUtil.backgroundRecursive(circle, c);
                        break;
                    case MGStatus.TYPE_DRUMPAD:
                        MGPad drum = focus_data.getDrumPad(focus_row, focus_column);
                        MXUtil.backgroundRecursive(drum, c);
                        break;
                }
            }
            
            MX32MixerData focus_data = getPage(port)._data;

            switch(type) {
                case MGStatus.TYPE_SLIDER:
                    MGSlider slider = focus_data.getSlider(row, column);
                    value = focus_data.getSliderStatus(row, column).toString();
                    if (_editingControl) {
                        MXUtil.backgroundRecursive(slider, colorEdit);
                    }else {
                        MXUtil.backgroundRecursive(slider, colorFocus);
                    }
                    break;
                case MGStatus.TYPE_CIRCLE:
                    MGCircle circle  = focus_data.getCircle(row, column);
                    value = focus_data.getCircleStatus(row, column).toString();
                    if (_editingControl) {
                        MXUtil.backgroundRecursive(circle, colorEdit);
                    }else {
                        MXUtil.backgroundRecursive(circle, colorFocus);
                    }
                    break;
                case MGStatus.TYPE_DRUMPAD:
                    MGPad drum = focus_data.getDrumPad(row, column);
                    value = focus_data.getDrumPadStatus(row, column).toString();
                    if (_editingControl) {
                        MXUtil.backgroundRecursive(drum, colorEdit);
                    }else {
                        MXUtil.backgroundRecursive(drum, colorFocus);
                    }
                    break;
                default:
                    value = "Unknown";
                    break;
            }
            focus_type = type;
            focus_port = port;
            focus_row = row;
            focus_column = column;
        }catch(Throwable e) {
            value = "ERR";
            e.printStackTrace();
        }
        
        _pageProcess[port]._view.setFocusString(value);
    }

    public void goNextFocus(final int port, final int keyCode) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() { public void run() { goNextFocus(port, keyCode); } });
            return;
        }

        int type = focus_type;
        int row = focus_row;
        int column = focus_column;
        
        if (type < 0) {
            type = 0;
            row = 0;
            column = 0;
        }else {
            switch(keyCode) {
                case 38: //UP
                    if (row > 0) {
                        row --;
                        break;
                    }
                    switch(-- type) {
                        case 0:
                            type = 1;
                            break;
                        case MGStatus.TYPE_CIRCLE:
                            row = 3;
                            break;
                        case MGStatus.TYPE_SLIDER:
                            row = 0;
                            break;
                        case MGStatus.TYPE_DRUMPAD:
                            row = 1;
                            break;
                    }
                    break;

                case 40: //DOWN
                    row ++;
                    switch(type) {
                        case MGStatus.TYPE_CIRCLE:
                            if (row >= 4) {
                                row = 0;
                                type = MGStatus.TYPE_SLIDER;
                            }
                            break;
                        case MGStatus.TYPE_SLIDER:
                            if (row >= 1) {
                                row = 0;
                                type = MGStatus.TYPE_DRUMPAD;
                            }
                            break;
                        case MGStatus.TYPE_DRUMPAD:
                            if (row >= 2) {
                                row --;
                            }
                            break;
                    }
                    break;

                case 37: //LEFT
                    column --;
                    if (column < 0) {
                        column = 0;
                    }
                    break;

                case 39: //RIGHT
                    column ++;
                    if (column >= MXStatic.SLIDER_COLUMN_COUNT) {
                        column --;
                    }
                    break;

                case ' ':
                case '\b':
                case '\n':
                    MX32MixerData focus_data = getPage(port)._data;

                    switch(type) {
                        case MGStatus.TYPE_SLIDER:
                            MGSlider slider = focus_data.getSlider(focus_row, focus_column);
                            switch(keyCode) {
                                case ' ': slider.increment(); break;
                                case '\b': slider.decriment(); break;
                                case '\n': slider.editContoller(); break;
                            }
                            break;
                        case MGStatus.TYPE_CIRCLE:
                            MGCircle circle = focus_data.getCircle(focus_row, focus_column);
                            switch(keyCode) {
                                case ' ': circle.increment(); break;
                                case '\b': circle.decriment(); break;
                                case '\n': circle.editContoller(); break;
                            }
                            break;
                        case MGStatus.TYPE_DRUMPAD:
                            MGPad drum = focus_data.getDrumPad(focus_row, focus_column);
                            switch(keyCode) {
                                case ' ': drum.increment(null); break;
                                case '\b': drum.decriment(null); break;
                                case '\n': drum.editContoller(); break;
                            }
                            break;
                        default:
                            break;
                    }                
                    break;
            }
        }
        showTextForFocus(type,  port, row,  column);
    }

    public void enterEditMode(boolean flag) {
        if (flag) {
            _editingControl = true;
            _rootView.lockAnothereTabs(true);
            MXMain.getMain().getMainWindow().setTitle(MXStatic.MX_EDITING);
        }else {
            _editingControl = false;
            _rootView.lockAnothereTabs(false);
            MXMain.getMain().getMainWindow().setTitle(MXStatic.MX_APPNAME_WITH_VERSION);
        }
    }
}


