/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx30controller;

import java.util.Comparator;
import java.util.TreeSet;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class DoubleCheck {
    TreeSet<MGStatus> _alreadyStatus = new TreeSet(new StatusComparator());
    
    static class StatusComparator implements Comparator<MGStatus> {
        @Override
        public int compare(MGStatus o1, MGStatus o2) {
            int p = o1.getPort() - o2.getPort();
            if (p < 0) return -1;
            if (p > 0) return 1;
            
            int z = o1.getUiType() - o2.getUiType();
            if (z < 0) return -1;
            if (z > 0) return 1;

            int y = o1.getRow() - o2.getRow();
            if (y < 0) return -1;
            if (y > 0) return 1;

            int x = o1.getColumn() - o2.getColumn();
            if (x < 0) return -1;
            if (x > 0) return 1;

            return 0;
        }
    }
    
    TreeSet<MXMessage> _alreadyMessage = new TreeSet(new MessageComparator());
    
    static class MessageComparator implements Comparator<MXMessage> {

        @Override
        public int compare(MXMessage o1, MXMessage o2) {
            if (o1._extype != o2._extype) {
                int x = o1._extype - o2._extype;
                if (x < 0) return -1;
                if (x > 0) return 1;
            }
            switch(o1._extype) {
                case MXMessage.EXTYPE_DUMMY:
                case MXMessage.EXTYPE_PROGRAM_INC:
                case MXMessage.EXTYPE_PROGRAM_DEC:
                    return 0;
            }
            int[] t1 = o1._template;
            int[] t2 = o2._template;

            int x = t1.length - t2.length;
            
            if (x < 0) return -1;
            if (x > 0) return 1;

            for (int i = 0; i < t1.length; ++ i) {
                x = t1[i] - t2[i];
                if (x < 0) return -1;
                if (x > 0) return 1;
            }
            
            x = o1.getPort() - o2.getPort();
            if (x < 0) return -1;
            if (x > 0) return 1;

            x = o1.getStatus()- o2.getStatus();
            if (x < 0) return -1;
            if (x > 0) return 1;

            x = o1.getGate()- o2.getGate();
            if (x < 0) return -1;
            if (x > 0) return 1;

            x = o1.getValue()- o2.getValue();
            if (x < 0) return -1;
            if (x > 0) return 1;
            
            return 0;
        }
    }
    
    final MXReceiver _receiver;
    
    public DoubleCheck(MXReceiver receiver) {
        _receiver = receiver;
    }
    
    public boolean checkAlready(MGStatus status) {
        return _alreadyStatus.contains(status);
    }
    
    public void push(MGStatus status) {
        _alreadyStatus.add(status);
    }
    
    public boolean sendOnlyNeed(MXMessage message) {
        if (message.getCommand() == MXMidi.COMMAND_NOTEON && message.getValue() == 0) {
            message = MXMessageFactory.fromClone(message);
            message.setStatus(MXMidi.COMMAND_NOTEOFF + message.getChannel());
        }
        if (_alreadyMessage.contains(message)) {
            return false;
        }
        if (_receiver == null) {
            //panel is under construction
            return false;
        }
        _alreadyMessage.add(message);
        _receiver.processMXMessage(message);
        return true;
    }
}
