/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.mx20patchbay;

import javax.swing.JPanel;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXUtilMidi;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX20Process extends MXReceiver implements  MXSettingTarget {
    private static final MXDebugConsole _debug = new MXDebugConsole(MX20Process.class);
    MX20Data _data;
    private MX20View _view;
    private MXSetting _setting;

    private boolean _typeInput = false;
    
    public boolean isTypeInput() {
        return _typeInput;
    }

    public MX20Process(boolean typeInput) {
        _typeInput = typeInput;
        _data = new MX20Data();
        _view = new MX20View(this);
        if (typeInput) {
            _setting = new MXSetting("InputPortPatching");
        }else {
            _setting = new MXSetting("OutputPortPatching");
        }
        _setting.setTarget(this);
    }
    
    public void readSettings() {
        _setting.readFile();
        if (_setting.isEmpty()) {
            initStraight();
        }
        _view.setupViewFromModel();
    }
    
    public MX20Data getData() {
        return _data;
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (isUsingThisRecipe() == false) {
            sendToNext(message); 
            return;
        }
        
        int port = message.getPort();
        boolean proc = false;
        
        for (int j = 0; j < MXStatic.TOTAL_PORT_COUNT; ++ j) {
            if (_data.isEnabled(port, j)) {
                if (port == j) {
                    proc = true;
                    sendToNext(message);
                }else {
                    proc = true;
                    MXMessage message2 = MXMessageFactory.fromClone(message);
                    message2.setPort(j);
                    sendToNext(message2);
                }
            }
        }
    }

    public void initStraight() {
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _data.set(i, i, true);
        }
        _view.setupViewFromModel();
    }

    public void initStraightAndAllLast() {
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _data.set(i, i, true);
        }
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _data.set(i, MXStatic.TOTAL_PORT_COUNT - 1, true);
        }
        _view.setupViewFromModel();
    }

    @Override
    public String getReceiverName() {
        if (_typeInput) {
            return "Input Patch";
        }else {
            return "Output Patch";
        }
    }

    @Override
    public JPanel getReceiverView() {
        return _view;
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        for(int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            setting.register("From" + MXUtilMidi.nameOfPort(i));
        }
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        _data.clear();

        for(int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String text = setting.getSetting("From" + MXUtilMidi.nameOfPort(port));
            if (text == null) {
                continue;
            }
            String[] array = text.split("[ ,]+");
            if (array.length == 0) {
                continue;
            }
            for (String elem : array) {
                try {
                   int to = Integer.parseInt(elem);
                   _data.set(port, to, true);
                }catch(NumberFormatException e) {
                }
            }
        }
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        setting.clearValue();
        for(int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            StringBuffer text = new StringBuffer();
            for (int to = 0; to < MXStatic.TOTAL_PORT_COUNT; ++ to) {
                if (_data.isEnabled(port, to)) {
                    if (text.length() != 0) {
                        text.append(",");
                    }
                    text.append(String.valueOf(to));
                }
            }
            setting.setSetting("From" + MXUtilMidi.nameOfPort(port), text.toString());
        }
    }
}
