/*
 * Copyright (C) 2022 yaman
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi.smf;

// https://qiita.com/takayoshi1968/items/8e3f901539c92a6aac16

import java.util.ArrayList;
import java.util.TreeSet;

public class SMFTempoTable {
    // ファイル内にテンポ指定がない場合は 120 bpm = 500 000 mpq とする.
    private static long DEFAULT_MPQ = 500000;
    // mpqStack[n] = n 個目の Set Tempo イベントが持つ MPQ
    private int[] mpqStack;
    // cumulativeTicks[n] = 曲の先頭から、n 個目の Set Tempo イベントが発生するまでの時間 (Ticks)
    private long[] cumulativeTicks;
    // cumulativeMicroseconds[n] = 曲の先頭から、n 個目の Set Tempo イベントが発生するまでの時間 (us)
    private long[] cumulativeMicroseconds;
    
    // 分解能(四分音符ひとつは何 Tick であるか)
    int _fileResolution;
    public void setFileResolution(int reso) {
        _fileResolution = reso;
    }
    
    public int getFileResolution() {
        return _fileResolution;
    }
    
    // 再生に当たって、NAudio.Midi.MidiEventCollection は実質的に Midi ファイルとして見なせる
    public SMFTempoTable(SMFParser parser, int fileResolution) {
        // Pulses Per Quater note
        int resolution = fileResolution;
        ArrayList<SMFMessage> list = parser.getMessageList().listAll();

        // TempoEvent のみを抜き出す (イベントは AbsoluteTime の昇順で並んでいる)
        // Set Tempo イベントは 0 番トラックにのみ現れるはずなので、midiEvents[0] のみから探す
        ArrayList<SMFMessage> tempoEvents = new ArrayList();
        for (SMFMessage message : list) {
            if (message.isMetaMessage()) {
                if (message.getDataType()== 0x51) {
                    tempoEvents.add(message);
                }
            }
        }

        if (tempoEvents.size() == 0 || (tempoEvents.get(0).getTick() != 0)) {
            byte b1 = (byte)(DEFAULT_MPQ >> 16);
            byte b2 = (byte)(DEFAULT_MPQ >> 8);
            byte b3 = (byte)(DEFAULT_MPQ );
            SMFMessage mm = new SMFMessage(0, 0xff, 0x51, new byte[] { b1, b2, b3 });
            if (tempoEvents.size() > 0) {
                mm = new SMFMessage(0, 0xff, 0x51, tempoEvents.get(0).getBinary());
            }
            // 先頭にテンポ指定がない場合はデフォルト値を入れる
            tempoEvents.add(0, mm);
        }

        this.mpqStack = new int[tempoEvents.size()];
        this.cumulativeTicks = new long[tempoEvents.size()];
        this.cumulativeMicroseconds = new long[tempoEvents.size()];

        // 0 Tick 時点での値を先に入れる
        mpqStack[0] = tempoEvents.get(0).getMetaTempo();
        cumulativeTicks[0] = cumulativeMicroseconds[0] = 0L;

        int pos = 0;
        for (SMFMessage event : tempoEvents) {
            if (pos == 0) {
                pos ++;
                continue;
            }
            long tick = event.getTick();
            cumulativeTicks[pos] = tick;
            // deltaTick = 前回の Set Tempo からの時間 (Ticks)
            long deltaTick = tick - cumulativeTicks[pos - 1];
            mpqStack[pos] = event.getMetaTempo();
            // deltaMicroseconds = 前回の Set Tempo からの時間 (us)
            // <= MPQ = mpqStack[pos - 1] で deltaTick だけ経過している
            long deltaMicroseconds = TicksToMicroseconds(deltaTick, mpqStack[pos - 1], resolution);
            cumulativeMicroseconds[pos] = cumulativeMicroseconds[pos - 1] + deltaMicroseconds;

            ++pos;
        }
        
        setFileResolution(resolution);
    }// Constructor

    public long MicrosecondsToTicks(long us) {
        // 曲の開始から us[マイクロ秒] 経過した時点は、
        // 曲の開始から 何Ticks 経過した時点であるかを計算する
        int index = GetIndexFromMicroseconds(us);
        //System.out.println(" index " + index +" for " + us +"  US");

        // 現在の MPQ は mpq である
        int mpq = mpqStack[index];
        //System.out.println(" last mpq " + mpq + " resolution " + _fileResolution);

        // 直前のテンポ変更があったのは cumUs(マイクロ秒) 経過した時点であった
        long cumUs = cumulativeMicroseconds[index];
        // 直前のテンポ変更があったのは cumTicks(Ticks) 経過した時点であった
        long cumTicks = cumulativeTicks[index];

        // 直前のテンポ変更から deltaUs(マイクロ秒)が経過している
        long deltaUs = us - cumUs;
        // 直前のテンポ変更から deltaTicks(Ticks)が経過している
        long deltaTicks = MicrosecondsToTicks(deltaUs, mpq, _fileResolution);

        return cumTicks + deltaTicks;
    }
    
    public long TicksToMicroseconds(long tick){ 
        int index = GetIndexFromTick(tick);

        int mpq = mpqStack[index];

        long cumUs = cumulativeMicroseconds[index];
        long cumTicks = cumulativeTicks[index];

        long deltaTick = tick - cumTicks;
        long deltaUs = TicksToMicroseconds(deltaTick, mpq, getFileResolution());
        
        return cumUs + deltaUs;
    }

    private int GetIndexFromMicroseconds(long us) {
        // 指定された時間(マイクロ秒)時点におけるインデックスを二分探索で探す
        int lo = -1;
        int hi = cumulativeMicroseconds.length;
        while ((hi - lo) > 1) {
            int m = hi - (hi - lo) / 2;
            if (cumulativeMicroseconds[m] <= us) lo = m;
            else hi = m;
        }
        return lo;
    }

    private int GetIndexFromTick(long tick) {
        // 指定された時間(マイクロ秒)時点におけるインデックスを二分探索で探す
        int lo = -1;
        int hi = cumulativeTicks.length;
        while ((hi - lo) > 1) {
            int m = hi - (hi - lo) / 2;
            if (cumulativeTicks[m] <= tick) lo = m;
            else hi = m;
        }
        return lo;
    }

    private static long MicrosecondsToTicks(long us, long mpq, int resolution) {
        return us * resolution / mpq;
    }

    private static long TicksToMicroseconds(long tick, long mqp, int resolution) {
        // 時間(Tick)を時間(マイクロ秒)に変換する
        return tick * mqp / resolution;
    }

}// class TempoData