/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import java.io.File;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.ShortMessage;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MultiThreadQueue;
import jp.synthtarou.midimixer.libs.domino.DTextXML;
import jp.synthtarou.midimixer.libs.midi.programlist.database.PDFileManager;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.programlist.database.PDFile;
import jp.synthtarou.midimixer.mx40layer.MXChannelInfoReceiver;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIOut {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMIDIOut.class);

    public static final MXMIDIOut OUTPUT_NONE = new MXMIDIOut("none", null);

    private MidiDevice _device;
    private String _name;
    private boolean _isopen;
    int _assigned;
    private MXChannelInfoReceiver _program;
    private File _DXMLFile;

    Receiver _receiver;

    Thread _thread;
    MultiThreadQueue<MXMessage> _queue;
    protected MXMIDIOut(String name, MidiDevice device)  {
        this(name, device, null);
        _assigned = -1;
        _program = new MXChannelInfoReceiver();
    }
    
    public int assignedPort() {
        return _assigned;
    }
    
    public boolean isOpen() {
        return _isopen;
    }
    
    public boolean isJaveShynthesizer() {
        if (_device != null) {
            Class cls = _device.getClass();
            if (cls.getName().endsWith("SoftSynthesizer")) {
                return true;
            }
        }
        return false;
    }

    protected MXMIDIOut(String name, MidiDevice device, Receiver receiver) {
        this._isopen = false;
        this._device = device;
        this._receiver = receiver;
        this._name = name;
        this._name = name;
    }

    public String getName() {
        return _name;
    }

    public File getDXMLFile() {
        return _DXMLFile;
    }
    
    public void setDXMLFile(File file) {
        try {
            PDFileManager manager = PDFileManager.getManager();
            if (_DXMLFile != null) {
                manager.unregist(_DXMLFile.getName());
            }
            PDFile parser = DTextXML.fromFile(file);
/*
            System.out.println("*********** dump module of " + file);
            parser.getModule().dump();
            
            System.out.println("*********** dump drum of " + file);
            parser.getDrumSet().dump();

            System.out.println("*********** dump cc of " + file);
            parser.getCCList().dump();
*/
            _DXMLFile = file;
            manager.register(parser);
        }catch(Exception e) {
             e.printStackTrace();
        }
    }
    
    public int hashCode() {
        return _name.hashCode();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        MXMIDIOut out = (MXMIDIOut) o;
        if (out._name == this._name) {
            return true;
        }
        return false;
    }

    public String toString() {
        return _name;
    }

    public long getTimestamp() {
        try {
            return _device.getMicrosecondPosition();
        }catch(Exception e) {
            return -1;
        }
    }
   
    public void processMidiOut(MXMessage message) {
        final boolean _needDebugTrace = true;
        if (_needDebugTrace) {
            try {
                processMidiOutInternal(message);
            }catch(Throwable e) {
                e.printStackTrace();
            }
        }else {
            _queue.push(message);
        }
    }

    public void processMidiOutInternal(MXMessage message) {
        if (_receiver == null || _isopen == false || _assigned < 0) {
            new Throwable().printStackTrace();
            return;
        }
        
        if (message._extype == MXMessage.EXTYPE_SYSEX) {
            try {
                MidiMessage msg = MXMessageFactory.toJavaMessage(message);
                _receiver.send(msg, this.getTimestamp());
            }catch(Throwable e) {
                e.printStackTrace();
            }
        }
        else if (message._extype == MXMessage.EXTYPE_SYSEX_SPECIAL) {
            try {
                MidiMessage msg = MXMessageFactory.toJavaMessage(message);
                _receiver.send(msg, this.getTimestamp());
            }catch(Throwable e) {
                e.printStackTrace();
            }
        }
        else if (message._extype == MXMessage.EXTYPE_SHORTMESSAGE) {
            _program.processMXMessage(message);
            try {
                boolean isDataentry = false;
                if (message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) {
                    switch (message.getGate()) {
                        case MXMidi.DATA1_CC_DATAENTRY:
                        case MXMidi.DATA1_CC_DATAINC:
                        case MXMidi.DATA1_CC_DATADEC:
                            isDataentry = true;
                            break;
                    }
                }
                if (isDataentry) {
                    MidiMessage msg1 = MXMessageFactory.toJavaDataentry1MSB(message);
                    MidiMessage msg2 = MXMessageFactory.toJavaDataentry2LSB(message);
                    MidiMessage data1 = MXMessageFactory.toDataentryValueMSB(message);
                    MidiMessage data2 = MXMessageFactory.toDataentryValueLSB(message);
                    synchronized(MXMIDIOut.class) {
                        if (msg1 != null && msg2 != null) {
                            System.out.println("dataentry " + MXUtil.dumpHexFF(msg1.getMessage()) + "/" + MXUtil.dumpHexFF(msg2.getMessage()));
                            _receiver.send(msg1, getTimestamp());
                            _receiver.send(msg2, getTimestamp());
                        }
                        _receiver.send(data1, getTimestamp());
                        if (data2 != null) {
                            _receiver.send(data2, getTimestamp());
                        }
                    }
                }else if ((message.getCommand() == MXMidi.COMMAND_CONTROLCHANGE) && message.isValue14bit()) {
                    MidiMessage data1 = new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + message.getChannel(), message.getGate(), (message.getValue() >> 7) & 0x7f);
                    MidiMessage data2 = new ShortMessage(MXMidi.COMMAND_CONTROLCHANGE + message.getChannel(), message.getGate() + 0x20, (message.getValue()) & 0x7f);
                    synchronized(MXMIDIOut.class) {
                        _receiver.send(data1, getTimestamp());
                        _receiver.send(data2, getTimestamp());
                    }
                }else {
                    MidiMessage msg = MXMessageFactory.toJavaMessage(message);
                    if (msg != null) {
                        synchronized(MXMIDIOut.class) {
                            _receiver.send(msg, getTimestamp());
                        }
                    }
                }
            }catch(Throwable e) {
                _debug.printStackTrace(e);
            }
        }
    }

    public boolean open() {
        MXMIDIOutManager manager = MXMIDIOutManager.getManager();
        synchronized(manager) {
            manager.clearMIDIOutCache();
            if (_assigned < 0) {
                return false;
            }
            if (_isopen) {
                return true;
            }
            if (_receiver == null && _device != null) {
                _receiver = null;
                try {
                    if (_device.isOpen() == false) {
                        _device.open();
                    }
                    _receiver = _device.getReceiver();
                }catch(MidiUnavailableException e) {
                    _device.close();
                    return false;
                }
                _isopen = true;

                if (_thread == null) {
                    //まだ立ち上がってない場合限定で
                    _thread = new Thread() {
                        public void run() {
                            while(true) {
                                MXMessage message = _queue.pop();
                                if (message == null) {
                                    break;
                                }
                                try {
                                    MXMIDIOut.this.processMidiOutInternal(message);
                                }catch(Throwable e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    };
                    _queue= new MultiThreadQueue<MXMessage>();
                    _thread.setPriority(Thread.MAX_PRIORITY);
                    _thread.start();
                }
                return true;
            }
            return false;
        }
    }
    
    public void allNoteOff() {
        for (int ch = 0; ch < 16; ++ ch) {
            try {
                MXMessage msg = MXMessageFactory.fromShortMessage(_assigned, MXMidi.COMMAND_CONTROLCHANGE + ch, 123, 127);
                processMidiOutInternal(msg);
            }catch(Throwable e) {
                
            }
        }
    }
    
    public void close() {
        MXMIDIOutManager manager = MXMIDIOutManager.getManager();
        synchronized(manager) {
            if (_isopen) {
                allNoteOff();
                manager.onClose(this);
                _isopen = false;
                if(_name.equals("Gervill")) {
                    
                }else {
                    _device.close();
                }
                _receiver = null;
            }
        }
    }
}
