/*
 * Copyright (C) 2022 SynthTAROU
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import java.util.ArrayList;
import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugConsole;
import jp.synthtarou.midimixer.libs.MXUtil;
import jp.synthtarou.midimixer.libs.MultiThreadQueue;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXMidiPlayerDispatch;
import jp.synthtarou.midimixer.libs.midi.MXNoteOffWatcher;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXMIDIIn {
    private static final MXDebugConsole _debug = new MXDebugConsole(MXMIDIIn.class);

    public static final MXMidiPlayerDispatch INTERNAL_SEQUENCER = new MXMidiPlayerDispatch();
    
    private boolean _isopen;
    private String _name;
    private MidiDevice _device;
    private MXNoteOffWatcher _noteOff = new MXNoteOffWatcher();
    protected int _assigned;
    private boolean[] _isToMaster = new boolean[16];
    
    public boolean isOpen() {
        return _isopen;
    }
    
    static Thread _prioritySetFlag = null;

    public void pushInput(MXMessage message) {
        /*
        if (message.getCommand() == ShortMessage.TIMING_CLOCK) {
            return;
        }*/

        Thread t = Thread.currentThread();
        if (t != _prioritySetFlag) {
            _prioritySetFlag = t;
            try {
                t.setPriority(Thread.MAX_PRIORITY);
            }catch(Exception e) {
                e.printStackTrace();
            }
        }

        if (_assigned >= 0 && _isopen) {
            MXMain.getMain().startMainPath(this, message);
        }
    }
    
    public MXMIDIIn(String name, MidiDevice device) {
        _isopen = false;
        _name = name;
        _assigned = -1;
        _device = device;
    }
    
    public int assigned() {
        return _assigned;
    }
    
    public boolean isToMaster(int channel) {
        return _isToMaster[channel];
    }
    
    public void setToMaster(int channel, boolean toMaster) {
        _isToMaster[channel] = toMaster;
    }

    public String getMasterList() {
        StringBuffer str = new StringBuffer();
        for (int ch = 0; ch < 16; ++ ch) {
            if (isToMaster(ch)) {
                if (str.length() == 0) {
                    str.append(Integer.toString(ch));
                }else {
                    str.append(",");
                    str.append(Integer.toString(ch));
                }
            }
        }
        return str.toString();
    }
    
    public void setMasterList(String text) {
        if (text == null) {
            return;
        }
        ArrayList<String> list = new ArrayList();
        MXUtil.split(text, list, ',');
        _isToMaster = new boolean[16];
        for (String x : list) {
            int ch = MXUtil.parseTextForNumber(x);
            if (ch >= 0) {
                _isToMaster[ch] = true;
            }
        }
    }
    
    /**
     * 
     * @return 
     */
    public String getName() {
        return _name;
    }

    /**
     * 
     * @return 
     */
    public String toString() {
        return _name;
    }

    static MultiThreadQueue<MXMessage> _queue;
    static Thread _thread;
    static {
        _thread = new Thread() {
            public void run() {
                while(true) {
                    MXMessage message = _queue.pop();
                    if (message == null) {
                        break;
                    }
                    try {
                        MXMIDIIn input = MXMIDIInManager.getManager().findMIDIInput(message.getPort());
                        input.processMidiInInternal(message);
                    }catch(Throwable e) {
                        e.printStackTrace();
                    }
                }
            }
        };
        _queue= new MultiThreadQueue<MXMessage>();
        _thread.setPriority(Thread.MAX_PRIORITY);
        _thread.start();
    }

    public void processMidiIn(MXMessage message) {
        final boolean _needDebugTrace = false;
        if (_needDebugTrace) {
            processMidiInInternal(message);
        }else {
            _queue.push(message);
        }
    }

    public void processMidiInInternal(MXMessage message) {
        if (message.isMessageTypeChannel()) {
            final int channel = message.getChannel();
            if (message.getCommand() == MXMidi.COMMAND_NOTEOFF) {
                _noteOff.notifyNoteOffEvent(_assigned, channel,  message.getNoteNumberFromBytes(), "@7");
                return;
            }
            else if (message.getCommand() == MXMidi.COMMAND_NOTEON && message.getVelocityFromBytes() == 0) {
                _noteOff.notifyNoteOffEvent(_assigned, channel,  message.getNoteNumberFromBytes(), "@6");
                return;
            }
            else if (message.getCommand() == MXMidi.COMMAND_NOTEON) {
                int note = message.getNoteNumberFromBytes();
                MXMessage msg2 = MXMessageFactory.fromShortMessage(_assigned, MXMidi.COMMAND_NOTEOFF + channel, note, 0);
                _noteOff.addListener(message, msg2, new MXNoteOffWatcher.Handler() {
                    @Override
                    public void onNoteOffEvent(MXMessage target) {
                        pushInput(target);
                    }
                });
            }
        }
        try {
            pushInput(message);
        }catch(Exception e) {
            e.printStackTrace();
        }
    }

    public void close()  {
        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            allNoteOff();
            if (this instanceof MXMidiPlayerDispatch) {
                _isopen = false;
                return;
            }
            if (_isopen) {
                MXMIDIInManager.getManager().onClose(this);
                if (_device != null) {
                    _device.close();
                }
                _isopen = false;
            }
        }
    }

    public boolean open() {
        if (_assigned < 0) {
            return false;
        }
        MXMIDIInManager manager = MXMIDIInManager.getManager();
        synchronized(manager) {
            if (_isopen) {
                return true;
            }
            if (this instanceof MXMidiPlayerDispatch) {
                //SEQUENCER
                _isopen = true;
                return true;
            }
            try {
                _device.open();
                _isopen = true;
                _device.getTransmitter().setReceiver(new Receiver() {
                    @Override
                    public void send(MidiMessage midi, long timeStamp) {
                        try {
                            MXMessage message = MXMessageFactory.fromJavaMessage(_assigned, midi);
                            processMidiIn(message);
                        }catch(Exception e) {
                            e.printStackTrace();
                        }
                    }

                    @Override
                    public void close() {
                        MXMIDIIn.this.close();
                    }
                });
                return true;
             }catch(MidiUnavailableException e) {
                 //e.printStackTrace();
                 close();
                 return false;
             }
        }
    }

    public int hashCode() {
        return _name.hashCode();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        MXMIDIIn in = (MXMIDIIn) o;
        if (in._name == this._name) {
            return true;
        }
        return false;
    }
    
    public void allNoteOff() {
        _noteOff.allNoteOff();
    }

    public String textForMasterChannel() {
        StringBuffer masterMark = new StringBuffer();
        for (int ch = 0; ch < 16; ++ ch) {
            if (isToMaster(ch)) {
                if (masterMark.length() != 0) {
                    masterMark.append(", ");
                }
                masterMark.append(Integer.valueOf(ch + 1));
            }
        }
        return masterMark.toString();
    }
    
    public int parseMasteredText(String text){ 
        if (text == null) {
            return 0;
        }
        String[] array = text.split("[ ,]+");
        int hit = 0;
        
        for (int i = 0; i < array.length; ++ i) {
            String parts = array[i];
            if (parts.length() >= 1) {
                int ch1 = parts.charAt(0) - '0';
                if (ch1 < 0 || ch1 > 9) {
                    continue;
                }
                if (parts.length() >= 2) {
                    int ch2 = parts.charAt(2) - '0';
                    if (ch2 < 0 || ch2 > 9) {
                        continue;
                    }
                    ch1 = ch1 * 10 + ch2;
                }
                if (ch1 >= 1 && ch1 <= 16) {
                    setToMaster(ch1 - 1, true);
                }
                hit ++;
            }
        }
        return hit;
    }
}
